import { useState } from "react";
import axios from "axios";
import { Card, CardHeader, CardTitle, CardDescription, CardContent } from "./ui/card";
import { Button } from "./ui/button";
import { Textarea } from "./ui/textarea";
import { toast } from "sonner";
import { Send, MessageCircle } from "lucide-react";

const API = process.env.REACT_APP_BACKEND_URL || 'http://localhost:8001';

export default function AdminTelegramBroadcast() {
  const [message, setMessage] = useState('');
  const [sending, setSending] = useState(false);

  const sendBroadcast = async () => {
    if (!message.trim()) {
      toast.error('❌ پیام نمی\u200cتواند خالی باشد');
      return;
    }

    if (!window.confirm('آیا مطمئن هستید که می\u200cخواهید این پیام را به تمام کاربران تلگرام ارسال کنید؟')) {
      return;
    }

    setSending(true);
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post(
        `${API}/api/admin/telegram/broadcast`,
        { message },
        { headers: { Authorization: `Bearer ${token}` } }
      );
      
      if (response.data.success) {
        toast.success(`✅ ${response.data.message}`, { duration: 5000 });
        setMessage('');
      } else {
        toast.error(`❌ ${response.data.message}`);
      }
    } catch (error) {
      console.error('❌ خطا:', error);
      toast.error('خطا در ارسال پیام عمومی');
    } finally {
      setSending(false);
    }
  };

  return (
    <Card className="shadow-lg border-0">
      <CardHeader className="bg-gradient-to-r from-blue-500 to-purple-600 text-white rounded-t-lg">
        <CardTitle className="flex items-center gap-2">
          <MessageCircle className="w-5 h-5" />
          📢 ارسال پیام عمومی به کاربران تلگرام
        </CardTitle>
        <CardDescription className="text-blue-50">
          این پیام به تمام کاربرانی که از طریق تلگرام ثبت\u200cنام کرده\u200cاند ارسال می\u200cشود
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4 p-6">
        <div className="space-y-2">
          <label className="text-sm font-semibold text-gray-700">
            متن پیام:
          </label>
          <Textarea
            value={message}
            onChange={(e) => setMessage(e.target.value)}
            placeholder="پیام خود را اینجا بنویسید...&#10;&#10;💡 می\u200cتوانید از Markdown استفاده کنید:&#10;*متن بولد*&#10;_متن ایتالیک_&#10;[لینک](https://example.com)"
            rows={8}
            className="w-full resize-none"
          />
          <p className="text-xs text-gray-500">
            تعداد کاراکتر: {message.length}
          </p>
        </div>

        <div className="bg-yellow-50 border border-yellow-300 rounded-lg p-4">
          <p className="text-sm text-yellow-800">
            ⚠️ <strong>هشدار:</strong> این پیام به تمام کاربران تلگرام ارسال خواهد شد. لطفاً قبل از ارسال، محتوای پیام را بررسی کنید.
          </p>
        </div>

        <div className="flex gap-3">
          <Button
            onClick={sendBroadcast}
            disabled={sending || !message.trim()}
            className="flex-1 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white"
          >
            {sending ? (
              <>
                <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin ml-2"></div>
                در حال ارسال...
              </>
            ) : (
              <>
                <Send className="w-4 h-4 ml-2" />
                ارسال پیام عمومی
              </>
            )}
          </Button>
          
          <Button
            onClick={() => setMessage('')}
            variant="outline"
            disabled={sending}
            className="px-6"
          >
            پاک کردن
          </Button>
        </div>

        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <p className="text-sm text-blue-800">
            💡 <strong>نکته:</strong> ارسال پیام به تعداد زیاد کاربر ممکن است چند دقیقه طول بکشد. نتیجه ارسال در انتها به شما نمایش داده می\u200cشود.
          </p>
        </div>
      </CardContent>
    </Card>
  );
}
