"""Stats Recalculation Service - بازسازی آمار برای رفع inconsistency"""

from typing import Dict, List
import logging

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class StatsRecalculationService:
    """سرویس بازسازی آمار برای رفع نا‌سازگاری‌ها"""
    
    @staticmethod
    async def recalculate_user_stats(user_id: str) -> Dict:
        """
        بازسازی کامل آمار یک کاربر
        
        محاسبه دوباره:
        - total_descendants
        - max_descendant_depth
        - direct_children_count
        - is_tree_complete
        
        Args:
            user_id: شناسه کاربر
            
        Returns:
            آمار جدید
        """
        
        # دریافت کاربر
        user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "status": 1}
        )
        
        if not user:
            raise ValueError(f"کاربر {user_id} یافت نشد")
        
        placement_path = user.get("placement_path", "")
        user_depth = user.get("depth", 0)
        
        # محاسبه total_descendants (تعداد کل زیرمجموعه‌ها)
        if placement_path:
            total_descendants = await _db.users.count_documents({
                "placement_path": {"$regex": f"^{placement_path}/"},
                "status": "active"
            })
        else:
            # seed - همه کاربران active
            total_descendants = await _db.users.count_documents({
                "status": "active",
                "user_id": {"$ne": "seed"}
            })
        
        # محاسبه max_descendant_depth (عمیق‌ترین descendant)
        if placement_path:
            deepest = await _db.users.find_one(
                {
                    "placement_path": {"$regex": f"^{placement_path}/"},
                    "status": "active"
                },
                {"_id": 0, "depth": 1},
                sort=[("depth", -1)]
            )
        else:
            # seed
            deepest = await _db.users.find_one(
                {
                    "status": "active",
                    "user_id": {"$ne": "seed"}
                },
                {"_id": 0, "depth": 1},
                sort=[("depth", -1)]
            )
        
        if deepest:
            max_descendant_depth = deepest["depth"] - user_depth
        else:
            max_descendant_depth = 0
        
        # محاسبه direct_children_count
        direct_children = await _db.users.find(
            {"placement_parent_id": user_id, "status": "active"},
            {"_id": 0, "user_id": 1}
        ).to_list(10)  # حداکثر 3 فرزند، ولی safe
        
        direct_children_ids = [c["user_id"] for c in direct_children]
        direct_children_count = len(direct_children_ids)
        
        # تعیین is_tree_complete
        is_tree_complete = max_descendant_depth >= 6
        
        # به‌روزرسانی
        update_result = await _db.users.update_one(
            {"user_id": user_id},
            {
                "$set": {
                    "total_descendants": total_descendants,
                    "max_descendant_depth": max_descendant_depth,
                    "direct_children": direct_children_ids,
                    "direct_children_count": direct_children_count,
                    "is_tree_complete": is_tree_complete
                }
            }
        )
        
        logger.info(f"✅ Recalculated stats for {user_id}: descendants={total_descendants}, max_depth={max_descendant_depth}, is_complete={is_tree_complete}")
        
        return {
            "user_id": user_id,
            "total_descendants": total_descendants,
            "max_descendant_depth": max_descendant_depth,
            "direct_children_count": direct_children_count,
            "is_tree_complete": is_tree_complete,
            "updated": update_result.modified_count > 0
        }
    
    @staticmethod
    async def recalculate_all_stats(batch_size: int = 100) -> Dict:
        """
        بازسازی آمار همه کاربران فعال
        
        Args:
            batch_size: تعداد کاربران در هر batch
            
        Returns:
            خلاصه عملیات
        """
        
        logger.info("🔄 Starting full stats recalculation...")
        
        # دریافت همه کاربران فعال
        total_users = await _db.users.count_documents({"status": "active"})
        
        processed = 0
        updated = 0
        errors = 0
        
        # پردازش batch به batch
        skip = 0
        
        while skip < total_users:
            users = await _db.users.find(
                {"status": "active"},
                {"_id": 0, "user_id": 1}
            ).skip(skip).limit(batch_size).to_list(batch_size)
            
            for user in users:
                try:
                    result = await StatsRecalculationService.recalculate_user_stats(user["user_id"])
                    processed += 1
                    if result["updated"]:
                        updated += 1
                except Exception as e:
                    logger.error(f"❌ Error recalculating {user['user_id']}: {e}")
                    errors += 1
            
            skip += batch_size
            logger.info(f"📊 Progress: {processed}/{total_users} users processed")
        
        logger.info(f"✅ Recalculation complete: {processed} processed, {updated} updated, {errors} errors")
        
        return {
            "total_users": total_users,
            "processed": processed,
            "updated": updated,
            "errors": errors
        }
    
    @staticmethod
    async def fix_inconsistencies() -> Dict:
        """
        شناسایی و رفع نا‌سازگاری‌ها
        
        بررسی:
        - کاربرانی که direct_children_count != len(direct_children)
        - کاربرانی که is_tree_complete اشتباه است
        - کاربرانی که max_descendant_depth نادرست است
        
        Returns:
            گزارش مشکلات و تعمیرات
        """
        
        logger.info("🔍 Checking for inconsistencies...")
        
        issues = {
            "children_count_mismatch": [],
            "tree_complete_incorrect": [],
            "max_depth_incorrect": []
        }
        
        # بررسی همه کاربران فعال
        users = await _db.users.find(
            {"status": "active"},
            {"_id": 0, "user_id": 1, "direct_children": 1, "direct_children_count": 1, 
             "max_descendant_depth": 1, "is_tree_complete": 1}
        ).to_list(None)
        
        for user in users:
            user_id = user["user_id"]
            
            # بررسی direct_children_count
            actual_count = len(user.get("direct_children", []))
            stored_count = user.get("direct_children_count", 0)
            
            if actual_count != stored_count:
                issues["children_count_mismatch"].append({
                    "user_id": user_id,
                    "stored": stored_count,
                    "actual": actual_count
                })
            
            # بررسی is_tree_complete
            max_depth = user.get("max_descendant_depth", 0)
            is_complete = user.get("is_tree_complete", False)
            should_be_complete = max_depth >= 6
            
            if is_complete != should_be_complete:
                issues["tree_complete_incorrect"].append({
                    "user_id": user_id,
                    "is_complete": is_complete,
                    "should_be": should_be_complete,
                    "max_depth": max_depth
                })
        
        # تعمیر خودکار
        fixed = 0
        
        for issue in issues["children_count_mismatch"]:
            await _db.users.update_one(
                {"user_id": issue["user_id"]},
                {"$set": {"direct_children_count": issue["actual"]}}
            )
            fixed += 1
        
        for issue in issues["tree_complete_incorrect"]:
            await _db.users.update_one(
                {"user_id": issue["user_id"]},
                {"$set": {"is_tree_complete": issue["should_be"]}}
            )
            fixed += 1
        
        logger.info(f"✅ Fixed {fixed} inconsistencies")
        
        return {
            "total_issues": sum(len(v) for v in issues.values()),
            "fixed": fixed,
            "details": issues
        }
