#!/usr/bin/env python3
"""
Migration: اصلاح is_tree_complete با منطق صحیح
هرم کامل = max_descendant_depth >= 6 AND total_descendants >= 1092
"""

import subprocess
import sys

def run_command(container, db_name, command):
    """اجرای دستور MongoDB"""
    cmd = f"docker exec {container} mongosh {db_name} --eval '{command}'"
    result = subprocess.run(cmd, shell=True, capture_output=True, text=True)
    return result.returncode == 0, result.stdout

def main():
    if len(sys.argv) < 2:
        print("استفاده: python3 fix_is_tree_complete.py <CONTAINER>")
        print("مثال: python3 fix_is_tree_complete.py mlm-mongodb")
        sys.exit(1)
    
    container = sys.argv[1]
    db_name = "mlm_production"
    
    print("=" * 80)
    print("Migration: اصلاح is_tree_complete")
    print("=" * 80)
    
    # مرحله 1: Reset همه به false
    print("\n[1/3] Reset کردن همه is_tree_complete به false...")
    
    cmd = '''
    db.users.updateMany(
        {},
        { $set: { is_tree_complete: false } }
    )
    '''
    
    success, output = run_command(container, db_name, cmd)
    if success:
        print("✅ همه رکوردها reset شد")
    else:
        print("❌ خطا در reset")
        return
    
    # مرحله 2: Set کردن صحیح
    print("\n[2/3] تنظیم is_tree_complete برای کاربران واجد شرایط...")
    print("    شرط: max_descendant_depth >= 6 AND total_descendants >= 1092")
    
    cmd = '''
    db.users.updateMany(
        {
            max_descendant_depth: { $gte: 6 },
            total_descendants: { $gte: 1092 },
            status: "active"
        },
        { $set: { is_tree_complete: true } }
    )
    '''
    
    success, output = run_command(container, db_name, cmd)
    if success:
        print("✅ is_tree_complete تنظیم شد")
        # استخراج تعداد
        if "modifiedCount" in output:
            import re
            match = re.search(r'modifiedCount:\s*(\d+)', output)
            if match:
                count = match.group(1)
                print(f"   تعداد: {count} کاربر")
    else:
        print("❌ خطا")
        return
    
    # مرحله 3: نمایش آمار
    print("\n[3/3] آمار نهایی...")
    
    cmd = '''
    printjson({
        total: db.users.countDocuments({}),
        active: db.users.countDocuments({status: "active"}),
        tree_complete: db.users.countDocuments({is_tree_complete: true}),
        almost_complete: db.users.countDocuments({
            max_descendant_depth: {$gte: 6},
            total_descendants: {$lt: 1092},
            status: "active"
        })
    })
    '''
    
    success, output = run_command(container, db_name, cmd)
    if success:
        print("\n📊 آمار:")
        print(output)
    
    print("\n" + "=" * 80)
    print("✅ Migration کامل شد!")
    print("=" * 80)
    print("\nتوضیحات:")
    print("- tree_complete: کاربرانی که هرمشان کاملاً پر شده (1092+ descendants)")
    print("- almost_complete: کاربرانی که لول 7 دارن ولی هنوز کامل نشده")
    print("\nحالا کد backend را هم باید آپدیت کنی:")
    print("  cp tree_stats_FIXED.py /path/to/backend/services/tree/tree_stats.py")
    print("  docker-compose restart backend")

if __name__ == "__main__":
    main()
