"""Setup Routes - راه‌اندازی اولیه"""

import logging
from datetime import datetime, timezone

import bcrypt
from fastapi import APIRouter

from database import get_db

router = APIRouter()


@router.get("/health")
async def health_check():
    """بررسی سلامت سیستم"""
    db = await get_db()
    
    result = {
        "status": "ok",
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "mongodb": False,
        "seed_user": False,
        "error": None
    }
    
    try:
        await db.command('ping')
        result["mongodb"] = True
        
        seed = await db.users.find_one({"user_id": "seed"}, {"_id": 0, "email": 1})
        if seed:
            result["seed_user"] = True
            result["seed_email"] = seed.get("email")
            
    except Exception as e:
        result["status"] = "error"
        result["error"] = str(e)
    
    return result


@router.get("/debug/test-login/{email}")
async def debug_test_login(email: str):
    """تست دستی برای بررسی کاربر"""
    db = await get_db()
    
    try:
        email = email.strip().lower()
        user = await db.users.find_one(
            {"email": email}, 
            {"_id": 0, "user_id": 1, "email": 1, "role": 1, "status": 1, "password": 1}
        )
        
        if not user:
            return {"found": False, "message": f"کاربر با ایمیل {email} یافت نشد"}
        
        has_password = bool(user.get("password"))
        password_hash_prefix = user.get("password", "")[:20] + "..." if has_password else "EMPTY"
        
        return {
            "found": True,
            "user_id": user.get("user_id"),
            "email": user.get("email"),
            "role": user.get("role"),
            "status": user.get("status"),
            "has_password": has_password,
            "password_hash_preview": password_hash_prefix
        }
        
    except Exception as e:
        return {"error": str(e)}


@router.post("/debug/reset-admin-password")
async def debug_reset_admin_password():
    """ریست رمز ادمین به admin123"""
    db = await get_db()
    
    try:
        new_hash = bcrypt.hashpw("admin123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        result = await db.users.update_one(
            {"user_id": "seed"},
            {"$set": {"password": new_hash}}
        )
        
        if result.modified_count > 0:
            return {"success": True, "message": "رمز ادمین به admin123 تغییر یافت"}
        else:
            return {"success": False, "message": "کاربر seed یافت نشد یا تغییری نکرد"}
            
    except Exception as e:
        return {"error": str(e)}


@router.get("/setup/test")
async def setup_test():
    """تست راه‌اندازی"""
    db = await get_db()
    
    result = {
        "database": False,
        "seed_user": False,
        "settings": False,
        "ready": False
    }
    
    try:
        await db.command('ping')
        result["database"] = True
        
        seed = await db.users.find_one({"user_id": "seed"})
        result["seed_user"] = bool(seed)
        
        settings = await db.settings.find_one({"setting_id": "rewards"})
        result["settings"] = bool(settings)
        
        result["ready"] = all([result["database"], result["seed_user"], result["settings"]])
        
    except Exception as e:
        result["error"] = str(e)
    
    return result


@router.post("/setup/init")
async def setup_init():
    """راه‌اندازی اولیه سیستم"""
    db = await get_db()
    
    # ایجاد تنظیمات پیش‌فرض
    settings = await db.settings.find_one({"setting_id": "rewards"})
    if not settings:
        default_settings = {
            "setting_id": "rewards",
            "activation_fee": 100.0,
            "seed_initial_payment": 100.0,
            "direct_first": 100.0,
            "direct_second": 75.0,
            "direct_third": 50.0,
            "direct_extra": 25.0,
            "level_3": 10.0,
            "level_5": 5.0,
            "level_7": 2.0
        }
        await db.settings.insert_one(default_settings)
    
    # ایجاد seed user
    seed = await db.users.find_one({"user_id": "seed"})
    if not seed:
        password_hash = bcrypt.hashpw("admin123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        seed_user = {
            "user_id": "seed",
            "email": "admin@mlm.com",
            "phone_number": None,
            "password": password_hash,
            "role": "admin",
            "status": "active",
            "referrer_id": None,
            "placement_parent_id": None,
            "direct_children": [],
            "depth": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": datetime.now(timezone.utc).isoformat(),
            "referral_code": "SEED"
        }
        await db.users.insert_one(seed_user)
        logging.info("✅ Seed user created: admin@mlm.com / admin123")
    
    return {
        "success": True,
        "message": "راه‌اندازی اولیه انجام شد",
        "admin_email": "admin@mlm.com",
        "admin_password": "admin123"
    }


@router.post("/setup/reset-admin")
async def setup_reset_admin():
    """ریست کاربر ادمین"""
    db = await get_db()
    
    password_hash = bcrypt.hashpw("admin123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
    
    result = await db.users.update_one(
        {"user_id": "seed"},
        {"$set": {
            "email": "admin@mlm.com",
            "password": password_hash,
            "role": "admin",
            "status": "active"
        }}
    )
    
    if result.modified_count > 0:
        return {
            "success": True,
            "message": "کاربر ادمین ریست شد",
            "email": "admin@mlm.com",
            "password": "admin123"
        }
    else:
        return {"success": False, "message": "کاربر seed یافت نشد"}
