import { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Save, BookOpen, GraduationCap, MessageCircle } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AdminConfig() {
  const [faqUrl, setFaqUrl] = useState('');
  const [tutorialUrl, setTutorialUrl] = useState('');
  const [telegramSupport, setTelegramSupport] = useState('');
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    fetchConfigs();
  }, []);

  const fetchConfigs = async () => {
    try {
      const token = localStorage.getItem('token');
      
      // Fetch each config individually with error handling
      const fetchConfig = async (key) => {
        try {
          const response = await axios.get(`${API}/admin/config/${key}`, { 
            headers: { Authorization: `Bearer ${token}` } 
          });
          return response.data?.value || '';
        } catch (err) {
          // Return empty string if config doesn't exist yet (404)
          return '';
        }
      };

      const [faq, tutorial, telegram] = await Promise.all([
        fetchConfig('faq_url'),
        fetchConfig('tutorial_url'),
        fetchConfig('telegram_support')
      ]);

      setFaqUrl(faq);
      setTutorialUrl(tutorial);
      setTelegramSupport(telegram);
    } catch (error) {
      console.error('Error fetching configs:', error);
    }
  };

  const handleSave = async (key, value) => {
    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      await axios.put(
        `${API}/admin/config`,
        { key, value },
        { headers: { Authorization: `Bearer ${token}` } }
      );
      toast.success('✅ تنظیمات ذخیره شد');
    } catch (error) {
      toast.error('خطا در ذخیره تنظیمات');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="max-w-4xl mx-auto space-y-6">
      <Card>
        <CardHeader className="bg-gradient-to-r from-indigo-500 to-purple-600 text-white">
          <CardTitle className="text-2xl">⚙️ تنظیمات سیستم</CardTitle>
          <CardDescription className="text-indigo-50">
            مدیریت لینک‌ها و اطلاعات عمومی
          </CardDescription>
        </CardHeader>
        <CardContent className="p-6 space-y-6">
          <div className="space-y-2">
            <Label htmlFor="faq-url" className="text-lg font-semibold flex items-center gap-2">
              <BookOpen className="w-5 h-5 text-blue-600" />
              لینک صفحه توضیحات
            </Label>
            <p className="text-sm text-gray-600 mb-2">
              لینک صفحه‌ای که توضیحات و سوالات متداول در آن قرار دارد
            </p>
            <div className="flex gap-2">
              <Input
                id="faq-url"
                value={faqUrl}
                onChange={(e) => setFaqUrl(e.target.value)}
                placeholder="https://example.com/faq"
                className="flex-1"
              />
              <Button
                onClick={() => handleSave('faq_url', faqUrl)}
                disabled={loading}
              >
                <Save className="w-4 h-4 ml-2" />
                ذخیره
              </Button>
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="tutorial-url" className="text-lg font-semibold flex items-center gap-2">
              <GraduationCap className="w-5 h-5 text-green-600" />
              لینک صفحه آموزش
            </Label>
            <p className="text-sm text-gray-600 mb-2">
              لینک صفحه‌ای که آموزش‌های سیستم در آن قرار دارد
            </p>
            <div className="flex gap-2">
              <Input
                id="tutorial-url"
                value={tutorialUrl}
                onChange={(e) => setTutorialUrl(e.target.value)}
                placeholder="https://example.com/tutorial"
                className="flex-1"
              />
              <Button
                onClick={() => handleSave('tutorial_url', tutorialUrl)}
                disabled={loading}
              >
                <Save className="w-4 h-4 ml-2" />
                ذخیره
              </Button>
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="telegram-support" className="text-lg font-semibold flex items-center gap-2">
              <MessageCircle className="w-5 h-5 text-cyan-600" />
              آیدی تلگرام پشتیبانی
            </Label>
            <p className="text-sm text-gray-600 mb-2">
              آیدی تلگرام برای پشتیبانی کاربران (مثلاً: @support)
            </p>
            <div className="flex gap-2">
              <Input
                id="telegram-support"
                value={telegramSupport}
                onChange={(e) => setTelegramSupport(e.target.value)}
                placeholder="@support_username"
                className="flex-1"
              />
              <Button
                onClick={() => handleSave('telegram_support', telegramSupport)}
                disabled={loading}
              >
                <Save className="w-4 h-4 ml-2" />
                ذخیره
              </Button>
            </div>
          </div>

          <div className="bg-blue-50 border-2 border-blue-200 p-4 rounded-lg mt-6">
            <h4 className="font-semibold text-blue-900 mb-2">💡 نکات مهم:</h4>
            <ul className="text-sm text-blue-800 space-y-1">
              <li>• برای صفحات توضیحات و آموزش می‌توانید از Google Docs استفاده کنید</li>
              <li>• لینک‌ها باید با http:// یا https:// شروع شوند</li>
              <li>• آیدی تلگرام باید با @ شروع شود</li>
              <li>• تغییرات بلافاصله برای کاربران اعمال می‌شود</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
