"""Path Manager - مدیریت Materialized Path"""

from typing import List, Dict, Optional
import logging

logger = logging.getLogger(__name__)


class PathManager:
    """مدیریت عملیات Materialized Path"""
    
    @staticmethod
    def build_child_path(parent_path: Optional[str], child_id: str) -> str:
        """
        ساخت path برای کاربر جدید (child)
        
        Args:
            parent_path: path والد (می‌تواند "" برای root/seed باشد)
            child_id: شناسه کاربر جدید (فرزند)
            
        Returns:
            path جدید
            
        Examples:
            >>> PathManager.build_child_path("", "user1")
            "user1"
            >>> PathManager.build_child_path("user1", "user4")
            "user1/user4"
        """
        # Special case: parent is root (empty path) - like seed
        if parent_path is None or parent_path == "":
            return child_id
        
        # Normal case: append to parent path
        return f"{parent_path}/{child_id}"
    
    @staticmethod
    def extract_ancestors(path: str) -> List[str]:
        """
        استخراج لیست اجداد از path
        
        Args:
            path: Materialized path
            
        Returns:
            لیست user_id های اجداد (از root تا parent)
            
        Examples:
            >>> PathManager.extract_ancestors("")
            []
            >>> PathManager.extract_ancestors("user1")
            []
            >>> PathManager.extract_ancestors("user1/user4")
            ["user1"]
            >>> PathManager.extract_ancestors("user1/user4/user7")
            ["user1", "user4"]
        """
        if not path:
            return []
        
        parts = path.split('/')
        # برگرداندن همه به جز آخری (خود کاربر)
        return parts[:-1] if len(parts) > 1 else []
    
    @staticmethod
    def calculate_depth(path: str) -> int:
        """
        محاسبه depth از path
        
        Args:
            path: Materialized path
            
        Returns:
            عمق در درخت (0 برای root)
            
        Examples:
            >>> PathManager.calculate_depth("")
            0
            >>> PathManager.calculate_depth("user1")
            1
            >>> PathManager.calculate_depth("user1/user4")
            2
            >>> PathManager.calculate_depth("user1/user4/user10")
            3
        """
        if not path or path == "":
            return 0
        
        parts = path.split('/')
        return len(parts)
    
    @staticmethod
    def is_descendant_of(child_path: str, parent_path: str) -> bool:
        """
        بررسی آیا child زیرمجموعه parent است
        
        Args:
            child_path: path فرزند
            parent_path: path والد احتمالی
            
        Returns:
            True اگر child زیر parent باشد
            
        Examples:
            >>> PathManager.is_descendant_of("user1/user4", "user1")
            True
            >>> PathManager.is_descendant_of("user1/user4", "user2")
            False
        """
        if not child_path:
            return False
        
        # parent_path خالی = root
        if not parent_path or parent_path == "":
            return True  # همه زیر root هستند
        
        # child باید با parent/ شروع شود
        return child_path.startswith(f"{parent_path}/")
    
    @staticmethod
    def build_ancestor_depths(path: str, ancestors_data: List[Dict]) -> List[Dict]:
        """
        ساخت آرایه ancestor_depths برای محاسبه سریع پاداش
        
        Args:
            path: path کاربر
            ancestors_data: لیست اطلاعات اجداد [{user_id, depth}, ...]
            
        Returns:
            آرایه مرتب شده از root تا parent: [{user_id, depth}, ...]
            
        Example:
            >>> ancestors = [
            ...     {"user_id": "seed", "depth": 0},
            ...     {"user_id": "user1", "depth": 1}
            ... ]
            >>> PathManager.build_ancestor_depths("user1/user4", ancestors)
            [{"user_id": "seed", "depth": 0}, {"user_id": "user1", "depth": 1}]
        """
        # استخراج ترتیب اجداد از path
        ancestor_ids = PathManager.extract_ancestors(path)
        
        # ساخت dict برای دسترسی سریع
        ancestors_dict = {a["user_id"]: a for a in ancestors_data}
        
        # ساخت آرایه مرتب
        ancestor_depths = []
        for ancestor_id in ancestor_ids:
            if ancestor_id in ancestors_dict:
                ancestor_depths.append({
                    "user_id": ancestor_id,
                    "depth": ancestors_dict[ancestor_id]["depth"]
                })
        
        return ancestor_depths
    
    @staticmethod
    def get_level_from_path(path: str, level: int) -> Optional[str]:
        """
        دریافت user_id در یک level خاص
        
        Args:
            path: path کاربر
            level: سطح مورد نظر (0 = root یعنی اولین user)
            
        Returns:
            user_id در آن level یا None
            
        Examples:
            >>> PathManager.get_level_from_path("user1/user4", 0)
            "user1"
            >>> PathManager.get_level_from_path("user1/user4/user7", 1)
            "user4"
            >>> PathManager.get_level_from_path("user1/user4", 3)
            None
        """
        if not path:
            return None
        
        parts = path.split('/')
        if level >= len(parts):
            return None
        
        return parts[level]
    
    @staticmethod
    def validate_path(path: str) -> bool:
        """
        اعتبارسنجی path
        
        Args:
            path: path برای بررسی
            
        Returns:
            True اگر path معتبر باشد
        """
        # Empty path is valid (root/seed)
        if path == "":
            return True
            
        if not path:
            return False
        
        # نباید با / شروع یا تمام شود
        if path.startswith('/') or path.endswith('/'):
            return False
        
        # نباید // داشته باشد
        if '//' in path:
            return False
        
        # همه parts باید غیرخالی باشند
        parts = path.split('/')
        return all(part for part in parts)
