"""User Routes - اندپوینت‌های کاربر"""

import logging
from fastapi import APIRouter, HTTPException, Depends

from models.user import ChangeOwnPasswordData
from services.auth_service import get_current_user, hash_password, verify_password
from cache_manager import cache
from database import get_db

router = APIRouter()


@router.get("/user/profile")
async def get_profile(current_user: dict = Depends(get_current_user)):
    """
    دریافت پروفایل کاربر
    
    - اطلاعات اصلی کاربر
    - تعداد زیرمجموعه‌ها
    - تعداد کاربران در هر سطح
    """
    db = await get_db()
    user_id = current_user["user_id"]
    
    # بررسی cache
    cache_key = f"profile:{user_id}"
    cached_profile = await cache.get(cache_key)
    if cached_profile:
        return cached_profile
    
    # اطلاعات معرف
    referrer = None
    if current_user.get("referrer_id"):
        ref = await db.users.find_one({"user_id": current_user["referrer_id"]}, {"_id": 0, "email": 1})
        if ref:
            referrer = ref["email"]
    
    # شمارش زیرمجموعه‌ها
    direct_referrals = await db.users.count_documents({"referrer_id": user_id})
    pending_referrals = await db.users.count_documents({"referrer_id": user_id, "status": "pending"})
    
    # شمارش سطوح
    level_counts = await _get_level_counts(db, user_id, current_user.get("depth", 0), current_user.get("direct_children", []))
    
    profile = {
        "user_id": user_id,
        "email": current_user["email"],
        "status": current_user["status"],
        "referral_code": current_user["referral_code"],
        "referrer": referrer,
        "total_points": current_user.get("total_points", 0.0),
        "direct_referrals": direct_referrals,
        "pending_referrals": pending_referrals,
        "level_counts": level_counts,
        "depth": current_user["depth"]
    }
    
    # ذخیره در cache (60 ثانیه)
    await cache.set(cache_key, profile, 60)
    
    return profile


async def _get_level_counts(db, user_id: str, current_depth: int, direct_children: list) -> dict:
    """محاسبه تعداد کاربران در هر سطح"""
    cache_key = f"level_counts:{user_id}"
    
    # بررسی cache
    cached = await cache.get(cache_key)
    if cached:
        return cached
    
    level_counts = {"level_1": 1}  # خود کاربر
    
    for level in range(2, 8):
        target_depth = current_depth + (level - 1)
        
        if level == 2:
            count = len(direct_children)
        else:
            if current_depth == 0:
                count = await db.users.count_documents({"depth": target_depth, "status": "active"})
            else:
                count = await _count_descendants_at_level(db, user_id, level - 1)
        
        level_counts[f"level_{level}"] = count
    
    # ذخیره در cache (5 دقیقه)
    await cache.set(cache_key, level_counts, 300)
    
    return level_counts


async def _count_descendants_at_level(db, user_id: str, target_level: int) -> int:
    """شمارش descendants در یک سطح خاص"""
    if target_level <= 0:
        return 1
    
    current_level = [user_id]
    
    for _ in range(target_level):
        if not current_level:
            return 0
        
        users = await db.users.find(
            {"user_id": {"$in": current_level}},
            {"_id": 0, "direct_children": 1}
        ).to_list(len(current_level))
        
        next_level = []
        for u in users:
            next_level.extend(u.get("direct_children", []))
        
        current_level = next_level
    
    return len(current_level)


@router.get("/user/tree")
async def get_user_tree(current_user: dict = Depends(get_current_user)):
    """
    دریافت درخت کاربر
    
    - تا 7 سطح زیرمجموعه
    """
    db = await get_db()
    
    async def build_tree(uid: str, max_depth: int = 3) -> dict:
        if max_depth <= 0:
            return None
        
        user = await db.users.find_one({"user_id": uid}, {"_id": 0})
        if not user:
            return None
        
        children = []
        for child_id in user.get("direct_children", []):
            child_tree = await build_tree(child_id, max_depth - 1)
            if child_tree:
                children.append(child_tree)
        
        return {
            "user_id": user["user_id"],
            "email": user["email"],
            "status": user["status"],
            "depth": user["depth"],
            "total_points": user["total_points"],
            "direct_children_count": len(user.get("direct_children", [])),
            "children": children
        }
    
    tree = await build_tree(current_user["user_id"], max_depth=7)
    return tree


@router.get("/user/transactions")
async def get_transactions(current_user: dict = Depends(get_current_user)):
    """دریافت تراکنش‌های کاربر"""
    db = await get_db()
    
    transactions = await db.transactions.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("timestamp", -1).limit(100).to_list(100)
    
    return transactions


@router.put("/user/change-password")
async def change_own_password(data: ChangeOwnPasswordData, current_user: dict = Depends(get_current_user)):
    """تغییر رمز عبور توسط خود کاربر"""
    db = await get_db()
    
    # بررسی رمز فعلی
    if not verify_password(data.current_password, current_user["password"]):
        raise HTTPException(status_code=400, detail="رمز عبور فعلی اشتباه است")
    
    # تغییر رمز
    new_hash = hash_password(data.new_password)
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$set": {"password": new_hash}}
    )
    
    return {"message": "رمز عبور با موفقیت تغییر یافت"}


@router.put("/user/wallet")
async def update_wallet(wallet_address: str, wallet_network: str, current_user: dict = Depends(get_current_user)):
    """بروزرسانی آدرس کیف پول"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$set": {
            "wallet_address": wallet_address,
            "wallet_network": wallet_network
        }}
    )
    
    return {"message": "آدرس کیف پول بروزرسانی شد"}
