#!/usr/bin/env python3
"""
Migration Script: اضافه کردن max_descendant_depth به کاربران موجود
این اسکریپت فیلد max_descendant_depth را به همه کاربران اضافه و محاسبه می‌کند
"""

import asyncio
import sys
from motor.motor_asyncio import AsyncIOMotorClient
from typing import List, Dict

# تنظیمات MongoDB
MONGO_URI = "mongodb://localhost:27017"
DB_NAME = "mlm_db"


async def add_max_descendant_depth_field():
    """
    اضافه کردن فیلد max_descendant_depth به همه کاربران
    """
    print("=" * 70)
    print("🔧 Migration: Add max_descendant_depth field")
    print("=" * 70)
    
    client = AsyncIOMotorClient(MONGO_URI)
    db = client[DB_NAME]
    
    try:
        await db.command('ping')
        print("✅ Connected to MongoDB\n")
        
        # دریافت همه کاربران
        print("📊 Fetching all users...")
        users = await db.users.find(
            {},
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "status": 1, "max_descendant_depth": 1}
        ).to_list(None)
        
        total_users = len(users)
        print(f"✅ Found {total_users} users\n")
        
        # شمارش کاربرانی که فیلد ندارند
        users_without_field = [u for u in users if "max_descendant_depth" not in u]
        print(f"📊 Users without max_descendant_depth: {len(users_without_field)}")
        
        if len(users_without_field) == 0:
            print("✅ All users already have max_descendant_depth field")
            
            # بررسی اینکه مقادیر صحیح هستند
            print("\n🔍 Verifying values...")
            await recalculate_all_max_depths(db)
            return
        
        # اضافه کردن فیلد با مقدار پیش‌فرض 0
        print(f"\n🔄 Adding field to {len(users_without_field)} users...")
        
        result = await db.users.update_many(
            {"max_descendant_depth": {"$exists": False}},
            {"$set": {"max_descendant_depth": 0}}
        )
        
        print(f"✅ Added field to {result.modified_count} users\n")
        
        # محاسبه مقادیر واقعی
        await recalculate_all_max_depths(db)
        
    except Exception as e:
        print(f"\n❌ Error: {e}")
        import traceback
        traceback.print_exc()
        
    finally:
        client.close()
        print("\n🔒 MongoDB connection closed")


async def recalculate_all_max_depths(db):
    """محاسبه مجدد max_descendant_depth برای همه کاربران"""
    
    print("🔄 Recalculating max_descendant_depth for all users...")
    
    # دریافت همه کاربران فعال
    users = await db.users.find(
        {"status": "active"},
        {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1}
    ).to_list(None)
    
    total = len(users)
    updated = 0
    
    for i, user in enumerate(users, 1):
        user_id = user["user_id"]
        placement_path = user.get("placement_path", "")
        user_depth = user.get("depth", 0)
        
        # یافتن عمیق‌ترین descendant
        if placement_path:
            deepest = await db.users.find_one(
                {
                    "placement_path": {"$regex": f"^{placement_path}/"},
                    "status": "active"
                },
                {"_id": 0, "depth": 1},
                sort=[("depth", -1)]
            )
        else:
            # seed
            deepest = await db.users.find_one(
                {
                    "status": "active",
                    "user_id": {"$ne": "seed"}
                },
                {"_id": 0, "depth": 1},
                sort=[("depth", -1)]
            )
        
        if deepest:
            max_descendant_depth = deepest["depth"] - user_depth
        else:
            max_descendant_depth = 0
        
        # به‌روزرسانی
        await db.users.update_one(
            {"user_id": user_id},
            {"$set": {"max_descendant_depth": max_descendant_depth}}
        )
        
        if max_descendant_depth > 0:
            updated += 1
        
        if i % 10 == 0 or i == total:
            print(f"  Progress: {i}/{total} users processed")
    
    print(f"✅ Recalculated {updated} users with descendants\n")


async def create_indexes():
    """ایجاد index های مورد نیاز برای max_descendant_depth"""
    
    print("=" * 70)
    print("🔧 Creating Required Indexes")
    print("=" * 70)
    
    client = AsyncIOMotorClient(MONGO_URI)
    db = client[DB_NAME]
    
    try:
        # Index برای check_tree_completion
        print("\n📊 Creating index: max_descendant_depth + is_tree_complete...")
        await db.users.create_index(
            [("max_descendant_depth", 1), ("is_tree_complete", 1), ("status", 1)],
            name="max_depth_tree_complete",
            background=True
        )
        print("✅ Index created")
        
        # Index برای audit logs
        print("\n📊 Creating index: audit_logs - user_id + timestamp...")
        await db.audit_logs.create_index(
            [("user_id", 1), ("timestamp", -1)],
            name="audit_user_time",
            background=True
        )
        print("✅ Index created")
        
        print("\n📊 Creating index: audit_logs - event_type...")
        await db.audit_logs.create_index(
            [("event_type", 1)],
            name="audit_event_type",
            background=True
        )
        print("✅ Index created")
        
    except Exception as e:
        if "already exists" in str(e):
            print(f"⏭️  Index already exists")
        else:
            print(f"❌ Error creating index: {e}")
    
    finally:
        client.close()


async def verify_migration():
    """بررسی صحت migration"""
    
    print("\n" + "=" * 70)
    print("🔍 Verification")
    print("=" * 70)
    
    client = AsyncIOMotorClient(MONGO_URI)
    db = client[DB_NAME]
    
    try:
        # بررسی کاربران بدون فیلد
        without_field = await db.users.count_documents({
            "max_descendant_depth": {"$exists": False}
        })
        
        print(f"\n📊 Users without max_descendant_depth: {without_field}")
        
        if without_field == 0:
            print("✅ All users have the field!")
        else:
            print(f"⚠️  {without_field} users still missing the field")
        
        # نمایش چند نمونه
        print("\n📊 Sample users with max_descendant_depth:")
        samples = await db.users.find(
            {"status": "active", "max_descendant_depth": {"$gt": 0}},
            {"_id": 0, "user_id": 1, "depth": 1, "max_descendant_depth": 1, "is_tree_complete": 1}
        ).limit(5).to_list(5)
        
        for sample in samples:
            complete_status = "✅ Complete" if sample.get("is_tree_complete") else "❌ Incomplete"
            print(f"  • {sample['user_id']}: depth={sample['depth']}, max_depth={sample['max_descendant_depth']} {complete_status}")
        
        # بررسی index ها
        print("\n📊 Checking indexes...")
        indexes = await db.users.index_information()
        
        required_indexes = ["max_depth_tree_complete"]
        for idx in required_indexes:
            if idx in indexes:
                print(f"  ✅ {idx}")
            else:
                print(f"  ❌ {idx} (missing)")
        
    finally:
        client.close()


async def main():
    print("""
╔══════════════════════════════════════════════════════════╗
║      Migration: Add max_descendant_depth Field          ║
║                                                          ║
║  این اسکریپت:                                           ║
║  1. فیلد max_descendant_depth را اضافه می‌کند          ║
║  2. مقادیر را برای همه کاربران محاسبه می‌کند          ║
║  3. Index های لازم را ایجاد می‌کند                     ║
╚══════════════════════════════════════════════════════════╝
""")
    
    response = input("\n⚠️  آیا از اجرای migration مطمئن هستید؟ (yes/no): ")
    
    if response.lower() not in ['yes', 'y']:
        print("❌ Migration لغو شد")
        return
    
    # اجرا
    await add_max_descendant_depth_field()
    await create_indexes()
    await verify_migration()
    
    print("\n" + "=" * 70)
    print("✅ Migration completed successfully!")
    print("=" * 70)


if __name__ == "__main__":
    asyncio.run(main())
