import { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from './ui/dialog';

const API = process.env.REACT_APP_BACKEND_URL || '';

export default function TicketsPage() {
  const [tickets, setTickets] = useState([]);
  const [showCreate, setShowCreate] = useState(false);
  const [selectedTicket, setSelectedTicket] = useState(null);
  const [messages, setMessages] = useState([]);
  const [newTicket, setNewTicket] = useState({ subject: '', message: '' });
  const [replyMessage, setReplyMessage] = useState('');

  useEffect(() => {
    fetchTickets();
  }, []);

  const fetchTickets = async () => {
    try {
      const res = await axios.get(`${API}/api/user/tickets`);
      setTickets(res.data);
    } catch (error) {
      toast.error('خطا در دریافت تیکت‌ها');
    }
  };

  const createTicket = async (e) => {
    e.preventDefault();
    try {
      await axios.post(`${API}/api/user/tickets`, newTicket);
      toast.success('تیکت ایجاد شد');
      setShowCreate(false);
      setNewTicket({ subject: '', message: '' });
      fetchTickets();
    } catch (error) {
      toast.error('خطا در ایجاد تیکت');
    }
  };

  const openTicket = async (ticketId) => {
    try {
      const res = await axios.get(`${API}/api/user/tickets/${ticketId}/messages`);
      setSelectedTicket(res.data.ticket);
      setMessages(res.data.messages);
    } catch (error) {
      toast.error('خطا در دریافت پیام‌ها');
    }
  };

  const sendReply = async (e) => {
    e.preventDefault();
    try {
      await axios.post(`${API}/api/user/tickets/${selectedTicket.ticket_id}/reply`, {
        message: replyMessage
      });
      toast.success('پاسخ ارسال شد');
      setReplyMessage('');
      openTicket(selectedTicket.ticket_id);
    } catch (error) {
      toast.error('خطا در ارسال پاسخ');
    }
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-bold">تیکت‌های پشتیبانی</h1>
        <Button onClick={() => setShowCreate(true)}>تیکت جدید</Button>
      </div>

      <div className="grid gap-4">
        {tickets.map((ticket) => (
          <Card key={ticket.ticket_id} className="cursor-pointer hover:shadow-md" onClick={() => openTicket(ticket.ticket_id)}>
            <CardContent className="p-4">
              <div className="flex justify-between items-center">
                <div>
                  <h3 className="font-semibold">{ticket.subject}</h3>
                  <p className="text-sm text-gray-500">{new Date(ticket.created_at).toLocaleDateString('fa-IR')}</p>
                </div>
                <span className={`px-3 py-1 rounded-full text-sm ${ticket.status === 'open' ? 'bg-green-100 text-green-700' : 'bg-gray-100'}`}>
                  {ticket.status === 'open' ? 'باز' : 'بسته'}
                </span>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Create Dialog */}
      <Dialog open={showCreate} onOpenChange={setShowCreate}>
        <DialogContent>
          <DialogHeader><DialogTitle>تیکت جدید</DialogTitle></DialogHeader>
          <form onSubmit={createTicket} className="space-y-4">
            <div><Label>موضوع</Label><Input value={newTicket.subject} onChange={(e) => setNewTicket({...newTicket, subject: e.target.value})} required /></div>
            <div><Label>پیام</Label><Textarea value={newTicket.message} onChange={(e) => setNewTicket({...newTicket, message: e.target.value})} rows={4} required /></div>
            <Button type="submit">ارسال</Button>
          </form>
        </DialogContent>
      </Dialog>

      {/* Messages Dialog */}
      <Dialog open={!!selectedTicket} onOpenChange={() => setSelectedTicket(null)}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader><DialogTitle>{selectedTicket?.subject}</DialogTitle></DialogHeader>
          <div className="space-y-4">
            {messages.map((msg) => (
              <div key={msg.message_id} className={`p-3 rounded ${msg.sender_role === 'admin' ? 'bg-blue-50' : 'bg-gray-50'}`}>
                <div className="font-semibold text-sm">{msg.sender_role === 'admin' ? 'پشتیبانی' : 'شما'}</div>
                <p className="mt-1">{msg.message}</p>
                <p className="text-xs text-gray-500 mt-1">{new Date(msg.created_at).toLocaleString('fa-IR')}</p>
              </div>
            ))}
          </div>
          {selectedTicket?.status === 'open' && (
            <form onSubmit={sendReply} className="mt-4 space-y-2">
              <Textarea value={replyMessage} onChange={(e) => setReplyMessage(e.target.value)} placeholder="پاسخ شما..." rows={3} required />
              <Button type="submit">ارسال پاسخ</Button>
            </form>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
