import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { Copy, CheckCircle, Clock, AlertCircle } from 'lucide-react';
import { QRCodeSVG } from 'qrcode.react';

const API = process.env.REACT_APP_BACKEND_URL || '';

export default function CryptoPaymentPage() {
  const [paymentData, setPaymentData] = useState(null);
  const [availableNetworks, setAvailableNetworks] = useState([]);
  const [selectedNetwork, setSelectedNetwork] = useState(null);
  const [loading, setLoading] = useState(true);
  const [checking, setChecking] = useState(false);
  const [cooldown, setCooldown] = useState(0);

  useEffect(() => {
    fetchPaymentData();
    // حذف چک خودکار - فقط وقتی کاربر دکمه را بزند
  }, []);

  // Cooldown timer
  useEffect(() => {
    if (cooldown > 0) {
      const timer = setTimeout(() => setCooldown(cooldown - 1), 1000);
      return () => clearTimeout(timer);
    }
  }, [cooldown]);

  const fetchPaymentData = async (network = null) => {
    try {
      const token = localStorage.getItem('token');
      
      // دریافت شبکه‌های فعال
      const networksRes = await axios.get(`${API}/api/user/crypto-payment/networks`);
      const networks = networksRes.data.networks || [];
      setAvailableNetworks(networks);
      
      // بررسی وضعیت پرداخت فعلی
      const res = await axios.get(`${API}/api/user/crypto-payment/status`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      if (res.data.status === 'none') {
        // ایجاد درخواست پرداخت جدید
        const createPayload = {};
        if (network) {
          createPayload.network = network;
        }
        
        const createRes = await axios.post(`${API}/api/user/crypto-payment/create`, createPayload, {
          headers: { Authorization: `Bearer ${token}` }
        });
        setPaymentData(createRes.data);
        setSelectedNetwork(createRes.data.blockchain);
      } else {
        setPaymentData(res.data);
        setSelectedNetwork(res.data.blockchain);
      }
    } catch (error) {
      toast.error('خطا در دریافت اطلاعات پرداخت');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const handleNetworkChange = async (network) => {
    setLoading(true);
    setSelectedNetwork(network);
    
    try {
      const token = localStorage.getItem('token');
      const createRes = await axios.post(`${API}/api/user/crypto-payment/create`, 
        { network }, 
        { headers: { Authorization: `Bearer ${token}` } }
      );
      setPaymentData(createRes.data);
      toast.success('شبکه تغییر کرد');
    } catch (error) {
      toast.error('خطا در تغییر شبکه');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const checkMyTransaction = async () => {
    if (checking || cooldown > 0) return;
    
    setChecking(true);
    setCooldown(20); // 20 ثانیه cooldown
    
    try {
      const token = localStorage.getItem('token');
      const res = await axios.post(
        `${API}/api/user/crypto-payment/check-my-transaction`,
        {},
        { headers: { Authorization: `Bearer ${token}` } }
      );
      
      if (res.data.success && res.data.status === 'confirmed') {
        toast.success('🎉 ' + res.data.message, { duration: 5000 });
        
        // reload کامل صفحه بعد از 2 ثانیه
        // App.js خودش user را fetch می‌کند و به dashboard می‌برد
        setTimeout(() => {
          window.location.reload();
        }, 2000);
      } else {
        toast.error(res.data.message || 'تراکنش هنوز تایید نشده است', { duration: 5000 });
        // رفرش اطلاعات
        await fetchPaymentData();
      }
    } catch (error) {
      console.error('Error checking transaction:', error);
      toast.error('خطا در بررسی تراکنش. لطفاً دوباره تلاش کنید.');
    } finally {
      setChecking(false);
    }
  };

  const copyAddress = () => {
    if (paymentData?.crypto_address) {
      navigator.clipboard.writeText(paymentData.crypto_address);
      toast.success('✅ آدرس کپی شد');
    }
  };

  if (loading) {
    return (
      <div className="p-6 text-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-emerald-500 mx-auto"></div>
        <p className="mt-4 text-gray-600">در حال بارگذاری...</p>
      </div>
    );
  }

  return (
    <div className="p-6 max-w-2xl mx-auto space-y-6">
      <Card className="shadow-lg border-emerald-200">
        <CardHeader className="bg-gradient-to-r from-emerald-500 to-teal-600 text-white">
          <CardTitle className="text-2xl">💳 پرداخت ارز دیجیتال</CardTitle>
          <CardDescription className="text-emerald-50">
            برای فعال‌سازی حساب خود، مبلغ زیر را از طریق USDT پرداخت کنید
          </CardDescription>
        </CardHeader>
        <CardContent className="p-6 space-y-6">
          {paymentData?.status === 'pending' ? (
            <>
              {/* انتخاب شبکه (اگر بیش از یک شبکه فعال است) */}
              {availableNetworks.length > 1 && (
                <div className="p-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-2 border-blue-200 rounded-xl">
                  <label className="block text-sm font-semibold mb-3 text-gray-700">
                    🌐 انتخاب شبکه پرداخت:
                  </label>
                  <div className="grid grid-cols-2 gap-3">
                    {availableNetworks.map((network) => (
                      <button
                        key={network.network}
                        onClick={() => handleNetworkChange(network.network)}
                        disabled={loading}
                        className={`p-4 rounded-lg border-2 transition-all ${
                          selectedNetwork === network.network
                            ? 'border-emerald-500 bg-emerald-50 shadow-lg'
                            : 'border-gray-300 bg-white hover:border-gray-400'
                        }`}
                      >
                        <div className="flex items-center justify-center gap-2 mb-2">
                          <div className={`w-8 h-8 rounded-full flex items-center justify-center font-bold text-white ${
                            network.network === 'TRON' ? 'bg-blue-500' : 'bg-yellow-500'
                          }`}>
                            {network.network === 'TRON' ? 'T' : 'B'}
                          </div>
                          <span className="font-bold text-sm">{network.network}</span>
                        </div>
                        <div className="text-xs text-gray-600">{network.code}</div>
                        {selectedNetwork === network.network && (
                          <div className="mt-2 text-emerald-600 text-xs font-semibold">✓ انتخاب شده</div>
                        )}
                      </button>
                    ))}
                  </div>
                </div>
              )}

              {/* مبلغ */}
              <div className="text-center p-6 bg-gradient-to-br from-emerald-50 to-teal-50 rounded-xl border-2 border-emerald-200">
                <div className="text-sm text-gray-600 mb-2">مبلغ قابل پرداخت</div>
                <div className="text-5xl font-bold text-emerald-700 mb-2">
                  {paymentData.unique_amount ? paymentData.unique_amount.toFixed(4) : paymentData.amount} <span className="text-2xl">USDT</span>
                </div>
                
                {/* دکمه کپی مبلغ */}
                {paymentData.unique_amount && (
                  <div className="mt-3 flex items-center justify-center gap-2">
                    <Button
                      onClick={() => {
                        const exactAmount = paymentData.unique_amount.toFixed(4);
                        navigator.clipboard.writeText(exactAmount);
                        toast.success('✅ مبلغ دقیق کپی شد: ' + exactAmount);
                      }}
                      size="sm"
                      className="bg-emerald-600 hover:bg-emerald-700 text-white px-4 py-2"
                    >
                      <Copy className="w-4 h-4 ml-2" />
                      کپی مبلغ دقیق (4 رقم اعشار)
                    </Button>
                  </div>
                )}
                
                <div className="flex items-center justify-center gap-2 mt-3">
                  <div className={`w-6 h-6 rounded-full flex items-center justify-center text-xs font-bold text-white ${
                    selectedNetwork === 'TRON' ? 'bg-blue-500' : 'bg-yellow-500'
                  }`}>
                    {selectedNetwork === 'TRON' ? 'T' : 'B'}
                  </div>
                  <span className="text-sm text-gray-500 bg-white px-3 py-1 rounded-full">
                    شبکه: {paymentData.network}
                  </span>
                </div>
              </div>

              {/* QR Code */}
              {paymentData.crypto_address && (
                <div className="flex justify-center">
                  <div className="bg-white p-6 rounded-xl shadow-lg border-2 border-gray-200">
                    <QRCodeSVG 
                      value={paymentData.crypto_address} 
                      size={220}
                      level="H"
                      includeMargin={true}
                    />
                  </div>
                </div>
              )}

              {/* آدرس کیف پول */}
              <div>
                <label className="block text-sm font-semibold mb-2 text-gray-700">
                  📋 آدرس کیف پول دریافت:
                </label>
                <div className="flex gap-2">
                  <input
                    type="text"
                    value={paymentData.crypto_address || ''}
                    readOnly
                    className="flex-1 p-3 border-2 border-gray-300 rounded-lg bg-gray-50 font-mono text-sm focus:outline-none focus:border-emerald-500"
                  />
                  <Button 
                    onClick={copyAddress} 
                    variant="outline"
                    className="border-emerald-500 text-emerald-600 hover:bg-emerald-50"
                  >
                    <Copy className="w-4 h-4" />
                  </Button>
                </div>
              </div>

              {/* هشدار مبلغ دقیق */}
              {paymentData.unique_amount && (
                <div className="bg-red-50 border-2 border-red-300 p-5 rounded-xl">
                  <h3 className="font-bold text-red-900 flex items-center gap-2 mb-3">
                    <AlertCircle className="w-5 h-5" />
                    ⚠️ بسیار مهم - مبلغ دقیق!
                  </h3>
                  <div className="space-y-2 text-sm text-red-800">
                    <p className="font-semibold">
                      حتماً دقیقاً مبلغ <span className="text-lg font-bold bg-red-200 px-2 py-1 rounded">{paymentData.unique_amount?.toFixed(4)} USDT</span> را واریز کنید.
                    </p>
                    <p>
                      ✓ این مبلغ منحصر به فرد شماست (4 رقم اعشار) و سیستم فقط با این مبلغ دقیق پرداخت شما را تایید می‌کند.
                    </p>
                    <p>
                      ✗ اگر مبلغ متفاوتی واریز کنید (مثلاً {paymentData.amount} USDT یا {(paymentData.unique_amount - 0.0001).toFixed(4)} USDT)، پرداخت شما تایید نمی‌شود!
                    </p>
                    <p className="text-xs mt-3 bg-red-100 p-2 rounded">
                      💡 حتماً از دکمه "کپی مبلغ دقیق" استفاده کنید و مبلغ را copy-paste کنید تا از دقت 100% اطمینان حاصل شود.
                    </p>
                  </div>
                </div>
              )}

              {/* دستورالعمل */}
              <div className="bg-blue-50 border-2 border-blue-200 p-5 rounded-xl space-y-3">
                <h3 className="font-bold text-blue-900 flex items-center gap-2">
                  <AlertCircle className="w-5 h-5" />
                  📌 دستورالعمل پرداخت:
                </h3>
                <ol className="list-decimal list-inside space-y-2 text-sm text-blue-800">
                  <li>کیف پول USDT خود را باز کنید</li>
                  <li>
                    <strong className="text-red-700">
                      دقیقاً مبلغ {paymentData.unique_amount ? paymentData.unique_amount.toFixed(4) : paymentData.amount} USDT
                    </strong> را به آدرس بالا ارسال کنید
                    {paymentData.unique_amount && (
                      <span className="block text-xs mt-1 text-red-600 font-bold">
                        ⚠️ توجه: حتماً 4 رقم اعشار را کامل وارد کنید (مثلاً 20.0021) - از دکمه "کپی مبلغ دقیق" استفاده کنید
                      </span>
                    )}
                  </li>
                  <li><strong>حتماً شبکه {paymentData.network}</strong> را انتخاب کنید</li>
                  <li>بعد از ارسال، منتظر تایید خودکار بمانید (معمولاً 5-10 دقیقه)</li>
                  <li>می‌توانید با دکمه زیر وضعیت را بررسی کنید</li>
                </ol>
              </div>

              {/* هشدار */}
              <div className="bg-amber-50 border-2 border-amber-300 p-4 rounded-lg">
                <p className="text-sm text-amber-800">
                  ⚠️ <strong>توجه:</strong> فقط از شبکه <strong>{paymentData.network}</strong> استفاده کنید.
                  ارسال از شبکه‌های دیگر منجر به از دست رفتن دارایی می‌شود.
                </p>
              </div>

              {/* وضعیت */}
              <div className="flex items-center justify-center gap-3 text-amber-600 bg-amber-50 p-4 rounded-lg">
                <Clock className="w-6 h-6 animate-pulse" />
                <span className="font-semibold">در انتظار پرداخت شما...</span>
                {checking && <span className="text-xs bg-amber-200 px-2 py-1 rounded">(در حال بررسی)</span>}
              </div>

              {/* دکمه بررسی تراکنش */}
              <Button 
                onClick={checkMyTransaction} 
                className="w-full bg-gradient-to-r from-emerald-500 to-green-600 hover:from-emerald-600 hover:to-green-700 text-white font-bold py-6 text-lg shadow-lg"
                disabled={checking || cooldown > 0}
              >
                {checking ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                    در حال بررسی تراکنش...
                  </>
                ) : cooldown > 0 ? (
                  `⏳ صبر کنید... (${cooldown} ثانیه)`
                ) : (
                  '✅ واریز کردم - بررسی تراکنش'
                )}
              </Button>
              
              {/* راهنمای استفاده */}
              <div className="bg-blue-50 border-2 border-blue-200 p-4 rounded-lg text-center">
                <p className="text-sm text-blue-800">
                  <strong>📌 بعد از واریز USDT:</strong><br/>
                  روی دکمه "واریز کردم" کلیک کنید تا سیستم تراکنش شما را بررسی کند.
                  معمولاً 2-5 دقیقه طول می‌کشد تا تراکنش در بلاکچین تایید شود.
                </p>
              </div>
            </>
          ) : paymentData?.status === 'confirmed' ? (
            <div className="text-center py-12">
              <CheckCircle className="w-20 h-20 text-green-500 mx-auto mb-6 animate-bounce" />
              <h3 className="text-3xl font-bold text-green-700 mb-3">✅ پرداخت تایید شد!</h3>
              <p className="text-gray-600 text-lg">حساب شما با موفقیت فعال شد</p>
              <p className="text-sm text-gray-500 mt-2">در حال انتقال به داشبورد...</p>
            </div>
          ) : (
            <div className="text-center py-12">
              <AlertCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
              <p className="text-gray-600">خطا در دریافت اطلاعات پرداخت</p>
              <Button onClick={fetchPaymentData} variant="outline" className="mt-4">
                تلاش مجدد
              </Button>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
