import { useState, useEffect } from 'react';
import axios from 'axios';
import { ChevronLeft, ChevronRight, ExternalLink } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AnnouncementSlider() {
  const [announcements, setAnnouncements] = useState([]);
  const [currentIndex, setCurrentIndex] = useState(0);

  useEffect(() => {
    fetchAnnouncements();
  }, []);

  useEffect(() => {
    if (announcements.length > 1) {
      const interval = setInterval(() => {
        setCurrentIndex((prev) => (prev + 1) % announcements.length);
      }, 5000); // تغییر هر 5 ثانیه
      return () => clearInterval(interval);
    }
  }, [announcements]);

  const fetchAnnouncements = async () => {
    try {
      const res = await axios.get(`${API}/user/announcements/slider`);
      setAnnouncements(res.data);
    } catch (error) {
      console.error('Error fetching announcements:', error);
    }
  };

  const handleNext = () => {
    setCurrentIndex((prev) => (prev + 1) % announcements.length);
  };

  const handlePrev = () => {
    setCurrentIndex((prev) => (prev - 1 + announcements.length) % announcements.length);
  };

  if (announcements.length === 0) {
    return null;
  }

  const current = announcements[currentIndex];

  // تابع برای اطمینان از صحت فرمت لینک
  const normalizeUrl = (url) => {
    if (!url) return '';
    // اگر قبلاً هر نوع پروتکلی دارد (http://, https://, ftp://, وغیره)، دست نزن
    if (url.includes('://')) {
      return url;
    }
    // در غیر این صورت، https:// اضافه کن
    return `https://${url}`;
  };

  return (
    <div className="relative bg-gradient-to-r from-blue-500 to-purple-600 rounded-xl p-4 text-white shadow-lg">
      <div className="flex items-center justify-between gap-4">
        {announcements.length > 1 && (
          <button
            onClick={handlePrev}
            className="p-2 hover:bg-white/20 rounded-full transition-colors"
          >
            <ChevronRight className="w-5 h-5" />
          </button>
        )}

        <div className="flex-1 text-center">
          <h3 className="font-bold text-lg mb-1">{current.title}</h3>
          <p className="text-sm text-blue-50">{current.content}</p>
          {current.link && (
            <a
              href={normalizeUrl(current.link)}
              target="_blank"
              rel="noopener noreferrer"
              className="inline-flex items-center gap-1 mt-2 text-sm underline hover:text-blue-100"
            >
              مشاهده بیشتر
              <ExternalLink className="w-3 h-3" />
            </a>
          )}
        </div>

        {announcements.length > 1 && (
          <button
            onClick={handleNext}
            className="p-2 hover:bg-white/20 rounded-full transition-colors"
          >
            <ChevronLeft className="w-5 h-5" />
          </button>
        )}
      </div>

      {announcements.length > 1 && (
        <div className="flex justify-center gap-2 mt-3">
          {announcements.map((_, idx) => (
            <button
              key={idx}
              onClick={() => setCurrentIndex(idx)}
              className={`w-2 h-2 rounded-full transition-all ${
                idx === currentIndex ? 'bg-white w-6' : 'bg-white/50'
              }`}
            />
          ))}
        </div>
      )}
    </div>
  );
}
