"""Tree Statistics Service - آمار درخت بدون traversal"""

from typing import Dict, List
import logging
from .path_manager import PathManager

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class TreeStatsService:
    """سرویس آمار درخت"""
    
    @staticmethod
    async def update_descendants_count(placement_path: str, new_user_depth: int, increment: int = 1):
        """
        به‌روزرسانی تعداد descendants و max_descendant_depth برای همه ancestors
        
        Args:
            placement_path: path کاربر جدید
            new_user_depth: depth کاربر جدید
            increment: مقدار افزایش (معمولاً 1)
            
        Queries: 1 (bulk update)
        Time: O(1)
        """
        # استخراج ancestors از path (بدون query)
        ancestor_ids = PathManager.extract_ancestors(placement_path)
        
        if not ancestor_ids:
            return
        
        # محاسبه max_descendant_depth برای هر ancestor
        # برای ancestor در depth X، اگر کاربر جدید depth Y داره، depth_diff = Y - X
        from pymongo import UpdateOne
        
        # ابتدا ancestors را دریافت کنیم تا depth هایشان را بدانیم
        ancestors_data = await _db.users.find(
            {"user_id": {"$in": ancestor_ids}},
            {"_id": 0, "user_id": 1, "depth": 1}
        ).to_list(len(ancestor_ids))
        
        # ساخت bulk operations
        bulk_ops = []
        
        for ancestor in ancestors_data:
            ancestor_depth = ancestor["depth"]
            depth_diff = new_user_depth - ancestor_depth
            
            # به‌روزرسانی total_descendants و max_descendant_depth
            bulk_ops.append(
                UpdateOne(
                    {"user_id": ancestor["user_id"]},
                    {
                        "$inc": {"total_descendants": increment},
                        "$max": {"max_descendant_depth": depth_diff}  # فقط اگر بزرگ‌تر باشد
                    }
                )
            )
        
        if bulk_ops:
            result = await _db.users.bulk_write(bulk_ops)
            logger.info(f"✅ Updated descendants count and max_depth for {len(ancestor_ids)} ancestors")
    
    @staticmethod
    async def check_tree_completion(placement_path: str):
        """
        بررسی تکمیل هرم برای ancestors - با استفاده از max_descendant_depth
        
        هرم 7 سطحی زمانی کامل است که:
        max_descendant_depth >= 6 (یعنی حداقل یک نفر در depth +6 وجود دارد)
        
        Args:
            placement_path: path کاربر جدید
            
        Queries: 1 (bulk update با شرط max_descendant_depth)
        Time: O(1) - بسیار سریع!
        """
        # استخراج ancestors
        ancestor_ids = PathManager.extract_ancestors(placement_path)
        
        if not ancestor_ids:
            return
        
        # 🚀 بهینه‌سازی: فقط با یک query، کسانی که max_descendant_depth >= 6 دارند را complete می‌کنیم
        # دیگر نیازی به regex query نیست!
        result = await _db.users.update_many(
            {
                "user_id": {"$in": ancestor_ids},
                "max_descendant_depth": {"$gte": 6},  # سطح 7 پر شده
                "is_tree_complete": False,
                "status": "active"
            },
            {"$set": {"is_tree_complete": True}}
        )
        
        if result.modified_count > 0:
            logger.info(f"✅ {result.modified_count} users completed their 7-level tree (max_descendant_depth >= 6)!")

    
    @staticmethod
    async def get_tree_for_display(user_id: str, max_depth: int = 3) -> Dict:
        """
        دریافت درخت برای نمایش - با استفاده از path query
        
        Args:
            user_id: شناسه کاربر ریشه
            max_depth: حداکثر عمق برای نمایش
            
        Returns:
            درخت به صورت nested dict
            
        Note: این متد برای نمایش UI است - نه برای محاسبات
        """
        # دریافت کاربر ریشه
        root_user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0, "user_id": 1, "email": 1, "status": 1, "depth": 1, 
             "total_points": 1, "direct_children_count": 1, "placement_path": 1}
        )
        
        if not root_user:
            return None
        
        root_path = root_user.get("placement_path", user_id)
        root_depth = root_user.get("depth", 0)
        
        # دریافت همه descendants تا max_depth با یک query!
        # استفاده از regex برای match کردن path
        descendants = await _db.users.find(
            {
                "placement_path": {"$regex": f"^{root_path}\\."}, # زیردرخت
                "depth": {"$lte": root_depth + max_depth}  # محدودیت عمق
            },
            {"_id": 0, "user_id": 1, "email": 1, "status": 1, "depth": 1,
             "total_points": 1, "direct_children_count": 1, "placement_parent_id": 1}
        ).to_list(1000)
        
        # ساخت درخت در حافظه (بدون recursion)
        nodes = {root_user["user_id"]: {
            "user_id": root_user["user_id"],
            "email": root_user["email"],
            "status": root_user["status"],
            "depth": root_user["depth"],
            "total_points": root_user["total_points"],
            "direct_children_count": root_user["direct_children_count"],
            "children": []
        }}
        
        for desc in descendants:
            nodes[desc["user_id"]] = {
                "user_id": desc["user_id"],
                "email": desc["email"],
                "status": desc["status"],
                "depth": desc["depth"],
                "total_points": desc["total_points"],
                "direct_children_count": desc["direct_children_count"],
                "children": []
            }
        
        # اتصال parent-child
        for desc in descendants:
            parent_id = desc.get("placement_parent_id")
            if parent_id and parent_id in nodes:
                nodes[parent_id]["children"].append(nodes[desc["user_id"]])
        
        return nodes[root_user["user_id"]]
    
    @staticmethod
    async def get_descendants_count(user_id: str) -> Dict:
        """
        دریافت تعداد descendants - بدون traversal!
        
        Args:
            user_id: شناسه کاربر
            
        Returns:
            آمار descendants
        """
        user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0, "total_descendants": 1}
        )
        
        if not user:
            return {"total": 0}
        
        return {
            "total": user.get("total_descendants", 0)
        }
