"""Placement Service - جایگذاری کاربر در درخت بدون BFS"""

from typing import Dict, Optional
import logging
from .path_manager import PathManager

logger = logging.getLogger(__name__)

# Database connection (باید از بیرون set شود)
_db = None


def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class PlacementService:
    """سرویس جایگذاری کاربر - بدون BFS"""
    
    @staticmethod
    async def find_available_parent(referrer_id: str) -> Optional[Dict]:
        """
        Two-Phase Placement Algorithm for MLM 3x7 Matrix
        
        Phase 1 (Priority): Local placement in referrer's subtree
        Phase 2 (Spillover): Global placement if subtree is full
        
        Rules:
        - No referral code → referrer_id = "seed"
        - With referral code → search in referrer's subtree first
        - If subtree full → spillover to nearest available slot
        - Sponsor always tracked for Direct Rewards
        
        Args:
            referrer_id: شناسه معرف (sponsor)
            
        Returns:
            Parent placement info or None
            {
                "parent_id": str,
                "parent_path": str,
                "parent_depth": int,
                "position": int (0-2)
            }
        """
        
        # ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        # PHASE 1: LOCAL PLACEMENT (Referrer's Subtree Priority)
        # ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        
        # Step 1.1: Check if referrer has direct space
        referrer = await _db.users.find_one(
            {
                "user_id": referrer_id,
                "status": "active",
                "direct_children_count": {"$lt": 3},
                "is_tree_complete": False  # نباید tree کامل باشد
            },
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "direct_children_count": 1}
        )
        
        if referrer:
            logger.info(f"✅ [Phase 1] Referrer {referrer_id} has direct space")
            return {
                "parent_id": referrer["user_id"],
                "parent_path": referrer.get("placement_path"),
                "parent_depth": referrer.get("depth", 0),
                "position": referrer.get("direct_children_count", 0)
            }
        
        # Step 1.2: Search in referrer's subtree (BFS by depth)
        referrer_data = await _db.users.find_one(
            {"user_id": referrer_id, "status": "active"},
            {"_id": 0, "placement_path": 1}
        )
        
        if not referrer_data:
            logger.error(f"❌ Referrer {referrer_id} not found or inactive")
            # Fall through to Phase 2
        else:
            referrer_path = referrer_data.get("placement_path")
            
            if referrer_path is None:
                logger.error(f"❌ Referrer {referrer_id} has no placement_path")
                # Fall through to Phase 2
            else:
                # Build subtree query
                if referrer_path == "":
                    # Referrer is seed (root) - entire tree is subtree
                    subtree_query = {
                        "status": "active",
                        "direct_children_count": {"$lt": 3},
                        "is_tree_complete": False  # نباید tree کامل باشد
                    }
                else:
                    # Search only in referrer's subtree
                    subtree_query = {
                        "status": "active",
                        "placement_path": {"$regex": f"^{referrer_path}/"},
                        "direct_children_count": {"$lt": 3},
                        "is_tree_complete": False  # نباید tree کامل باشد
                    }
                
                local_parent = await _db.users.find_one(
                    subtree_query,
                    {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "direct_children_count": 1},
                    sort=[("depth", 1)]  # Closest to referrer first
                )
                
                if local_parent:
                    logger.info(f"✅ [Phase 1] Found in subtree: {local_parent['user_id']} at depth {local_parent['depth']}")
                    return {
                        "parent_id": local_parent["user_id"],
                        "parent_path": local_parent.get("placement_path"),
                        "parent_depth": local_parent.get("depth", 0),
                        "position": local_parent.get("direct_children_count", 0)
                    }
        
        # Subtree is full - proceed to spillover
        logger.warning(f"⚠️ Referrer {referrer_id} subtree is full - activating spillover")
        
        # ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        # PHASE 2: GLOBAL SPILLOVER (Nearest Available Slot)
        # ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
        
        global_parent = await _db.users.find_one(
            {
                "status": "active",
                "direct_children_count": {"$lt": 3},
                "is_tree_complete": False  # نباید tree کامل باشد
            },
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "direct_children_count": 1},
            sort=[("depth", 1)]  # Fair distribution - closest to top
        )
        
        if global_parent:
            logger.info(f"✅ [Phase 2 - Spillover] Placed at {global_parent['user_id']} (depth {global_parent['depth']})")
            return {
                "parent_id": global_parent["user_id"],
                "parent_path": global_parent.get("placement_path"),
                "parent_depth": global_parent.get("depth", 0),
                "position": global_parent.get("direct_children_count", 0)
            }
        
        # Critical: Entire system is full (should never happen with 3^7 = 2,187 per branch)
        logger.error("❌ CRITICAL: No available placement in entire system!")
        return None
    
    @staticmethod
    async def place_user_in_tree(user_id: str, parent_info: Dict) -> Dict:
        """
        محاسبه اطلاعات placement برای کاربر - بدون query
        
        Args:
            user_id: شناسه کاربر جدید
            parent_info: اطلاعات parent از find_available_parent
            
        Returns:
            اطلاعات placement
            {
                "placement_path": str,
                "placement_parent_id": str,
                "placement_position": int,
                "depth": int,
                "ancestor_ids": List[str]
            }
        """
        parent_id = parent_info["parent_id"]
        parent_path = parent_info["parent_path"]
        parent_depth = parent_info["parent_depth"]
        position = parent_info["position"]
        
        # ساخت path جدید (محاسبه ساده - بدون query)
        new_path = PathManager.build_child_path(parent_path, user_id)
        
        # محاسبه depth (محاسبه ساده - بدون query)
        new_depth = parent_depth + 1
        
        # استخراج ancestors (محاسبه ساده - بدون query)
        ancestor_ids = PathManager.extract_ancestors(new_path)
        
        # 🔧 FIX: seed همیشه ancestor همه کاربران است (به جز خود seed)
        # برای کاربر a (مستقیم زیر seed): path="a" → ancestor_ids=[] → باید seed اضافه شود
        # برای کاربر x (زیر a): path="a/x" → ancestor_ids=["a"] → باید seed اضافه شود
        if user_id != "seed" and "seed" not in ancestor_ids:
            ancestor_ids.insert(0, "seed")
        
        logger.info(f"📍 Placed {user_id} under {parent_id}: path={new_path}, depth={new_depth}")
        
        return {
            "placement_path": new_path,
            "placement_parent_id": parent_id,
            "placement_position": position,
            "depth": new_depth,
            "ancestor_ids": ancestor_ids
        }
    
    @staticmethod
    async def reserve_slot_atomic(parent_id: str, child_id: str, position: int) -> bool:
        """
        Reserve کردن atomic یک slot در parent
        
        Args:
            parent_id: شناسه parent
            child_id: شناسه child جدید
            position: موقعیت (0, 1, 2)
            
        Returns:
            True اگر موفق باشد
        """
        result = await _db.users.find_one_and_update(
            {
                "user_id": parent_id,
                "status": "active",
                "direct_children_count": {"$lt": 3},
                "is_tree_complete": False  # نباید tree کامل باشد
            },
            {
                "$push": {"direct_children": child_id},
                "$inc": {"direct_children_count": 1}
            },
            return_document=True
        )
        
        return result is not None
