"""Tickets Routes - اندپوینت‌های تیکت"""

import uuid
from datetime import datetime, timezone

from fastapi import APIRouter, HTTPException, Depends

from models.ticket import TicketCreate, TicketReply
from services.auth_service import get_current_user, get_admin_user
from database import get_db

router = APIRouter()


@router.post("/user/tickets")
async def create_ticket(data: TicketCreate, current_user: dict = Depends(get_current_user)):
    """ایجاد تیکت جدید"""
    db = await get_db()
    
    ticket = {
        "ticket_id": str(uuid.uuid4()),
        "user_id": current_user["user_id"],
        "user_email": current_user["email"],
        "subject": data.subject,
        "priority": data.priority,
        "status": "open",
        "messages": [{
            "message_id": str(uuid.uuid4()),
            "sender_id": current_user["user_id"],
            "sender_email": current_user["email"],
            "sender_role": current_user["role"],
            "message": data.message,
            "timestamp": datetime.now(timezone.utc).isoformat()
        }],
        "created_at": datetime.now(timezone.utc).isoformat(),
        "updated_at": datetime.now(timezone.utc).isoformat(),
        "closed_at": None
    }
    
    await db.tickets.insert_one(ticket)
    
    return {"ticket_id": ticket["ticket_id"], "message": "تیکت ایجاد شد"}


@router.get("/user/tickets")
async def get_user_tickets(current_user: dict = Depends(get_current_user)):
    """لیست تیکت‌های کاربر"""
    db = await get_db()
    
    tickets = await db.tickets.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("updated_at", -1).to_list(100)
    
    return tickets


@router.get("/user/tickets/{ticket_id}")
async def get_ticket_detail(ticket_id: str, current_user: dict = Depends(get_current_user)):
    """جزئیات تیکت"""
    db = await get_db()
    
    ticket = await db.tickets.find_one(
        {"ticket_id": ticket_id, "user_id": current_user["user_id"]},
        {"_id": 0}
    )
    
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    return ticket


@router.post("/user/tickets/{ticket_id}/reply")
async def reply_to_ticket(ticket_id: str, data: TicketReply, current_user: dict = Depends(get_current_user)):
    """پاسخ به تیکت"""
    db = await get_db()
    
    ticket = await db.tickets.find_one({"ticket_id": ticket_id, "user_id": current_user["user_id"]})
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    if ticket["status"] == "closed":
        raise HTTPException(status_code=400, detail="تیکت بسته شده است")
    
    new_message = {
        "message_id": str(uuid.uuid4()),
        "sender_id": current_user["user_id"],
        "sender_email": current_user["email"],
        "sender_role": current_user["role"],
        "message": data.message,
        "timestamp": datetime.now(timezone.utc).isoformat()
    }
    
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {
            "$push": {"messages": new_message},
            "$set": {
                "updated_at": datetime.now(timezone.utc).isoformat(),
                "status": "open"
            }
        }
    )
    
    return {"message": "پاسخ ارسال شد"}


# Admin Routes

@router.get("/admin/tickets")
async def get_all_tickets(status: str = None, admin_user: dict = Depends(get_admin_user)):
    """لیست همه تیکت‌ها"""
    db = await get_db()
    
    filter_query = {}
    if status:
        filter_query["status"] = status
    
    tickets = await db.tickets.find(
        filter_query,
        {"_id": 0}
    ).sort("updated_at", -1).to_list(200)
    
    return tickets


@router.get("/admin/tickets/{ticket_id}")
async def admin_get_ticket(ticket_id: str, admin_user: dict = Depends(get_admin_user)):
    """جزئیات تیکت برای ادمین"""
    db = await get_db()
    
    ticket = await db.tickets.find_one({"ticket_id": ticket_id}, {"_id": 0})
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    return ticket


@router.post("/admin/tickets/{ticket_id}/reply")
async def admin_reply_ticket(ticket_id: str, data: TicketReply, admin_user: dict = Depends(get_admin_user)):
    """پاسخ ادمین به تیکت"""
    db = await get_db()
    
    new_message = {
        "message_id": str(uuid.uuid4()),
        "sender_id": admin_user["user_id"],
        "sender_email": admin_user["email"],
        "sender_role": "admin",
        "message": data.message,
        "timestamp": datetime.now(timezone.utc).isoformat()
    }
    
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {
            "$push": {"messages": new_message},
            "$set": {
                "updated_at": datetime.now(timezone.utc).isoformat(),
                "status": "answered"
            }
        }
    )
    
    return {"message": "پاسخ ارسال شد"}


@router.put("/admin/tickets/{ticket_id}/close")
async def close_ticket(ticket_id: str, admin_user: dict = Depends(get_admin_user)):
    """بستن تیکت"""
    db = await get_db()
    
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {"$set": {
            "status": "closed",
            "closed_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "تیکت بسته شد"}
