"""Reward Calculator - محاسبه پاداش بدون traversal"""

from typing import Dict, List, Optional
import logging

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class RewardCalculator:
    """محاسبه‌گر پاداش - real-time و بدون traversal"""
    
    @staticmethod
    async def calculate_direct_reward(
        referrer_id: str,
        new_user_id: str,
        settings: Dict
    ) -> Optional[Dict]:
        """
        محاسبه پاداش مستقیم
        
        Args:
            referrer_id: شناسه معرف
            new_user_id: شناسه کاربر جدید
            settings: تنظیمات پاداش
            
        Returns:
            اطلاعات پاداش یا None
            {
                "recipient_id": str,
                "amount": float,
                "reward_type": "direct",
                "description": str
            }
            
        Queries: 2
        Time: O(1)
        """
        
        # Query 1: Get referrer
        referrer = await _db.users.find_one(
            {"user_id": referrer_id},
            {"_id": 0, "user_id": 1, "blocked_direct_reward": 1, "is_tree_complete": 1}
        )
        
        if not referrer:
            logger.warning(f"⚠️ Referrer {referrer_id} not found")
            return None
        
        # Check blocks
        if referrer.get("blocked_direct_reward") or referrer.get("is_tree_complete"):
            logger.info(f"🚫 Direct reward blocked for {referrer_id}")
            return None
        
        # Query 2: Count active direct children (excluding new user)
        active_children_count = await _db.users.count_documents({
            "referrer_id": referrer_id,
            "status": "active",
            "user_id": {"$ne": new_user_id}
        })
        
        # محاسبه مبلغ بر اساس ترتیب
        if active_children_count == 0:
            amount = settings.get("direct_first", 100.0)
            desc = "مستقیم نفر اول"
        elif active_children_count == 1:
            amount = settings.get("direct_second", 75.0)
            desc = "مستقیم نفر دوم"
        elif active_children_count == 2:
            amount = settings.get("direct_third", 50.0)
            desc = "مستقیم نفر سوم"
        else:
            amount = settings.get("direct_extra", 25.0)
            desc = "مستقیم اضافی"
        
        logger.info(f"💰 Direct reward for {referrer_id}: ${amount} ({desc})")
        
        return {
            "recipient_id": referrer_id,
            "amount": amount,
            "reward_type": "direct",
            "description": f"پاداش {desc}"
        }
    
    @staticmethod
    async def calculate_level_rewards(
        new_user: Dict,
        settings: Dict
    ) -> List[Dict]:
        """
        محاسبه پاداش‌های سطحی - بدون traversal!
        
        این متد از ancestor_depths استفاده می‌کند که در user موجود است
        هیچ while loop یا query پیمایشی ندارد!
        
        Args:
            new_user: اطلاعات کاربر جدید (با ancestor_depths)
            settings: تنظیمات پاداش
            
        Returns:
            لیست پاداش‌های سطحی
            [{
                "recipient_id": str,
                "amount": float,
                "reward_type": "level_3" | "level_5" | "level_7",
                "description": str
            }, ...]
            
        Queries: 1 (برای validate ancestors)
        Time: O(ancestors) = O(7) = O(1)
        """
        
        rewards_to_distribute = []
        new_user_depth = new_user.get("depth", 0)
        
        # 🚀 استفاده از ancestor_depths - بدون while loop!
        ancestor_depths = new_user.get("ancestor_depths", [])
        
        if not ancestor_depths:
            logger.warning(f"⚠️ No ancestor_depths for user {new_user['user_id']}")
            return []
        
        # محاسبه ریاضی - بدون query!
        for ancestor in ancestor_depths:
            ancestor_id = ancestor["user_id"]
            ancestor_depth = ancestor["depth"]
            
            # محاسبه تفاضل depth
            depth_diff = new_user_depth - ancestor_depth
            
            # محدودیت 7 سطح
            if depth_diff > 7:
                break
            
            # تعیین نوع پاداش
            reward_amount = None
            reward_type = None
            level_name = None
            
            if depth_diff == 2:
                reward_amount = settings.get("level_3", 10.0)
                reward_type = "level_3"
                level_name = "سطح ۳"
                
            elif depth_diff == 4:
                reward_amount = settings.get("level_5", 5.0)
                reward_type = "level_5"
                level_name = "سطح ۵"
                
            elif depth_diff == 6:
                reward_amount = settings.get("level_7", 2.0)
                reward_type = "level_7"
                level_name = "سطح ۷"
            
            if reward_amount:
                rewards_to_distribute.append({
                    "recipient_id": ancestor_id,
                    "amount": reward_amount,
                    "reward_type": reward_type,
                    "description": f"پاداش {level_name} - عمق {new_user_depth}"
                })
        
        logger.info(f"💰 Calculated {len(rewards_to_distribute)} level rewards")
        
        # Query 1: Fetch ancestors برای validate (blocked, tree_complete)
        if rewards_to_distribute:
            ancestor_ids = [r["recipient_id"] for r in rewards_to_distribute]
            ancestors_data = await _db.users.find(
                {"user_id": {"$in": ancestor_ids}},
                {"_id": 0, "user_id": 1, "blocked_level_reward": 1, "is_tree_complete": 1}
            ).to_list(len(ancestor_ids))
            
            # فیلتر کردن ancestors مسدود شده
            blocked_ids = {
                a["user_id"]
                for a in ancestors_data
                if a.get("blocked_level_reward") or a.get("is_tree_complete")
            }
            
            # حذف rewards برای مسدود شده‌ها
            rewards_to_distribute = [
                r for r in rewards_to_distribute
                if r["recipient_id"] not in blocked_ids
            ]
            
            logger.info(f"✅ {len(rewards_to_distribute)} rewards after filtering blocked users")
        
        return rewards_to_distribute
