"""Announcements Routes - اندپوینت‌های اطلاعیه"""

import uuid
from datetime import datetime, timezone

from fastapi import APIRouter, HTTPException, Depends

from models.announcement import AnnouncementCreate
from services.auth_service import get_current_user, get_admin_user
from database import get_db

router = APIRouter()


@router.get("/user/announcements/slider")
async def get_slider_announcements():
    """دریافت اطلاعیه‌های اسلایدر"""
    db = await get_db()
    
    announcements = await db.announcements.find(
        {"is_active": True, "is_modal": False},
        {"_id": 0}
    ).sort("created_at", -1).limit(10).to_list(10)
    
    return announcements


@router.get("/user/announcements/modal")
async def get_modal_announcement(current_user: dict = Depends(get_current_user)):
    """دریافت اطلاعیه مودال"""
    db = await get_db()
    
    announcement = await db.announcements.find_one(
        {
            "is_active": True,
            "is_modal": True,
            "seen_by": {"$ne": current_user["user_id"]}
        },
        {"_id": 0}
    )
    
    return announcement


@router.post("/user/announcements/{announcement_id}/mark-seen")
async def mark_announcement_seen(announcement_id: str, current_user: dict = Depends(get_current_user)):
    """علامت‌گذاری به عنوان دیده شده"""
    db = await get_db()
    
    await db.announcements.update_one(
        {"announcement_id": announcement_id},
        {"$addToSet": {"seen_by": current_user["user_id"]}}
    )
    
    return {"message": "علامت‌گذاری شد"}


# Admin Routes

@router.post("/admin/announcements")
async def create_announcement(data: AnnouncementCreate, admin_user: dict = Depends(get_admin_user)):
    """ایجاد اطلاعیه جدید"""
    db = await get_db()
    
    announcement = {
        "announcement_id": str(uuid.uuid4()),
        "title": data.title,
        "content": data.content,
        "link": data.link,
        "is_modal": data.is_modal,
        "is_active": data.is_active,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "created_by": admin_user["user_id"],
        "seen_by": []
    }
    
    await db.announcements.insert_one(announcement)
    
    return {"announcement_id": announcement["announcement_id"], "message": "اطلاعیه ایجاد شد"}


@router.get("/admin/announcements")
async def get_all_announcements(admin_user: dict = Depends(get_admin_user)):
    """لیست همه اطلاعیه‌ها"""
    db = await get_db()
    
    announcements = await db.announcements.find(
        {},
        {"_id": 0}
    ).sort("created_at", -1).to_list(100)
    
    return announcements


@router.delete("/admin/announcements/{announcement_id}")
async def delete_announcement(announcement_id: str, admin_user: dict = Depends(get_admin_user)):
    """حذف اطلاعیه"""
    db = await get_db()
    await db.announcements.delete_one({"announcement_id": announcement_id})
    return {"message": "اطلاعیه حذف شد"}


@router.put("/admin/announcements/{announcement_id}")
async def update_announcement(announcement_id: str, data: AnnouncementCreate, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی اطلاعیه"""
    db = await get_db()
    
    await db.announcements.update_one(
        {"announcement_id": announcement_id},
        {"$set": {
            "title": data.title,
            "content": data.content,
            "link": data.link,
            "is_modal": data.is_modal,
            "is_active": data.is_active
        }}
    )
    
    return {"message": "اطلاعیه بروزرسانی شد"}
