"""
Cache Manager برای بهینه‌سازی سیستم MLM
استفاده از Redis برای caching queries سنگین
"""

import os
import json
import logging
from typing import Optional, Any
from datetime import timedelta

try:
    import redis.asyncio as redis
    REDIS_AVAILABLE = True
except ImportError:
    REDIS_AVAILABLE = False

logger = logging.getLogger(__name__)

class CacheManager:
    """
    مدیریت Cache با پشتیبانی از Redis
    اگر Redis در دسترس نباشد، از in-memory cache استفاده می‌شود
    """
    
    def __init__(self):
        self.redis_client = None
        self.memory_cache = {}  # Fallback cache
        self.default_ttl = 300  # 5 minutes
        
    async def connect(self):
        """اتصال به Redis"""
        if not REDIS_AVAILABLE:
            logger.warning("⚠️ Redis package not available, using memory cache")
            return False
            
        redis_url = os.environ.get('REDIS_URL', 'redis://localhost:6379')
        
        try:
            self.redis_client = redis.from_url(
                redis_url,
                encoding="utf-8",
                decode_responses=True
            )
            # تست اتصال
            await self.redis_client.ping()
            logger.info("✅ Connected to Redis")
            return True
        except Exception as e:
            logger.warning(f"⚠️ Redis connection failed: {e}, using memory cache")
            self.redis_client = None
            return False
    
    async def get(self, key: str) -> Optional[Any]:
        """دریافت مقدار از cache"""
        try:
            if self.redis_client:
                value = await self.redis_client.get(key)
                if value:
                    return json.loads(value)
            else:
                # Memory cache
                if key in self.memory_cache:
                    data = self.memory_cache[key]
                    # Check expiry
                    import time
                    if data['expiry'] > time.time():
                        return data['value']
                    else:
                        del self.memory_cache[key]
        except Exception as e:
            logger.error(f"Cache get error: {e}")
        return None
    
    async def set(self, key: str, value: Any, ttl: int = None) -> bool:
        """ذخیره مقدار در cache"""
        ttl = ttl or self.default_ttl
        try:
            if self.redis_client:
                await self.redis_client.setex(
                    key,
                    ttl,
                    json.dumps(value)
                )
            else:
                # Memory cache
                import time
                self.memory_cache[key] = {
                    'value': value,
                    'expiry': time.time() + ttl
                }
            return True
        except Exception as e:
            logger.error(f"Cache set error: {e}")
            return False
    
    async def delete(self, key: str) -> bool:
        """حذف از cache"""
        try:
            if self.redis_client:
                await self.redis_client.delete(key)
            else:
                if key in self.memory_cache:
                    del self.memory_cache[key]
            return True
        except Exception as e:
            logger.error(f"Cache delete error: {e}")
            return False
    
    async def delete_pattern(self, pattern: str) -> int:
        """حذف کلیدهای مطابق با pattern"""
        try:
            if self.redis_client:
                keys = await self.redis_client.keys(pattern)
                if keys:
                    return await self.redis_client.delete(*keys)
            else:
                # Memory cache
                import fnmatch
                keys_to_delete = [k for k in self.memory_cache.keys() if fnmatch.fnmatch(k, pattern)]
                for k in keys_to_delete:
                    del self.memory_cache[k]
                return len(keys_to_delete)
        except Exception as e:
            logger.error(f"Cache delete pattern error: {e}")
        return 0
    
    async def invalidate_user_cache(self, user_id: str):
        """پاکسازی cache مربوط به یک کاربر"""
        await self.delete_pattern(f"user:{user_id}:*")
        await self.delete_pattern(f"level_counts:{user_id}")
        await self.delete_pattern(f"profile:{user_id}")
    
    async def invalidate_all_level_counts(self):
        """پاکسازی تمام cache های level_counts"""
        await self.delete_pattern("level_counts:*")
    
    async def close(self):
        """بستن اتصال"""
        if self.redis_client:
            await self.redis_client.close()


# Singleton instance
cache = CacheManager()


# Decorator for caching
def cached(key_prefix: str, ttl: int = 300):
    """Decorator برای cache کردن نتایج توابع async"""
    def decorator(func):
        async def wrapper(*args, **kwargs):
            # ساخت کلید cache
            cache_key = f"{key_prefix}:{':'.join(str(a) for a in args)}"
            
            # بررسی cache
            cached_value = await cache.get(cache_key)
            if cached_value is not None:
                return cached_value
            
            # اجرای تابع
            result = await func(*args, **kwargs)
            
            # ذخیره در cache
            await cache.set(cache_key, result, ttl)
            
            return result
        return wrapper
    return decorator
