#!/bin/bash

################################################################################
# MLM Materialized Path - اسکریپت نصب حرفه‌ای و کامل
# Compatible: Ubuntu 18+, Debian 10+, CentOS 7+, Rocky Linux
# Auto-installs: همه وابستگی‌ها + Docker + MongoDB + Redis
# Version: 3.0 Final
################################################################################

set -e
set -o pipefail

# رنگ‌ها
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m'

# تنظیمات
MONGO_PORT=${MONGO_PORT:-27017}
REDIS_PORT=${REDIS_PORT:-6379}
BACKEND_PORT=${BACKEND_PORT:-8001}
FRONTEND_PORT=${FRONTEND_PORT:-3000}
DB_NAME=${DB_NAME:-mlm_production}
INSTALL_DIR=$(pwd)
LOG_FILE="$INSTALL_DIR/install.log"

# تولید JWT secret امن
JWT_SECRET=$(openssl rand -hex 32 2>/dev/null || echo "mlm-secret-$(date +%s)-$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | fold -w 32 | head -n 1)")

# Detect running as root
if [ "$EUID" -eq 0 ]; then
    SUDO_CMD=""
else
    SUDO_CMD="sudo"
fi

# Logging
exec 1> >(tee -a "$LOG_FILE")
exec 2>&1

# Header
clear
cat << "EOF"
════════════════════════════════════════════════════════════════════════
   🚀 MLM Materialized Path - نصب خودکار حرفه‌ای
════════════════════════════════════════════════════════════════════════
EOF
echo ""
echo -e "${BLUE}📍 نسخه: 3.0 Final - Zero Traversal${NC}"
echo -e "${BLUE}📍 تاریخ: $(date '+%Y-%m-%d %H:%M:%S')${NC}"
echo ""

# تشخیص IP سرور
SERVER_IP=$(curl -s ifconfig.me 2>/dev/null || curl -s api.ipify.org 2>/dev/null || hostname -I | awk '{print $1}')
echo -e "${GREEN}✔ IP سرور: $SERVER_IP${NC}"
echo ""
sleep 1

# تشخیص OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS_NAME=$ID
    OS_PRETTY=$PRETTY_NAME
else
    echo -e "${RED}❌ نمی‌توانم سیستم‌عامل را تشخیص دهم${NC}"
    exit 1
fi

echo -e "${GREEN}✔ سیستم‌عامل: $OS_PRETTY${NC}"
echo -e "${GREEN}✔ دسترسی: $([ "$EUID" -eq 0 ] && echo 'Root' || echo 'Sudo User')${NC}"
echo ""

################################################################################
# توابع کمکی
################################################################################

log_step() {
    echo ""
    echo "════════════════════════════════════════════════════════════════════════"
    echo -e "${YELLOW}$1${NC}"
    echo "════════════════════════════════════════════════════════════════════════"
    echo ""
}

check_command() {
    command -v "$1" >/dev/null 2>&1
}

check_port() {
    local port=$1
    if $SUDO_CMD netstat -tuln 2>/dev/null | grep -q ":$port " || \
       $SUDO_CMD ss -tuln 2>/dev/null | grep -q ":$port " || \
       $SUDO_CMD lsof -Pi :$port -sTCP:LISTEN -t >/dev/null 2>&1; then
        return 1
    else
        return 0
    fi
}

################################################################################
# 1. پاکسازی نصب‌های قبلی
################################################################################

log_step "🧹 پاکسازی نصب‌های قبلی و پوشه‌های موقت..."

# حذف containers قدیمی
echo -e "${YELLOW}⏳ حذف containers قدیمی MLM...${NC}"
$SUDO_CMD docker rm -f mlm-mongodb mlm-redis mlm-backend mlm-frontend 2>/dev/null || true
echo -e "${GREEN}✔ Containers قدیمی پاک شد${NC}"

# حذف virtual environments قدیمی
if [ -d "backend/venv" ]; then
    echo -e "${YELLOW}⏳ حذف virtual environment قدیمی...${NC}"
    rm -rf backend/venv
    echo -e "${GREEN}✔ Virtual environment قدیمی پاک شد${NC}"
fi

# حذف node_modules قدیمی
if [ -d "frontend/node_modules" ]; then
    echo -e "${YELLOW}⏳ حذف node_modules قدیمی...${NC}"
    rm -rf frontend/node_modules
    echo -e "${GREEN}✔ node_modules قدیمی پاک شد${NC}"
fi

# حذف build قدیمی
if [ -d "frontend/build" ]; then
    rm -rf frontend/build
fi

################################################################################
# 2. نصب وابستگی‌ها
################################################################################

log_step "📦 بررسی و نصب وابستگی‌های سیستم..."

case $OS_NAME in
    ubuntu|debian)
        echo -e "${YELLOW}⏳ به‌روزرسانی package lists...${NC}"
        $SUDO_CMD apt-get update -qq
        
        echo -e "${YELLOW}⏳ نصب ابزارهای پایه...${NC}"
        $SUDO_CMD apt-get install -y \
            curl wget git \
            ca-certificates gnupg lsb-release \
            apt-transport-https software-properties-common \
            openssl net-tools \
            build-essential \
            python3 python3-venv python3-pip python3-dev \
            > /dev/null 2>&1
        ;;
        
    centos|rhel|rocky)
        echo -e "${YELLOW}⏳ نصب ابزارهای پایه...${NC}"
        $SUDO_CMD yum install -y \
            curl wget git \
            ca-certificates gnupg2 \
            openssl net-tools \
            gcc gcc-c++ make \
            python3 python3-pip python3-devel \
            yum-utils \
            > /dev/null 2>&1
        ;;
esac

echo -e "${GREEN}✔ وابستگی‌های پایه نصب شد${NC}"

################################################################################
# 3. نصب Docker
################################################################################

log_step "🐳 نصب و راه‌اندازی Docker..."

if check_command docker; then
    echo -e "${GREEN}✔ Docker موجود است: $(docker --version | cut -d ' ' -f 3 | cut -d ',' -f 1)${NC}"
else
    echo -e "${YELLOW}⏳ نصب Docker...${NC}"
    
    case $OS_NAME in
        ubuntu|debian)
            $SUDO_CMD mkdir -p /etc/apt/keyrings
            curl -fsSL https://download.docker.com/linux/$OS_NAME/gpg | $SUDO_CMD gpg --dearmor -o /etc/apt/keyrings/docker.gpg
            echo "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.gpg] https://download.docker.com/linux/$OS_NAME $(lsb_release -cs) stable" | $SUDO_CMD tee /etc/apt/sources.list.d/docker.list > /dev/null
            $SUDO_CMD apt-get update -qq
            $SUDO_CMD apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
            ;;
            
        centos|rhel|rocky)
            $SUDO_CMD yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo
            $SUDO_CMD yum install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
            ;;
    esac
    
    $SUDO_CMD systemctl start docker
    $SUDO_CMD systemctl enable docker
    echo -e "${GREEN}✔ Docker نصب و راه‌اندازی شد${NC}"
fi

# Add user to docker group
if [ "$EUID" -ne 0 ]; then
    $SUDO_CMD usermod -aG docker $USER 2>/dev/null || true
fi

################################################################################
# 4. نصب Node.js
################################################################################

log_step "📗 نصب Node.js..."

if check_command node; then
    NODE_VER=$(node --version | cut -d 'v' -f 2 | cut -d '.' -f 1)
    if [ "$NODE_VER" -ge 16 ]; then
        echo -e "${GREEN}✔ Node.js موجود است: $(node --version)${NC}"
    else
        INSTALL_NODE=true
    fi
else
    INSTALL_NODE=true
fi

if [ "$INSTALL_NODE" = true ]; then
    echo -e "${YELLOW}⏳ نصب Node.js 18...${NC}"
    
    case $OS_NAME in
        ubuntu|debian)
            curl -fsSL https://deb.nodesource.com/setup_18.x | $SUDO_CMD -E bash -
            $SUDO_CMD apt-get install -y nodejs
            ;;
        centos|rhel|rocky)
            curl -fsSL https://rpm.nodesource.com/setup_18.x | $SUDO_CMD bash -
            $SUDO_CMD yum install -y nodejs
            ;;
    esac
    
    echo -e "${GREEN}✔ Node.js نصب شد: $(node --version)${NC}"
fi

# نصب Yarn
if ! check_command yarn; then
    echo -e "${YELLOW}⏳ نصب Yarn...${NC}"
    $SUDO_CMD npm install -g yarn
fi
echo -e "${GREEN}✔ Yarn: $(yarn --version)${NC}"

################################################################################
# 5. راه‌اندازی MongoDB
################################################################################

log_step "🍃 راه‌اندازی MongoDB (Docker)..."

# حذف container قدیمی
$SUDO_CMD docker rm -f mlm-mongodb 2>/dev/null || true

# چک کردن port
if ! check_port $MONGO_PORT; then
    echo -e "${RED}❌ Port $MONGO_PORT در حال استفاده است${NC}"
    echo -e "${YELLOW}💡 MongoDB/Service دیگری در حال اجرا است${NC}"
    
    # تلاش برای stop کردن
    $SUDO_CMD systemctl stop mongodb 2>/dev/null || true
    $SUDO_CMD systemctl stop mongod 2>/dev/null || true
    sleep 2
    
    if ! check_port $MONGO_PORT; then
        echo -e "${RED}❌ نمی‌توانم port را آزاد کنم. لطفاً دستی متوقف کنید:${NC}"
        echo "   sudo systemctl stop mongodb"
        exit 1
    fi
fi

echo -e "${YELLOW}⏳ ایجاد MongoDB container...${NC}"

$SUDO_CMD docker run -d \
    --name mlm-mongodb \
    --restart unless-stopped \
    -p 127.0.0.1:$MONGO_PORT:27017 \
    -v mlm-mongodb-data:/data/db \
    -e MONGO_INITDB_DATABASE=$DB_NAME \
    mongo:7.0 \
    > /dev/null

echo -e "${YELLOW}⏳ در انتظار آماده شدن MongoDB...${NC}"
for i in {1..30}; do
    if $SUDO_CMD docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
        echo -e "${GREEN}✔ MongoDB آماده است${NC}"
        break
    fi
    sleep 1
done

if ! $SUDO_CMD docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
    echo -e "${RED}❌ MongoDB راه‌اندازی نشد${NC}"
    $SUDO_CMD docker logs mlm-mongodb
    exit 1
fi

################################################################################
# 6. راه‌اندازی Redis
################################################################################

log_step "🔴 راه‌اندازی Redis (Docker)..."

$SUDO_CMD docker rm -f mlm-redis 2>/dev/null || true

if check_port $REDIS_PORT; then
    echo -e "${YELLOW}⏳ ایجاد Redis container...${NC}"
    
    $SUDO_CMD docker run -d \
        --name mlm-redis \
        --restart unless-stopped \
        -p 127.0.0.1:$REDIS_PORT:6379 \
        -v mlm-redis-data:/data \
        redis:7-alpine \
        redis-server --appendonly yes \
        > /dev/null
    
    sleep 3
    
    if $SUDO_CMD docker exec mlm-redis redis-cli ping 2>/dev/null | grep -q "PONG"; then
        echo -e "${GREEN}✔ Redis راه‌اندازی شد${NC}"
        REDIS_ENABLED=true
    else
        echo -e "${YELLOW}⚠️  Redis اختیاری - ادامه بدون cache${NC}"
        REDIS_ENABLED=false
    fi
else
    echo -e "${YELLOW}⚠️  Port $REDIS_PORT در حال استفاده - Redis نصب نمی‌شود${NC}"
    REDIS_ENABLED=false
fi

################################################################################
# 7. نصب Backend
################################################################################

log_step "🔧 نصب Backend (FastAPI + Materialized Path)..."

cd "$INSTALL_DIR/backend"

# پاکسازی
rm -rf venv __pycache__ *.pyc 2>/dev/null || true

# ایجاد virtual environment
echo -e "${YELLOW}⏳ ایجاد Python virtual environment...${NC}"
python3 -m venv venv

if [ ! -f "venv/bin/activate" ]; then
    echo -e "${RED}❌ Virtual environment ایجاد نشد${NC}"
    exit 1
fi

# نصب dependencies
echo -e "${YELLOW}⏳ نصب Python packages (2-3 دقیقه)...${NC}"
source venv/bin/activate

pip install --upgrade pip setuptools wheel > /dev/null 2>&1

if ! pip install -r requirements.txt; then
    echo -e "${RED}❌ خطا در نصب Python packages${NC}"
    cat "$LOG_FILE" | tail -20
    exit 1
fi

deactivate
echo -e "${GREEN}✔ Backend dependencies نصب شد${NC}"

# ایجاد .env امن
echo -e "${YELLOW}⏳ ایجاد فایل .env امن...${NC}"

cat > .env << EOF
# Database Configuration
MONGO_URL=mongodb://localhost:$MONGO_PORT
DB_NAME=$DB_NAME

# Security (Generated)
JWT_SECRET=$JWT_SECRET

# Server Configuration
HOST=0.0.0.0
PORT=$BACKEND_PORT

# CORS - دسترسی از همه جا
CORS_ORIGINS=http://localhost:$FRONTEND_PORT,http://127.0.0.1:$FRONTEND_PORT,http://$SERVER_IP:$FRONTEND_PORT

# Performance
RATE_LIMIT_CALLS=1000
RATE_LIMIT_PERIOD=60

# Redis (Optional)
$([ "$REDIS_ENABLED" = true ] && echo "REDIS_URL=redis://localhost:$REDIS_PORT" || echo "#REDIS_URL=redis://localhost:$REDIS_PORT")
EOF

echo -e "${GREEN}✔ Backend .env ایجاد شد${NC}"

cd "$INSTALL_DIR"

################################################################################
# 8. نصب و Build Frontend
################################################################################

log_step "🎨 نصب و Build Frontend (React)..."

cd "$INSTALL_DIR/frontend"

# پاکسازی
rm -rf node_modules build .cache 2>/dev/null || true

# نصب dependencies
echo -e "${YELLOW}⏳ نصب Node packages (3-5 دقیقه)...${NC}"

if ! yarn install; then
    echo -e "${RED}❌ خطا در yarn install${NC}"
    exit 1
fi

echo -e "${GREEN}✔ Frontend dependencies نصب شد${NC}"

# ایجاد .env
cat > .env << EOF
REACT_APP_BACKEND_URL=http://$SERVER_IP:$BACKEND_PORT
REACT_APP_API_URL=http://$SERVER_IP:$BACKEND_PORT/api
EOF

echo -e "${GREEN}✔ Frontend .env ایجاد شد (Server IP: $SERVER_IP)${NC}"

# Build برای production (optional - می‌توان development mode هم اجرا کرد)
echo -e "${YELLOW}⏳ Build کردن Frontend (2-3 دقیقه)...${NC}"

if yarn build > /dev/null 2>&1; then
    echo -e "${GREEN}✔ Frontend build شد (production ready)${NC}"
    BUILD_CREATED=true
else
    echo -e "${YELLOW}⚠️  Build failed - می‌توانید در development mode اجرا کنید${NC}"
    BUILD_CREATED=false
fi

cd "$INSTALL_DIR"

################################################################################
# 9. بررسی نهایی
################################################################################

log_step "✅ بررسی نهایی سیستم..."

# چک MongoDB
echo -n "MongoDB: "
if $SUDO_CMD docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
    echo -e "${GREEN}✔ Running${NC}"
    MONGO_STATUS="✅ Running"
else
    echo -e "${RED}✘ Failed${NC}"
    MONGO_STATUS="❌ Failed"
    exit 1
fi

# چک Redis
echo -n "Redis: "
if [ "$REDIS_ENABLED" = true ] && $SUDO_CMD docker exec mlm-redis redis-cli ping 2>/dev/null | grep -q "PONG"; then
    echo -e "${GREEN}✔ Running${NC}"
    REDIS_STATUS="✅ Running"
else
    echo -e "${YELLOW}○ Not installed${NC}"
    REDIS_STATUS="○ Optional"
fi

# چک Python venv
echo -n "Python venv: "
if [ -f "backend/venv/bin/python" ]; then
    echo -e "${GREEN}✔ OK${NC}"
else
    echo -e "${RED}✘ Missing${NC}"
    exit 1
fi

# چک Node modules
echo -n "Node modules: "
if [ -d "frontend/node_modules" ]; then
    echo -e "${GREEN}✔ OK${NC}"
else
    echo -e "${RED}✘ Missing${NC}"
    exit 1
fi

################################################################################
# نمایش اطلاعات نهایی
################################################################################

clear
cat << "EOF"

════════════════════════════════════════════════════════════════════════
          ✅ نصب با موفقیت کامل شد! 🎉
════════════════════════════════════════════════════════════════════════

EOF

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                     ${BOLD}وضعیت سیستم${NC}                                   ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "   MongoDB:     $MONGO_STATUS"
echo -e "   Redis:       $REDIS_STATUS"
echo -e "   Backend:     ✅ Installed"
echo -e "   Frontend:    ✅ Installed $([ "$BUILD_CREATED" = true ] && echo '+ Built' || echo '')"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🔐 اطلاعات ورود${NC}                                  ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${BOLD}${GREEN}   Email:    admin@mlm.com${NC}"
echo -e "${BOLD}${GREEN}   Password: admin123${NC}"
echo ""
echo -e "${RED}   ⚠️  حتماً رمز را بعد از اولین ورود تغییر دهید!${NC}"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🚀 راه‌اندازی${NC}                                     ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""

echo -e "${YELLOW}【 Backend 】${NC}"
echo ""
echo -e "${CYAN}cd backend${NC}"
echo -e "${CYAN}source venv/bin/activate${NC}"
echo -e "${CYAN}python server.py${NC}"
echo ""
echo -e "   ✅ آماده در: ${BOLD}http://$SERVER_IP:$BACKEND_PORT${NC}"
echo -e "   ✅ Health: http://$SERVER_IP:$BACKEND_PORT/api/health"
echo ""

echo -e "${YELLOW}【 Frontend 】${NC} (terminal جدید)"
echo ""
echo -e "${CYAN}cd frontend${NC}"

if [ "$BUILD_CREATED" = true ]; then
    echo -e "${CYAN}npx serve -s build -l $FRONTEND_PORT --single${NC}"
    echo ""
    echo -e "   ✅ Production: ${BOLD}http://$SERVER_IP:$FRONTEND_PORT${NC}"
else
    echo -e "${CYAN}yarn start${NC}"
    echo ""
    echo -e "   ✅ Development: ${BOLD}http://$SERVER_IP:$FRONTEND_PORT${NC}"
fi

echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🧪 تست سیستم${NC}                                     ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""

echo -e "${YELLOW}بعد از راه‌اندازی Backend:${NC}"
echo ""
echo -e "${CYAN}curl http://localhost:$BACKEND_PORT/api/health${NC}"
echo ""
echo "   انتظار: {\"status\":\"ok\",\"mongodb\":true,\"seed_user\":true}"
echo ""

echo -e "${CYAN}╔══════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}                   ${BOLD}🛠️  مدیریت${NC}                                       ${CYAN}║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════════════════════════╝${NC}"
echo ""

echo -e "${YELLOW}Containers:${NC}"
echo -e "${CYAN}   docker ps                    ${NC}# مشاهده"
echo -e "${CYAN}   docker stop mlm-mongodb      ${NC}# توقف MongoDB"
echo -e "${CYAN}   docker start mlm-mongodb     ${NC}# شروع MongoDB"
echo -e "${CYAN}   docker logs mlm-mongodb -f   ${NC}# لاگ MongoDB"
echo ""

echo -e "${YELLOW}Firewall (اگر فعال است):${NC}"
echo -e "${CYAN}   sudo ufw allow $BACKEND_PORT${NC}"
echo -e "${CYAN}   sudo ufw allow $FRONTEND_PORT${NC}"
echo ""

echo -e "${YELLOW}Log نصب:${NC}"
echo -e "${CYAN}   cat $LOG_FILE${NC}"
echo ""

echo "════════════════════════════════════════════════════════════════════════"
echo -e "${BOLD}${GREEN}              🎉 سیستم آماده استفاده است! 🚀${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""
echo -e "${BLUE}📍 دسترسی از بیرون: http://$SERVER_IP:$FRONTEND_PORT${NC}"
echo -e "${BLUE}📍 API: http://$SERVER_IP:$BACKEND_PORT/api${NC}"
echo ""
