import { useState, useEffect } from "react";
import axios from "axios";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { toast } from "sonner";
import { LogOut, Users, TrendingUp, Settings, TestTube, UserCheck, UserX, Ban, CheckCircle, XCircle, Clock, Wallet, MessageSquare, FileText, Bell, Sliders, Lock } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import AdminCryptoManagement from "@/components/AdminCryptoManagement";
import MessagingSystem from "@/components/MessagingSystem";
import UserReport from "@/components/UserReport";
import TransferTest from "@/components/TransferTest";
import AdminSettings from "@/components/AdminSettings";
import AdminAnnouncements from "@/components/AdminAnnouncements";
import AdminConfig from "@/components/AdminConfig";
import AdminLoginTabs from "@/components/AdminLoginTabs";
import AdminTelegramSettings from "@/components/AdminTelegramSettingsSimple";

// کامپوننت جداگانه برای تنظیم امتیاز
function AdjustPointsDialog({ userId, currentPoints, onAdjust }) {
  const [amount, setAmount] = useState("");
  const [open, setOpen] = useState(false);

  const handleAdjust = () => {
    const value = parseFloat(amount);
    if (!isNaN(value) && value !== 0) {
      onAdjust(userId, value);
      setAmount("");
      setOpen(false);
    }
  };

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button size="sm" variant="outline" data-testid={`adjust-points-button-${userId}`}>
          <TrendingUp className="w-4 h-4" />
        </Button>
      </DialogTrigger>
      <DialogContent>
        <DialogHeader>
          <DialogTitle>تنظیم امتیاز</DialogTitle>
          <DialogDescription>
            امتیاز فعلی: {(currentPoints || 0).toFixed(0)}
          </DialogDescription>
        </DialogHeader>
        <div className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor={`points-input-${userId}`}>مقدار تغییر</Label>
            <Input
              id={`points-input-${userId}`}
              type="number"
              placeholder="مثال: -100 یا +50"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              onKeyDown={(e) => {
                if (e.key === 'Enter') {
                  handleAdjust();
                }
              }}
            />
            <p className="text-xs text-gray-500">
              برای کم کردن از علامت منفی استفاده کنید
            </p>
          </div>
          <Button
            onClick={handleAdjust}
            className="w-full"
            disabled={!amount || parseFloat(amount) === 0}
          >
            اعمال تغییرات
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

function AdminDashboard({ user, onLogout }) {
  const [stats, setStats] = useState(null);
  const [financialStats, setFinancialStats] = useState(null);
  const [users, setUsers] = useState([]);
  const [settings, setSettings] = useState(null);
  const [loading, setLoading] = useState(true);
  const [testCount, setTestCount] = useState(10);
  const [testReferrer, setTestReferrer] = useState("");
  const [fillUserId, setFillUserId] = useState("");
  const [fillLevels, setFillLevels] = useState(3);
  const [statusFilter, setStatusFilter] = useState("all"); // all, pending, active, tree_complete
  const [activeTab, setActiveTab] = useState("users");
  
  // Pagination & Search
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [searchQuery, setSearchQuery] = useState("");
  const [searchType, setSearchType] = useState("email"); // email, phone, referral_code
  const [totalUsers, setTotalUsers] = useState(0);
  
  // برای گزارش کاربر
  const [selectedUserId, setSelectedUserId] = useState(null);
  const [selectedUserEmail, setSelectedUserEmail] = useState(null);

  useEffect(() => {
    fetchData(currentPage);
  }, [currentPage, searchQuery, searchType, statusFilter]);

  const fetchData = async (page = 1) => {
    try {
      // ساخت query parameters
      const params = new URLSearchParams({
        page: page.toString(),
        limit: '20'
      });
      
      if (searchQuery) {
        params.append('search', searchQuery);
        params.append('search_type', searchType);
      }
      
      if (statusFilter !== 'all') {
        params.append('status_filter', statusFilter);
      }
      
      const [statsRes, usersRes, settingsRes, financialRes] = await Promise.all([
        axios.get(`${API}/admin/stats`),
        axios.get(`${API}/admin/users?${params.toString()}`),
        axios.get(`${API}/admin/settings`),
        axios.get(`${API}/admin/financial-stats`),
      ]);

      setStats(statsRes.data);
      setUsers(usersRes.data.users);
      setTotalUsers(usersRes.data.total);
      setCurrentPage(usersRes.data.page);
      setTotalPages(usersRes.data.total_pages);
      setSettings(settingsRes.data);
      setFinancialStats(financialRes.data);
    } catch (error) {
      toast.error("خطا در دریافت اطلاعات");
    } finally {
      setLoading(false);
    }
  };

  const activateUser = async (userId) => {
    try {
      await axios.post(`${API}/admin/users/${userId}/activate`);
      toast.success("کاربر فعال شد");
      fetchData(currentPage);
    } catch (error) {
      toast.error(error.response?.data?.detail || "خطا در فعال‌سازی");
    }
  };
  
  const handleSearch = () => {
    setCurrentPage(1);
    fetchData(1);
  };
  
  const handlePageChange = (newPage) => {
    if (newPage >= 1 && newPage <= totalPages) {
      setCurrentPage(newPage);
    }
  };

  const deactivateUser = async (userId) => {
    try {
      await axios.post(`${API}/admin/users/${userId}/deactivate`);
      toast.success("کاربر غیرفعال شد");
      fetchData();
    } catch (error) {
      toast.error("خطا در غیرفعال‌سازی");
    }
  };

  const blockRewards = async (userId, blockDirect, blockLevel) => {
    try {
      await axios.post(
        `${API}/admin/users/${userId}/block-rewards?block_direct=${blockDirect}&block_level=${blockLevel}`
      );
      toast.success("پاداش‌ها مسدود شدند");
      fetchData();
    } catch (error) {
      toast.error("خطا در مسدودسازی");
    }
  };

  const unblockRewards = async (userId) => {
    try {
      await axios.post(`${API}/admin/users/${userId}/unblock-rewards`);
      toast.success("پاداش‌ها رفع مسدودیت شدند");
      fetchData();
    } catch (error) {
      toast.error("خطا در رفع مسدودیت");
    }
  };

  const adjustPoints = async (userId, amount) => {
    try {
      const response = await axios.post(`${API}/admin/users/${userId}/adjust-points`, {
        amount: amount,
        reason: amount > 0 ? "افزایش دستی" : "کاهش دستی"
      });
      toast.success(response.data.message);
      fetchData();
    } catch (error) {
      toast.error("خطا در تنظیم امتیاز");
    }
  };

  const updateSettings = async (e) => {
    e.preventDefault();
    try {
      await axios.put(`${API}/admin/settings`, settings);
      toast.success("تنظیمات به‌روزرسانی شد");
    } catch (error) {
      toast.error("خطا در به‌روزرسانی تنظیمات");
    }
  };

  const createTestUsers = async () => {
    try {
      const response = await axios.post(`${API}/admin/test/create-users`, {
        count: testCount,
        auto_activate: true,
        referral_code: testReferrer || null,
      });
      toast.success(response.data.message);
      fetchData();
    } catch (error) {
      toast.error(error.response?.data?.detail || "خطا در ساخت کاربران تست");
    }
  };

  const clearTestUsers = async () => {
    try {
      const response = await axios.delete(`${API}/admin/test/clear-test-users`);
      toast.success(response.data.message);
      fetchData();
    } catch (error) {
      toast.error("خطا در حذف کاربران تست");
    }
  };

  const clearAllUsers = async () => {
    try {
      const response = await axios.delete(`${API}/admin/users/clear-all`);
      toast.success(response.data.message);
      fetchData();
    } catch (error) {
      toast.error("خطا در حذف کاربران");
    }
  };

  const fillTree = async () => {
    if (!fillUserId) {
      toast.error("لطفاً کد معرف را وارد کنید");
      return;
    }
    try {
      const response = await axios.post(`${API}/admin/test/fill-tree?referral_code=${fillUserId}&levels=${fillLevels}`);
      toast.success(response.data.message);
      fetchData();
    } catch (error) {
      toast.error(error.response?.data?.detail || "خطا در پر کردن درخت");
    }
  };

  const getStatusBadge = (status, isTreeComplete) => {
    const statusMap = {
      pending: { label: "در انتظار", variant: "outline", icon: Clock },
      active: { label: "فعال", variant: "default", icon: CheckCircle },
    };
    const statusInfo = statusMap[status] || statusMap.pending;
    const Icon = statusInfo.icon;
    
    return (
      <div className="flex items-center gap-2">
        <Badge variant={statusInfo.variant} className="flex items-center gap-1">
          <Icon className="w-3 h-3" />
          {statusInfo.label}
        </Badge>
        {isTreeComplete && (
          <Badge variant="secondary" className="flex items-center gap-1">
            <CheckCircle className="w-3 h-3" />
            هرم کامل
          </Badge>
        )}
      </div>
    );
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 to-teal-100">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-emerald-600 mx-auto"></div>
          <p className="mt-4 text-emerald-700 font-vazir">در حال بارگذاری...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50" data-testid="admin-dashboard">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <div className="flex items-center gap-3">
            <div className="w-12 h-12 bg-gradient-to-br from-emerald-500 to-teal-600 rounded-xl flex items-center justify-center shadow-md">
              <Settings className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-emerald-900">پنل مدیریت</h1>
              <p className="text-sm text-gray-600">{user.email}</p>
            </div>
          </div>
          <Button
            onClick={onLogout}
            variant="outline"
            className="flex items-center gap-2 border-red-200 text-red-600 hover:bg-red-50"
            data-testid="logout-button"
          >
            <LogOut className="w-4 h-4" />
            خروج
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-4 py-8">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <Card 
            className="bg-white shadow-lg border-0 cursor-pointer hover:shadow-xl transition-shadow" 
            data-testid="total-users-card"
            onClick={() => {
              setStatusFilter("all");
              setActiveTab("users");
            }}
          >
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <Users className="w-4 h-4" />
                کل کاربران
              </CardDescription>
              <CardTitle className="text-3xl text-emerald-600">
                {stats?.total_users}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card 
            className="bg-white shadow-lg border-0 cursor-pointer hover:shadow-xl transition-shadow" 
            data-testid="active-users-card"
            onClick={() => {
              setStatusFilter("active");
              setActiveTab("users");
            }}
          >
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <UserCheck className="w-4 h-4" />
                کاربران فعال
              </CardDescription>
              <CardTitle className="text-3xl text-teal-600">
                {stats?.active_users}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card 
            className="bg-white shadow-lg border-0 cursor-pointer hover:shadow-xl transition-shadow border-2 border-yellow-200" 
            data-testid="pending-users-card"
            onClick={() => {
              setStatusFilter("pending");
              setActiveTab("users");
            }}
          >
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <Clock className="w-4 h-4" />
                در انتظار (کلیک کنید)
              </CardDescription>
              <CardTitle className="text-3xl text-yellow-600">
                {stats?.pending_users}
              </CardTitle>
            </CardHeader>
          </Card>

          <Card 
            className="bg-white shadow-lg border-0 cursor-pointer hover:shadow-xl transition-shadow" 
            data-testid="tree-complete-users-card"
            onClick={() => {
              setStatusFilter("tree_complete");
              setActiveTab("users");
            }}
          >
            <CardHeader className="pb-3">
              <CardDescription className="text-gray-600 flex items-center gap-2">
                <CheckCircle className="w-4 h-4" />
                هرم کامل
              </CardDescription>
              <CardTitle className="text-3xl text-purple-600">
                {stats?.tree_complete_users || 0}
              </CardTitle>
            </CardHeader>
          </Card>
        </div>

        {/* Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="flex flex-wrap gap-2 bg-white shadow-md rounded-xl p-3 h-auto">
            <TabsTrigger value="users" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-blue-50 data-[state=active]:bg-blue-600 data-[state=active]:text-white" data-testid="users-tab">
              <Users className="w-4 h-4 ml-2" />
              کاربران
            </TabsTrigger>
            <TabsTrigger value="crypto" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-emerald-50 data-[state=active]:bg-emerald-600 data-[state=active]:text-white" data-testid="crypto-tab">
              <Wallet className="w-4 h-4 ml-2" />
              کریپتو
            </TabsTrigger>
            <TabsTrigger value="transfer-test" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 bg-purple-50 hover:bg-purple-100 data-[state=active]:bg-purple-600 data-[state=active]:text-white">
              <TestTube className="w-4 h-4 ml-2" />
              تست انتقال
            </TabsTrigger>
            <TabsTrigger value="messages" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-pink-50 data-[state=active]:bg-pink-600 data-[state=active]:text-white">
              <MessageSquare className="w-4 h-4 ml-2" />
              پیام‌ها
            </TabsTrigger>
            <TabsTrigger value="announcements" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-yellow-50 data-[state=active]:bg-yellow-600 data-[state=active]:text-white">
              <Bell className="w-4 h-4 ml-2" />
              اطلاعیه‌ها
            </TabsTrigger>
            <TabsTrigger value="settings" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-amber-50 data-[state=active]:bg-amber-600 data-[state=active]:text-white" data-testid="settings-tab">
              <Settings className="w-4 h-4 ml-2" />
              پاداش‌ها
            </TabsTrigger>
            <TabsTrigger value="config" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-indigo-50 data-[state=active]:bg-indigo-600 data-[state=active]:text-white">
              <Sliders className="w-4 h-4 ml-2" />
              تنظیمات سایت
            </TabsTrigger>
            <TabsTrigger value="telegram" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-cyan-50 data-[state=active]:bg-cyan-600 data-[state=active]:text-white">
              <MessageSquare className="w-4 h-4 ml-2" />
              ربات تلگرام
            </TabsTrigger>
            <TabsTrigger value="test" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-teal-50 data-[state=active]:bg-teal-600 data-[state=active]:text-white" data-testid="test-tab">
              <TestTube className="w-4 h-4 ml-2" />
              بررسی سیستم
            </TabsTrigger>
            <TabsTrigger value="reports" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-orange-50 data-[state=active]:bg-orange-600 data-[state=active]:text-white">
              <FileText className="w-4 h-4 ml-2" />
              گزارش‌ها
            </TabsTrigger>
            <TabsTrigger value="financial" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-green-50 data-[state=active]:bg-green-600 data-[state=active]:text-white" data-testid="financial-tab">
              <TrendingUp className="w-4 h-4 ml-2" />
              مالی
            </TabsTrigger>
            <TabsTrigger value="admin-settings" className="flex-1 min-w-[130px] rounded-lg font-semibold border-2 px-4 py-3 hover:bg-red-50 data-[state=active]:bg-red-600 data-[state=active]:text-white">
              <Lock className="w-4 h-4 ml-2" />
              حساب من
            </TabsTrigger>
          </TabsList>

          <TabsContent value="users" data-testid="users-content">
            <Card className="shadow-lg border-0">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <Users className="w-5 h-5 text-emerald-600" />
                    لیست کاربران
                    {statusFilter !== "all" && (
                      <Badge variant="secondary" className="mr-2">
                        {statusFilter === "pending" && "فقط در انتظار"}
                        {statusFilter === "active" && "فقط فعال"}
                        {statusFilter === "tree_complete" && "فقط هرم کامل"}
                      </Badge>
                    )}
                  </CardTitle>
                  {statusFilter !== "all" && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setStatusFilter("all")}
                      className="text-xs"
                    >
                      نمایش همه
                    </Button>
                  )}
                </div>
              </CardHeader>
              <CardContent>
                {/* Search Box */}
                <div className="mb-4 flex gap-2">
                  <select
                    value={searchType}
                    onChange={(e) => setSearchType(e.target.value)}
                    className="border rounded px-3 py-2"
                  >
                    <option value="email">ایمیل</option>
                    <option value="phone">شماره تماس</option>
                    <option value="referral_code">کد دعوت</option>
                  </select>
                  <input
                    type="text"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    placeholder="جستجو..."
                    className="border rounded px-3 py-2 flex-1"
                    onKeyPress={(e) => {
                      if (e.key === 'Enter') handleSearch();
                    }}
                  />
                  <Button onClick={handleSearch}>جستجو</Button>
                  {searchQuery && (
                    <Button variant="outline" onClick={() => {
                      setSearchQuery("");
                      setCurrentPage(1);
                    }}>پاک کردن</Button>
                  )}
                </div>
                
                {/* نمایش تعداد کل */}
                <div className="mb-2 text-sm text-gray-600">
                  {totalUsers} کاربر یافت شد
                </div>
                
                <div className="space-y-3 max-h-[600px] overflow-y-auto">
                  {users.map((u) => (
                    <div
                      key={u.user_id}
                      className="flex items-center justify-between p-4 bg-gradient-to-r from-emerald-50 to-teal-50 rounded-lg"
                      data-testid="user-item"
                    >
                      <div className="flex-1">
                        <p className="font-semibold text-gray-800">{u.email}</p>
                        <div className="flex items-center gap-2">
                          <p className="text-sm text-gray-600">کد معرف: {u.referral_code}</p>
                          <button
                            onClick={() => {
                              // استفاده از روش سازگار با همه مرورگرها
                              try {
                                if (navigator.clipboard && navigator.clipboard.writeText) {
                                  navigator.clipboard.writeText(u.referral_code).then(() => {
                                    toast.success("کد معرف کپی شد");
                                  }).catch(() => {
                                    // Fallback به روش قدیمی
                                    const textarea = document.createElement('textarea');
                                    textarea.value = u.referral_code;
                                    textarea.style.position = 'fixed';
                                    textarea.style.opacity = '0';
                                    document.body.appendChild(textarea);
                                    textarea.select();
                                    document.execCommand('copy');
                                    document.body.removeChild(textarea);
                                    toast.success("کد معرف کپی شد");
                                  });
                                } else {
                                  // Fallback به روش قدیمی
                                  const textarea = document.createElement('textarea');
                                  textarea.value = u.referral_code;
                                  textarea.style.position = 'fixed';
                                  textarea.style.opacity = '0';
                                  document.body.appendChild(textarea);
                                  textarea.select();
                                  document.execCommand('copy');
                                  document.body.removeChild(textarea);
                                  toast.success("کد معرف کپی شد");
                                }
                              } catch (err) {
                                toast.error("خطا در کپی - لطفاً دستی کپی کنید");
                              }
                            }}
                            className="text-xs text-emerald-600 hover:text-emerald-700 cursor-pointer"
                          >
                            📋
                          </button>
                        </div>
                        <p className="text-sm text-gray-600">عمق: {u.depth} | فرزندان: {u.direct_children?.length || 0}</p>
                        <p className="text-sm text-emerald-600 font-semibold">امتیاز: {(u.total_points || 0).toFixed(0)}</p>
                      </div>
                      <div className="flex items-center gap-2">
                        {getStatusBadge(u.status, u.is_tree_complete)}
                        
                        {/* دکمه‌های فعال/غیرفعال و مسدود - فقط برای کاربران عادی */}
                        {u.role !== "admin" && (
                          <>
                            {u.status === "pending" ? (
                              <Button
                                size="sm"
                                onClick={() => activateUser(u.user_id)}
                                className="bg-emerald-600 hover:bg-emerald-700"
                                data-testid={`activate-button-${u.user_id}`}
                              >
                                <UserCheck className="w-4 h-4" />
                              </Button>
                            ) : (
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => deactivateUser(u.user_id)}
                                data-testid={`deactivate-button-${u.user_id}`}
                              >
                                <UserX className="w-4 h-4" />
                              </Button>
                            )}
                            <Dialog>
                              <DialogTrigger asChild>
                                <Button size="sm" variant="destructive" data-testid={`block-button-${u.user_id}`}>
                                  <Ban className="w-4 h-4" />
                                </Button>
                              </DialogTrigger>
                              <DialogContent>
                                <DialogHeader>
                                  <DialogTitle>مسدود کردن پاداش‌ها</DialogTitle>
                                  <DialogDescription>
                                    کدام پاداش‌ها را می‌خواهید مسدود کنید؟
                                  </DialogDescription>
                                </DialogHeader>
                                <div className="space-y-3">
                                  <Button
                                    onClick={() => blockRewards(u.user_id, true, false)}
                                    className="w-full"
                                  >
                                    مسدود کردن پاداش مستقیم
                                  </Button>
                                  <Button
                                    onClick={() => blockRewards(u.user_id, false, true)}
                                    className="w-full"
                                  >
                                    مسدود کردن پاداش سطح
                                  </Button>
                                  <Button
                                    onClick={() => blockRewards(u.user_id, true, true)}
                                    className="w-full"
                                    variant="destructive"
                                  >
                                    مسدود کردن همه پاداش‌ها
                                  </Button>
                                  <Button
                                    onClick={() => unblockRewards(u.user_id)}
                                    className="w-full"
                                    variant="outline"
                                  >
                                    رفع مسدودیت همه
                                  </Button>
                                </div>
                              </DialogContent>
                            </Dialog>
                          </>
                        )}
                        
                        {/* دکمه تنظیم امتیاز - برای همه کاربران از جمله seed */}
                        <AdjustPointsDialog userId={u.user_id} currentPoints={u.total_points} onAdjust={adjustPoints} />
                        
                        {/* دکمه گزارش */}
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            setSelectedUserId(u.user_id);
                            setSelectedUserEmail(u.email);
                            setActiveTab('reports');
                          }}
                          className="bg-blue-50 hover:bg-blue-100"
                        >
                          <FileText className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  ))}
                  {users.length === 0 && (
                    <p className="text-center text-gray-500 py-8">
                      هیچ کاربری یافت نشد
                    </p>
                  )}
                </div>
                
                {/* Pagination Controls */}
                {totalPages > 1 && (
                  <div className="mt-4 flex items-center justify-between border-t pt-4">
                    <div className="text-sm text-gray-600">
                      صفحه {currentPage} از {totalPages}
                    </div>
                    <div className="flex gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handlePageChange(1)}
                        disabled={currentPage === 1}
                      >
                        اول
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handlePageChange(currentPage - 1)}
                        disabled={currentPage === 1}
                      >
                        قبلی
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handlePageChange(currentPage + 1)}
                        disabled={currentPage === totalPages}
                      >
                        بعدی
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handlePageChange(totalPages)}
                        disabled={currentPage === totalPages}
                      >
                        آخر
                      </Button>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="reports" data-testid="reports-content">
            {selectedUserId && selectedUserEmail ? (
              <UserReport 
                userId={selectedUserId} 
                userEmail={selectedUserEmail}
                onClose={() => {
                  setSelectedUserId(null);
                  setSelectedUserEmail(null);
                }}
              />
            ) : (
              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <FileText className="w-5 h-5 text-emerald-600" />
                    گزارش کاربران
                  </CardTitle>
                  <CardDescription>
                    برای مشاهده گزارش، روی دکمه گزارش کنار هر کاربر کلیک کنید
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-center py-12">
                    <FileText className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                    <p className="text-gray-500">هیچ کاربری انتخاب نشده است</p>
                    <p className="text-sm text-gray-400 mt-2">
                      از تب کاربران، روی دکمه آبی گزارش کلیک کنید
                    </p>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="financial" data-testid="financial-content">
            <div className="space-y-6">
              {/* کارت‌های اصلی مالی */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <Card className="bg-gradient-to-br from-green-50 to-emerald-100 shadow-lg border-0">
                  <CardHeader className="pb-3">
                    <CardDescription className="text-gray-700 font-semibold">💰 کل دریافتی</CardDescription>
                    <CardTitle className="text-3xl text-green-700">
                      ${financialStats?.total_revenue?.toLocaleString() || 0}
                    </CardTitle>
                    <p className="text-xs text-gray-600 mt-1">
                      {financialStats?.total_payments_count || 0} پرداخت
                    </p>
                  </CardHeader>
                </Card>

                <Card className="bg-gradient-to-br from-red-50 to-rose-100 shadow-lg border-0">
                  <CardHeader className="pb-3">
                    <CardDescription className="text-gray-700 font-semibold">💸 کل پرداختی</CardDescription>
                    <CardTitle className="text-3xl text-red-700">
                      ${financialStats?.total_paid?.toLocaleString() || 0}
                    </CardTitle>
                    <p className="text-xs text-gray-600 mt-1">
                      {financialStats?.unique_users_with_rewards || 0} کاربر جایزه گرفته
                    </p>
                  </CardHeader>
                </Card>

                <Card className="bg-gradient-to-br from-blue-50 to-cyan-100 shadow-lg border-0">
                  <CardHeader className="pb-3">
                    <CardDescription className="text-gray-700 font-semibold">✨ سود خالص</CardDescription>
                    <CardTitle className="text-3xl text-blue-700">
                      ${financialStats?.profit?.toLocaleString() || 0}
                    </CardTitle>
                  </CardHeader>
                </Card>

                <Card className="bg-gradient-to-br from-purple-50 to-violet-100 shadow-lg border-0">
                  <CardHeader className="pb-3">
                    <CardDescription className="text-gray-700 font-semibold">📊 درصد سود</CardDescription>
                    <CardTitle className="text-3xl text-purple-700">
                      {financialStats?.profit_percentage?.toFixed(2) || 0}%
                    </CardTitle>
                  </CardHeader>
                </Card>
              </div>

              {/* جدول تفکیک جوایز */}
              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5 text-emerald-600" />
                    تفکیک جوایز پرداختی
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="overflow-x-auto">
                    <table className="w-full">
                      <thead className="bg-emerald-50">
                        <tr>
                          <th className="px-4 py-3 text-right text-sm font-semibold text-gray-700">نوع جایزه</th>
                          <th className="px-4 py-3 text-right text-sm font-semibold text-gray-700">مبلغ کل</th>
                          <th className="px-4 py-3 text-right text-sm font-semibold text-gray-700">تعداد</th>
                          <th className="px-4 py-3 text-right text-sm font-semibold text-gray-700">درصد از کل</th>
                        </tr>
                      </thead>
                      <tbody className="divide-y divide-gray-200">
                        <tr className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-900">🎁 جوایز مستقیم</td>
                          <td className="px-4 py-3 text-sm font-semibold text-green-600">
                            ${financialStats?.rewards_breakdown?.direct?.total?.toLocaleString() || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.rewards_breakdown?.direct?.count || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.total_paid > 0 
                              ? ((financialStats?.rewards_breakdown?.direct?.total / financialStats?.total_paid) * 100).toFixed(1)
                              : 0}%
                          </td>
                        </tr>
                        <tr className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-900">📍 جوایز سطح ۳</td>
                          <td className="px-4 py-3 text-sm font-semibold text-teal-600">
                            ${financialStats?.rewards_breakdown?.level_3?.total?.toLocaleString() || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.rewards_breakdown?.level_3?.count || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.total_paid > 0 
                              ? ((financialStats?.rewards_breakdown?.level_3?.total / financialStats?.total_paid) * 100).toFixed(1)
                              : 0}%
                          </td>
                        </tr>
                        <tr className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-900">📍 جوایز سطح ۵</td>
                          <td className="px-4 py-3 text-sm font-semibold text-blue-600">
                            ${financialStats?.rewards_breakdown?.level_5?.total?.toLocaleString() || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.rewards_breakdown?.level_5?.count || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.total_paid > 0 
                              ? ((financialStats?.rewards_breakdown?.level_5?.total / financialStats?.total_paid) * 100).toFixed(1)
                              : 0}%
                          </td>
                        </tr>
                        <tr className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-900">📍 جوایز سطح ۷</td>
                          <td className="px-4 py-3 text-sm font-semibold text-purple-600">
                            ${financialStats?.rewards_breakdown?.level_7?.total?.toLocaleString() || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.rewards_breakdown?.level_7?.count || 0}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {financialStats?.total_paid > 0 
                              ? ((financialStats?.rewards_breakdown?.level_7?.total / financialStats?.total_paid) * 100).toFixed(1)
                              : 0}%
                          </td>
                        </tr>
                      </tbody>
                    </table>
                  </div>
                </CardContent>
              </Card>

              {/* اطلاعات کاربران جایزه‌گرفته */}
              <Card className="shadow-lg border-0 bg-gradient-to-br from-amber-50 to-orange-100">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-amber-900">
                    <Users className="w-5 h-5" />
                    آمار کاربران جایزه‌گرفته
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="text-center">
                    <div className="text-5xl font-bold text-amber-700 mb-2">
                      {financialStats?.unique_users_with_rewards || 0}
                    </div>
                    <p className="text-amber-800 font-semibold">کاربر تاکنون جایزه دریافت کرده‌اند</p>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="messages" data-testid="messages-content">
            <MessagingSystem userType="admin" />
          </TabsContent>

          <TabsContent value="transfer-test">
            <TransferTest />
          </TabsContent>

          <TabsContent value="announcements">
            <AdminAnnouncements />
          </TabsContent>

          <TabsContent value="config">
            <div className="space-y-6">
              <AdminConfig />
              <AdminLoginTabs />
            </div>
          </TabsContent>

          <TabsContent value="telegram">
            <AdminTelegramSettings />
          </TabsContent>

          <TabsContent value="admin-settings">
            <AdminSettings />
          </TabsContent>

          <TabsContent value="crypto" data-testid="crypto-content">
            <AdminCryptoManagement />
          </TabsContent>

          <TabsContent value="settings" data-testid="settings-content">
            <Card className="shadow-lg border-0">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="w-5 h-5 text-emerald-600" />
                  تنظیمات پاداش‌ها
                </CardTitle>
                <CardDescription>مقادیر پاداش‌ها را تنظیم کنید</CardDescription>
              </CardHeader>
              <CardContent>
                {settings && (
                  <form onSubmit={updateSettings} className="space-y-6">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-2 md:col-span-2">
                        <Label htmlFor="activation_fee" className="font-semibold text-lg text-emerald-700">💰 هزینه ورود به سیستم (دلار)</Label>
                        <Input
                          id="activation_fee"
                          type="number"
                          step="0.01"
                          value={settings.activation_fee || 100}
                          onChange={(e) => setSettings({ ...settings, activation_fee: parseFloat(e.target.value) })}
                          className="h-11 text-lg font-bold"
                          data-testid="activation-fee-input"
                        />
                        <p className="text-xs text-gray-500">مبلغی که هر کاربر برای فعال‌سازی باید پرداخت کند</p>
                      </div>
                      <div className="space-y-2 md:col-span-2">
                        <Label htmlFor="seed_initial_payment" className="font-semibold text-lg text-blue-700">💎 سرمایه اولیه سیستم (seed)</Label>
                        <Input
                          id="seed_initial_payment"
                          type="number"
                          step="0.01"
                          value={settings.seed_initial_payment || 100}
                          onChange={(e) => setSettings({ ...settings, seed_initial_payment: parseFloat(e.target.value) })}
                          className="h-11 text-lg font-bold"
                          data-testid="seed-initial-payment-input"
                        />
                        <p className="text-xs text-gray-500">این مبلغ به عنوان سرمایه اولیه در کل دریافتی‌ها محاسبه می‌شود</p>
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="direct_first" className="font-semibold">پاداش مستقیم نفر اول</Label>
                        <Input
                          id="direct_first"
                          type="number"
                          step="0.01"
                          value={settings.direct_first}
                          onChange={(e) => setSettings({ ...settings, direct_first: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="direct-first-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="direct_second" className="font-semibold">پاداش مستقیم نفر دوم</Label>
                        <Input
                          id="direct_second"
                          type="number"
                          step="0.01"
                          value={settings.direct_second}
                          onChange={(e) => setSettings({ ...settings, direct_second: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="direct-second-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="direct_third" className="font-semibold">پاداش مستقیم نفر سوم</Label>
                        <Input
                          id="direct_third"
                          type="number"
                          step="0.01"
                          value={settings.direct_third}
                          onChange={(e) => setSettings({ ...settings, direct_third: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="direct-third-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="direct_extra" className="font-semibold">پاداش مستقیم اضافی</Label>
                        <Input
                          id="direct_extra"
                          type="number"
                          step="0.01"
                          value={settings.direct_extra}
                          onChange={(e) => setSettings({ ...settings, direct_extra: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="direct-extra-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="level_3" className="font-semibold">پاداش سطح ۳</Label>
                        <Input
                          id="level_3"
                          type="number"
                          step="0.01"
                          value={settings.level_3}
                          onChange={(e) => setSettings({ ...settings, level_3: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="level-3-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="level_5" className="font-semibold">پاداش سطح ۵</Label>
                        <Input
                          id="level_5"
                          type="number"
                          step="0.01"
                          value={settings.level_5}
                          onChange={(e) => setSettings({ ...settings, level_5: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="level-5-input"
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="level_7" className="font-semibold">پاداش سطح ۷</Label>
                        <Input
                          id="level_7"
                          type="number"
                          step="0.01"
                          value={settings.level_7}
                          onChange={(e) => setSettings({ ...settings, level_7: parseFloat(e.target.value) })}
                          className="h-11"
                          data-testid="level-7-input"
                        />
                      </div>
                    </div>
                    <Button
                      type="submit"
                      className="w-full bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 h-12"
                      data-testid="save-settings-button"
                    >
                      ذخیره تنظیمات
                    </Button>
                  </form>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="test" data-testid="test-content">
            <div className="space-y-6">
              {/* Delete All Users - Warning Card */}
              <Card className="shadow-lg border-2 border-red-300 bg-red-50">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-red-700">
                    <XCircle className="w-5 h-5" />
                    حذف تمام کاربران
                  </CardTitle>
                  <CardDescription className="text-red-600">
                    ⚠️ خطرناک! این عملیات تمام کاربران (به جز seed و admin) را حذف می‌کند
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button
                        variant="destructive"
                        className="w-full h-12 text-base font-bold"
                        data-testid="clear-all-users-button"
                      >
                        <XCircle className="w-5 h-5 ml-2" />
                        حذف تمام کاربران و ریست سیستم
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle className="text-red-600 text-xl">
                          ⚠️ هشدار خطرناک!
                        </DialogTitle>
                        <DialogDescription className="text-base pt-4">
                          <div className="space-y-4">
                            <p className="text-red-600 font-bold">
                              آیا مطمئن هستید که می‌خواهید تمام کاربران را حذف کنید؟
                            </p>
                            <div className="bg-red-50 p-4 rounded-lg border-2 border-red-200">
                              <p className="text-gray-700 mb-2">این عملیات:</p>
                              <ul className="list-disc list-inside space-y-1 text-gray-600 text-sm">
                                <li>تمام کاربران (به جز seed و admin) را حذف می‌کند</li>
                                <li>تمام تراکنش‌ها پاک می‌شوند</li>
                                <li>سیستم کاملاً ریست می‌شود</li>
                                <li>این عملیات قابل بازگشت نیست!</li>
                              </ul>
                            </div>
                          </div>
                        </DialogDescription>
                      </DialogHeader>
                      <div className="flex gap-3 pt-4">
                        <DialogTrigger asChild>
                          <Button variant="outline" className="flex-1">
                            انصراف
                          </Button>
                        </DialogTrigger>
                        <Button
                          variant="destructive"
                          className="flex-1 font-bold"
                          onClick={() => {
                            clearAllUsers();
                            // Close dialog after action
                            document.querySelector('[data-state="open"]')?.click();
                          }}
                        >
                          بله، تمام کاربران را حذف کن
                        </Button>
                      </div>
                    </DialogContent>
                  </Dialog>
                </CardContent>
              </Card>

              {/* Clear All Data - NEW */}
              <Card className="shadow-lg border-2 border-orange-300 bg-orange-50">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-orange-700">
                    <XCircle className="w-5 h-5" />
                    پاکسازی کامل دیتا (Full Reset)
                  </CardTitle>
                  <CardDescription className="text-orange-600">
                    ⚠️ خطرناک! این عملیات تمام کاربران، تراکنش‌ها، پرداخت‌ها، و برداشت‌ها را حذف می‌کند
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <Dialog>
                    <DialogTrigger asChild>
                      <Button
                        variant="destructive"
                        className="w-full h-12 text-base font-bold bg-orange-600 hover:bg-orange-700"
                      >
                        <XCircle className="w-5 h-5 ml-2" />
                        پاکسازی کامل سیستم (Full Reset)
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle className="text-orange-600 text-xl">
                          ⚠️ هشدار: پاکسازی کامل!
                        </DialogTitle>
                        <DialogDescription className="text-base pt-4">
                          <div className="space-y-4">
                            <p className="text-orange-600 font-bold">
                              آیا مطمئن هستید که می‌خواهید تمام دیتا را پاک کنید؟
                            </p>
                            <div className="bg-orange-50 p-4 rounded-lg border-2 border-orange-200">
                              <p className="text-gray-700 mb-2 font-semibold">این عملیات موارد زیر را حذف می‌کند:</p>
                              <ul className="list-disc list-inside space-y-1 text-gray-600 text-sm">
                                <li>✖️ تمام کاربران (به جز seed و admin)</li>
                                <li>✖️ تمام تراکنش‌ها (transactions)</li>
                                <li>✖️ تمام پرداخت‌ها (payments)</li>
                                <li>✖️ تمام پرداخت‌های کریپتو (crypto_payments)</li>
                                <li>✖️ تمام درخواست‌های برداشت (withdrawals)</li>
                                <li>✖️ امتیاز seed به صفر ریست می‌شود</li>
                              </ul>
                              <div className="mt-3 p-2 bg-red-100 border border-red-300 rounded">
                                <p className="text-red-700 text-xs font-bold">
                                  🚨 این عملیات قابل بازگشت نیست!
                                </p>
                              </div>
                            </div>
                          </div>
                        </DialogDescription>
                      </DialogHeader>
                      <div className="flex gap-3 pt-4">
                        <DialogTrigger asChild>
                          <Button variant="outline" className="flex-1">
                            انصراف
                          </Button>
                        </DialogTrigger>
                        <Button
                          variant="destructive"
                          className="flex-1 font-bold bg-orange-600 hover:bg-orange-700"
                          onClick={() => {
                            clearAllUsers();
                            // Close dialog after action
                            document.querySelector('[data-state="open"]')?.click();
                          }}
                        >
                          بله، همه چیز را پاک کن
                        </Button>
                      </div>
                    </DialogContent>
                  </Dialog>
                  <p className="text-xs text-orange-600 mt-3 text-center">
                    💡 تنظیمات (API keys، آدرس‌ها) حفظ می‌شوند
                  </p>
                </CardContent>
              </Card>

              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TestTube className="w-5 h-5 text-emerald-600" />
                    ساخت کاربران تست
                  </CardTitle>
                  <CardDescription>کاربران تست را با فعال‌سازی خودکار ایجاد کنید</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="testCount">تعداد کاربران</Label>
                    <Input
                      id="testCount"
                      type="number"
                      value={testCount}
                      onChange={(e) => setTestCount(parseInt(e.target.value))}
                      className="h-11"
                      data-testid="test-count-input"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="testReferrer">کد معرف دعوت‌کننده (اختیاری)</Label>
                    <Input
                      id="testReferrer"
                      type="text"
                      placeholder="SEED یا کد معرف کاربر"
                      value={testReferrer}
                      onChange={(e) => setTestReferrer(e.target.value.toUpperCase())}
                      className="h-11"
                      data-testid="test-referrer-input"
                    />
                    <p className="text-xs text-gray-500">
                      کد معرف کاربر را از لیست کاربران کپی کنید
                    </p>
                  </div>
                  <div className="flex gap-3">
                    <Button
                      onClick={createTestUsers}
                      className="flex-1 bg-emerald-600 hover:bg-emerald-700 h-11"
                      data-testid="create-test-users-button"
                    >
                      ساخت کاربران تست
                    </Button>
                    <Button
                      onClick={clearTestUsers}
                      variant="destructive"
                      className="h-11"
                      data-testid="clear-test-users-button"
                    >
                      حذف کاربران تست
                    </Button>
                  </div>
                </CardContent>
              </Card>

              <Card className="shadow-lg border-0">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="w-5 h-5 text-emerald-600" />
                    پر کردن درخت
                  </CardTitle>
                  <CardDescription>درخت کاربر را با زیرمجموعه‌های تست پر کنید</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="fillUserId">کد معرف کاربر</Label>
                    <Input
                      id="fillUserId"
                      type="text"
                      placeholder="کد معرف (مثلاً SEED)"
                      value={fillUserId}
                      onChange={(e) => setFillUserId(e.target.value.toUpperCase())}
                      className="h-11"
                      data-testid="fill-user-id-input"
                    />
                    <p className="text-xs text-gray-500">
                      کد معرف کاربر را از لیست کاربران کپی کنید
                    </p>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="fillLevels">تعداد سطح‌ها</Label>
                    <Input
                      id="fillLevels"
                      type="number"
                      min="1"
                      max="7"
                      value={fillLevels}
                      onChange={(e) => setFillLevels(parseInt(e.target.value))}
                      className="h-11"
                      data-testid="fill-levels-input"
                    />
                  </div>
                  <Button
                    onClick={fillTree}
                    className="w-full bg-teal-600 hover:bg-teal-700 h-11"
                    data-testid="fill-tree-button"
                  >
                    پر کردن درخت
                  </Button>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="announcements">
            <AdminAnnouncements />
          </TabsContent>

          <TabsContent value="admin-settings">
            <Card className="shadow-lg border-0">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Lock className="w-5 h-5 text-emerald-600" />
                  تنظیمات حساب ادمین
                </CardTitle>
                <CardDescription>مدیریت حساب کاربری ادمین</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center py-12">
                  <Lock className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                  <p className="text-gray-500">تنظیمات حساب ادمین</p>
                  <p className="text-sm text-gray-400 mt-2">
                    این بخش در حال توسعه است
                  </p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}

export default AdminDashboard;
