from fastapi import FastAPI, APIRouter, HTTPException, Depends, status
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
import time
from pathlib import Path
from pydantic import BaseModel, Field, ConfigDict, EmailStr
from typing import List, Optional, Dict, Any
import uuid
from datetime import datetime, timezone, timedelta
import bcrypt
import jwt
from collections import deque, defaultdict
import httpx
import asyncio

# Cache Manager برای بهینه‌سازی
from cache_manager import cache, cached

# 🆕 Materialized Path Services
from services.tree.placement import PlacementService
from services.tree.placement import set_database as set_tree_placement_db
from services.tree.tree_stats import TreeStatsService
from services.tree.tree_stats import set_database as set_tree_stats_db
from services.rewards.calculator import RewardCalculator
from services.rewards.calculator import set_database as set_reward_calc_db
from services.rewards.distributor import RewardDistributor
from services.rewards.distributor import set_database as set_reward_dist_db
from services.user.activation import UserActivationService
from services.user.activation import set_database as set_user_activation_db
from services.user.registration import UserRegistrationService
from services.user.registration import set_database as set_user_reg_db

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# MongoDB connection با optimization برای production
mongo_url = os.environ.get('MONGO_URL', 'mongodb://localhost:27017')
client = AsyncIOMotorClient(
    mongo_url,
    maxPoolSize=200,      # افزایش برای concurrent requests
    minPoolSize=20,        # minimum connections
    maxIdleTimeMS=30000,  # idle timeout
    serverSelectionTimeoutMS=5000
)
db = client[os.environ.get('DB_NAME', 'mlm_production')]

# Create the main app without a prefix
app = FastAPI()

# Create a router with the /api prefix
api_router = APIRouter(prefix="/api")

# Security
security = HTTPBearer()
JWT_SECRET = os.environ.get('JWT_SECRET', 'mlm-secret-key-change-in-production')
JWT_ALGORITHM = 'HS256'

# ==================== AUDIT LOGGING ====================

async def log_audit_event(event_type: str, user_id: str, details: dict, ip: str = None):
    """ثبت رویدادهای مهم برای audit"""
    try:
        audit_entry = {
            "event_type": event_type,
            "user_id": user_id,
            "details": details,
            "ip_address": ip,
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        await db.audit_logs.insert_one(audit_entry)
        logging.info(f"Audit: {event_type} by {user_id}")
    except Exception as e:
        logging.error(f"Failed to log audit event: {e}")

# ==================== SECURITY UTILITIES ====================

# Input Sanitization
import re
import html

def sanitize_string(text: str, max_length: int = 1000) -> str:
    """پاکسازی ورودی از کاراکترهای خطرناک"""
    if not text:
        return ""
    
    # حذف HTML tags
    text = html.escape(text)
    
    # حذف کاراکترهای کنترلی
    text = re.sub(r'[\x00-\x1f\x7f-\x9f]', '', text)
    
    # محدود کردن طول
    return text[:max_length].strip()

def validate_email(email: str) -> bool:
    """بررسی فرمت ایمیل"""
    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    return bool(re.match(pattern, email))

def validate_password_strength(password: str) -> tuple[bool, str]:
    """بررسی قدرت رمز عبور"""
    if len(password) < 6:
        return False, "رمز عبور باید حداقل 6 کاراکتر باشد"
    
    if len(password) > 128:
        return False, "رمز عبور بیش از حد طولانی است"
    
    # حداقل یک حرف داشته باشد
    if not re.search(r'[a-zA-Z]', password):
        return False, "رمز عبور باید حداقل شامل یک حرف باشد"
    
    return True, "رمز عبور معتبر است"

def validate_wallet_address(address: str, network: str) -> bool:
    """بررسی صحت آدرس wallet"""
    if not address or not network:
        return False
    
    network = network.lower()
    
    # TRC20 (TRON)
    if network == 'trc20':
        return address.startswith('T') and len(address) == 34
    
    # ERC20/BEP20 (Ethereum/BSC)
    if network in ['erc20', 'bep20']:
        return address.startswith('0x') and len(address) == 42
    
    return False

# Password hashing
def hash_password(password: str) -> str:
    return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')

def verify_password(plain_password: str, hashed_password: str) -> bool:
    try:
        return bcrypt.checkpw(plain_password.encode('utf-8'), hashed_password.encode('utf-8'))
    except Exception:
        return False

def create_access_token(data: dict) -> str:
    to_encode = data.copy()
    expire = datetime.now(timezone.utc) + timedelta(days=30)
    to_encode.update({"exp": expire})
    return jwt.encode(to_encode, JWT_SECRET, algorithm=JWT_ALGORITHM)

def generate_unique_amount(base_amount: float, payment_id: str) -> float:
    """
    تولید مبلغ unique برای هر payment
    مثال: اگر base_amount = 20 باشد → 20.0021, 20.0123, 20.0456
    فرمت: XX.0YYY (4 رقم اعشار، رقم اول همیشه 0)
    این روش 999 مبلغ منحصر به فرد تولید می‌کند (0.0001 تا 0.0999)
    """
    # از hash payment_id برای تولید عدد بین 1 تا 999 استفاده می‌کنیم
    hash_value = sum(ord(c) for c in payment_id)
    unique_suffix = (hash_value % 999) + 1  # عددی بین 1 تا 999
    
    # مبلغ نهایی با 4 رقم اعشار: base + 0.0XXX
    decimal_part = unique_suffix / 10000.0  # 0.0001 تا 0.0999
    unique_amount = base_amount + decimal_part
    return round(unique_amount, 4)

def decode_token(token: str) -> dict:
    """Decode و validate کردن JWT token"""
    try:
        payload = jwt.decode(token, JWT_SECRET, algorithms=[JWT_ALGORITHM])
        
        # بررسی expiration
        if 'exp' not in payload:
            raise HTTPException(status_code=401, detail="توکن فاقد تاریخ انقضا است")
        
        # بررسی user_id
        if 'user_id' not in payload:
            raise HTTPException(status_code=401, detail="توکن نامعتبر است")
        
        return payload
        
    except jwt.ExpiredSignatureError:
        raise HTTPException(status_code=401, detail="توکن منقضی شده است. لطفاً دوباره وارد شوید")
    except jwt.InvalidTokenError:
        raise HTTPException(status_code=401, detail="توکن نامعتبر است")
    except Exception as e:
        logging.error(f"Token decode error: {e}")
        raise HTTPException(status_code=401, detail="خطا در اعتبارسنجی توکن")

async def get_current_user(credentials: HTTPAuthorizationCredentials = Depends(security)):
    """دریافت کاربر فعلی از طریق JWT token"""
    payload = decode_token(credentials.credentials)
    
    user = await db.users.find_one({"user_id": payload.get("user_id")}, {"_id": 0})
    if not user:
        raise HTTPException(status_code=401, detail="کاربر یافت نشد")
    
    # بررسی فعال بودن حساب (اختیاری)
    if user.get("status") == "banned":
        raise HTTPException(status_code=403, detail="حساب کاربری شما مسدود شده است")
    
    return user

async def get_admin_user(current_user: dict = Depends(get_current_user)):
    """بررسی دسترسی ادمین"""
    if current_user.get("role") not in ["admin", "seed"]:
        logging.warning(f"Unauthorized admin access attempt by user {current_user.get('user_id')}")
        raise HTTPException(status_code=403, detail="دسترسی محدود به ادمین")
    return current_user

# ==================== Blockchain Services ====================

# Contract Addresses
USDT_CONTRACTS = {
    "TRON": "TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t",
    "BSC": "0x55d398326f99059fF775485246999027B3197955"
}

class TronGridService:
    """سرویس تعامل با TronGrid API"""
    
    def __init__(self):
        self.base_url = "https://api.trongrid.io"
        self.http_client: Optional[httpx.AsyncClient] = None
    
    async def get_client(self) -> httpx.AsyncClient:
        """دریافت یا ایجاد HTTP client"""
        if not self.http_client:
            # دریافت API key از settings
            settings = await db.settings.find_one({"setting_id": "crypto"})
            api_key = settings.get("trongrid_api_key") if settings else None
            
            headers = {}
            if api_key:
                headers["TRON-PRO-API-KEY"] = api_key
            
            self.http_client = httpx.AsyncClient(
                headers=headers,
                timeout=30.0
            )
        return self.http_client
    
    async def close(self):
        """بستن HTTP client"""
        if self.http_client:
            await self.http_client.aclose()
            self.http_client = None
    
    async def get_trc20_transactions(
        self,
        address: str,
        limit: int = 50,
        only_to: bool = True
    ) -> List[Dict]:
        """دریافت تراکنش‌های TRC20 یک آدرس"""
        try:
            client = await self.get_client()
            
            url = f"{self.base_url}/v1/accounts/{address}/transactions/trc20"
            params = {
                "limit": limit,
                "only_to": str(only_to).lower(),
                "only_confirmed": "true"
            }
            
            response = await client.get(url, params=params)
            response.raise_for_status()
            
            data = response.json()
            transactions = data.get("data", [])
            
            logging.info(f"✅ دریافت {len(transactions)} تراکنش برای آدرس {address}")
            
            return transactions
            
        except httpx.HTTPError as e:
            logging.error(f"❌ خطا در دریافت تراکنش‌ها: {str(e)}")
            raise HTTPException(status_code=500, detail="خطا در اتصال به TronGrid")
        except Exception as e:
            logging.error(f"❌ خطا: {str(e)}")
            raise HTTPException(status_code=500, detail="خطا در پردازش تراکنش‌ها")
    
    async def filter_usdt_transactions(self, transactions: List[Dict]) -> List[Dict]:
        """فیلتر کردن تراکنش‌های USDT"""
        usdt_transactions = []
        
        for tx in transactions:
            token_info = tx.get("token_info", {})
            
            # چک کردن آیا تراکنش USDT است
            if token_info.get("address") == USDT_CONTRACTS["TRON"]:
                usdt_transactions.append(tx)
        
        logging.info(f"✅ {len(usdt_transactions)} تراکنش USDT یافت شد")
        
        return usdt_transactions
    
    async def parse_transaction_amount(self, transaction: Dict) -> float:
        """تبدیل مقدار تراکنش از واحد بلاکچین به USDT"""
        # USDT دارای 6 رقم اعشار است
        raw_amount = int(transaction.get("value", 0))
        usdt_amount = raw_amount / (10 ** 6)
        
        return usdt_amount
    
    async def validate_tron_address(self, address: str) -> bool:
        """اعتبارسنجی آدرس TRON"""
        try:
            # آدرس‌های TRON با T شروع می‌شوند و 34 کاراکتر دارند
            if not address.startswith("T") or len(address) != 34:
                return False
            
            client = await self.get_client()
            url = f"{self.base_url}/wallet/validateaddress"
            
            response = await client.post(url, json={"address": address})
            response.raise_for_status()
            
            data = response.json()
            return data.get("result", False)
            
        except Exception as e:
            logging.error(f"❌ خطا در اعتبارسنجی آدرس: {str(e)}")
            return False

class BSCService:
    """سرویس تعامل با BSC Blockchain از طریق Web3 RPC"""
    
    def __init__(self):
        # استفاده از Public RPC Nodes (رایگان و بدون نیاز به API key)
        self.rpc_urls = [
            "https://bsc.publicnode.com",
            "https://bsc-dataseed.binance.org/",
            "https://bsc-dataseed1.defibit.io/",
            "https://bsc-dataseed1.ninicoin.io/",
            "https://bsc-dataseed2.defibit.io/"
        ]
        self.http_client: Optional[httpx.AsyncClient] = None
    
    async def get_client(self) -> httpx.AsyncClient:
        """دریافت یا ایجاد HTTP client"""
        if not self.http_client:
            self.http_client = httpx.AsyncClient(timeout=30.0)
        return self.http_client
    
    async def close(self):
        """بستن HTTP client"""
        if self.http_client:
            await self.http_client.aclose()
            self.http_client = None
    
    async def get_working_rpc(self, client: httpx.AsyncClient) -> Optional[str]:
        """پیدا کردن یک RPC که کار می‌کند"""
        block_payload = {
            "jsonrpc": "2.0",
            "id": 1,
            "method": "eth_blockNumber",
            "params": []
        }
        
        for rpc_url in self.rpc_urls:
            try:
                resp = await client.post(rpc_url, json=block_payload, timeout=8.0)
                data = resp.json()
                if "result" in data:
                    logging.info(f"✅ RPC فعال: {rpc_url}")
                    return rpc_url
            except Exception as e:
                logging.warning(f"⚠️ RPC {rpc_url} در دسترس نیست: {str(e)}")
                continue
        
        return None
    
    async def get_bep20_transactions(
        self,
        address: str,
        contract_address: str,
        limit: int = 50
    ) -> List[Dict]:
        """دریافت تراکنش‌های BEP20 از blockchain با استفاده از eth_getLogs"""
        try:
            client = await self.get_client()
            
            # پیدا کردن RPC فعال
            rpc_url = await self.get_working_rpc(client)
            if not rpc_url:
                logging.error("❌ هیچ RPC فعالی برای BSC یافت نشد")
                return []
            
            # دریافت block number فعلی
            block_payload = {
                "jsonrpc": "2.0",
                "id": 1,
                "method": "eth_blockNumber",
                "params": []
            }
            
            resp = await client.post(rpc_url, json=block_payload, timeout=10.0)
            data = resp.json()
            current_block = int(data["result"], 16)
            
            logging.info(f"📊 Block فعلی BSC: {current_block}")
            
            # جستجو در 10000 بلاک اخیر (حدود 8 ساعت در BSC - هر بلاک ~3 ثانیه)
            from_block = max(0, current_block - 10000)
            
            # Transfer event signature: Transfer(address,address,uint256)
            transfer_topic = "0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef"
            
            # آدرس مقصد به فرمت topic (32 bytes با padding)
            to_topic = "0x000000000000000000000000" + address[2:].lower()
            
            logs_payload = {
                "jsonrpc": "2.0",
                "id": 2,
                "method": "eth_getLogs",
                "params": [{
                    "fromBlock": hex(from_block),
                    "toBlock": "latest",
                    "address": contract_address,
                    "topics": [
                        transfer_topic,  # Transfer event
                        None,            # from (هر آدرسی)
                        to_topic         # to (آدرس ما)
                    ]
                }]
            }
            
            logging.info(f"🔍 جستجوی تراکنش‌های BSC از بلاک {from_block} تا {current_block}")
            
            # دریافت لاگ‌ها
            resp = await client.post(rpc_url, json=logs_payload, timeout=20.0)
            data = resp.json()
            
            if "error" in data:
                logging.error(f"❌ خطای RPC: {data['error']}")
                return []
            
            if "result" not in data:
                logging.warning("⚠️ پاسخ RPC بدون result")
                return []
            
            logs = data["result"]
            logging.info(f"✅ {len(logs)} تراکنش Transfer یافت شد برای {address}")
            
            # تبدیل لاگ‌ها به فرمت transaction
            transactions = []
            for log in logs[-limit:]:  # فقط limit تا آخرین
                # Parse amount از data (hex to decimal)
                amount_hex = log.get("data", "0x0")
                try:
                    amount = int(amount_hex, 16) if amount_hex else 0
                except:
                    amount = 0
                
                tx = {
                    "hash": log.get("transactionHash"),
                    "to": address,
                    "value": str(amount),
                    "tokenDecimal": "18",  # USDT BEP20 has 18 decimals
                    "contractAddress": contract_address,
                    "blockNumber": log.get("blockNumber")
                }
                transactions.append(tx)
            
            return transactions
                
        except Exception as e:
            logging.error(f"❌ خطا در دریافت تراکنش‌های BSC: {str(e)}")
            return []
    
    async def parse_transaction_amount(self, transaction: Dict) -> float:
        """تبدیل مقدار تراکنش از واحد بلاکچین به USDT"""
        # USDT BEP20 دارای 18 رقم اعشار است (مثل ERC20)
        raw_amount = int(transaction.get("value", 0))
        
        # چک کردن token decimals - USDT BEP20 = 18 decimals
        token_decimal = int(transaction.get("tokenDecimal", 18))
        usdt_amount = raw_amount / (10 ** token_decimal)
        
        return usdt_amount
    
    async def validate_bsc_address(self, address: str) -> bool:
        """اعتبارسنجی آدرس BSC"""
        try:
            # آدرس‌های BSC با 0x شروع می‌شوند و 42 کاراکتر دارند
            if not address.startswith("0x") or len(address) != 42:
                return False
            
            # بررسی hexadecimal
            try:
                int(address[2:], 16)
                return True
            except ValueError:
                return False
                
        except Exception as e:
            logging.error(f"❌ خطا در اعتبارسنجی آدرس BSC: {str(e)}")
            return False

# ========================================
# توابع ارسال تراکنش (Auto-Transfer)
# ========================================

async def get_tron_wallet_balance(private_key: str) -> dict:
    """دریافت موجودی کیف پول TRON"""
    try:
        from tronpy import Tron
        from tronpy.keys import PrivateKey
        
        # پاک کردن 0x اگر وجود دارد
        if private_key.startswith('0x'):
            private_key = private_key[2:]
        
        client = Tron()
        priv_key = PrivateKey(bytes.fromhex(private_key))
        address = priv_key.public_key.to_base58check_address()
        
        # موجودی TRX
        account = client.get_account(address)
        trx_balance = account.get('balance', 0) / 1_000_000
        
        # موجودی USDT
        usdt_contract = "TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t"
        contract = client.get_contract(usdt_contract)
        usdt_balance = contract.functions.balanceOf(address) / 1_000_000
        
        return {
            "success": True,
            "address": address,
            "trx_balance": trx_balance,
            "usdt_balance": usdt_balance,
            "network": "TRON"
        }
    except Exception as e:
        logging.error(f"❌ خطا در دریافت موجودی TRON: {str(e)}")
        return {
            "success": False,
            "error": str(e)
        }

async def get_bsc_wallet_balance(private_key: str) -> dict:
    """دریافت موجودی کیف پول BSC"""
    try:
        from web3 import Web3
        
        # اضافه کردن 0x اگر وجود ندارد
        if not private_key.startswith('0x'):
            private_key = '0x' + private_key
        
        rpc_url = "https://bsc.publicnode.com"
        w3 = Web3(Web3.HTTPProvider(rpc_url))
        
        if not w3.is_connected():
            raise Exception("اتصال به BSC برقرار نشد")
        
        account = w3.eth.account.from_key(private_key)
        address = account.address
        
        # موجودی BNB
        bnb_balance = w3.eth.get_balance(address)
        bnb_balance = float(w3.from_wei(bnb_balance, 'ether'))
        
        # موجودی USDT
        usdt_contract_address = "0x55d398326f99059fF775485246999027B3197955"
        usdt_abi = [{"constant":True,"inputs":[{"name":"_owner","type":"address"}],"name":"balanceOf","outputs":[{"name":"balance","type":"uint256"}],"type":"function"}]
        
        contract = w3.eth.contract(
            address=Web3.to_checksum_address(usdt_contract_address),
            abi=usdt_abi
        )
        
        usdt_balance = contract.functions.balanceOf(Web3.to_checksum_address(address)).call()
        usdt_balance = float(usdt_balance / (10 ** 18))
        
        return {
            "success": True,
            "address": address,
            "bnb_balance": bnb_balance,
            "usdt_balance": usdt_balance,
            "network": "BSC"
        }
    except Exception as e:
        logging.error(f"❌ خطا در دریافت موجودی BSC: {str(e)}")
        return {
            "success": False,
            "error": str(e)
        }

async def send_tron_usdt(
    private_key: str,
    to_address: str,
    amount_usdt: float
) -> dict:
    """ارسال USDT روی شبکه TRON"""
    try:
        from tronpy import Tron
        from tronpy.keys import PrivateKey
        
        # پاک کردن 0x اگر وجود دارد
        if private_key.startswith('0x'):
            private_key = private_key[2:]
        
        # اتصال به شبکه TRON
        client = Tron()
        
        # ایجاد کیف پول از private key
        priv_key = PrivateKey(bytes.fromhex(private_key))
        
        # آدرس USDT در TRON
        usdt_contract = "TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t"
        
        # تبدیل مبلغ به واحد کوچکتر (6 decimals در TRON)
        amount_sun = int(amount_usdt * (10 ** 6))
        
        # ساخت تراکنش
        contract = client.get_contract(usdt_contract)
        txn = (
            contract.functions.transfer(to_address, amount_sun)
            .with_owner(priv_key.public_key.to_base58check_address())
            .fee_limit(100_000_000)  # 100 TRX max fee
            .build()
            .sign(priv_key)
        )
        
        # ارسال تراکنش
        result = txn.broadcast()
        
        logging.info(f"✅ تراکنش TRON ارسال شد: {result.get('txid')}")
        
        return {
            "success": True,
            "tx_hash": result.get("txid"),
            "network": "TRON",
            "amount": amount_usdt
        }
        
    except Exception as e:
        logging.error(f"❌ خطا در ارسال USDT TRON: {str(e)}")
        return {
            "success": False,
            "error": str(e),
            "network": "TRON"
        }

async def send_bsc_usdt(
    private_key: str,
    to_address: str,
    amount_usdt: float
) -> dict:
    """ارسال USDT روی شبکه BSC"""
    try:
        from web3 import Web3
        
        # اضافه کردن 0x اگر وجود ندارد
        if not private_key.startswith('0x'):
            private_key = '0x' + private_key
        
        # اتصال به BSC
        rpc_url = "https://bsc.publicnode.com"
        w3 = Web3(Web3.HTTPProvider(rpc_url))
        
        if not w3.is_connected():
            raise Exception("اتصال به BSC RPC برقرار نشد")
        
        # آدرس USDT در BSC
        usdt_contract_address = "0x55d398326f99059fF775485246999027B3197955"
        
        # تبدیل مبلغ به واحد کوچکتر (18 decimals در BSC)
        amount_wei = int(amount_usdt * (10 ** 18))
        
        # دریافت account از private key
        account = w3.eth.account.from_key(private_key)
        
        # ABI مینیمال برای transfer
        usdt_abi = [
            {
                "constant": False,
                "inputs": [
                    {"name": "_to", "type": "address"},
                    {"name": "_value", "type": "uint256"}
                ],
                "name": "transfer",
                "outputs": [{"name": "", "type": "bool"}],
                "type": "function"
            }
        ]
        
        # ایجاد contract instance
        contract = w3.eth.contract(
            address=Web3.to_checksum_address(usdt_contract_address),
            abi=usdt_abi
        )
        
        # ساخت تراکنش
        nonce = w3.eth.get_transaction_count(account.address)
        
        tx = contract.functions.transfer(
            Web3.to_checksum_address(to_address),
            amount_wei
        ).build_transaction({
            'from': account.address,
            'nonce': nonce,
            'gas': 100000,
            'gasPrice': w3.eth.gas_price
        })
        
        # امضای تراکنش
        signed_tx = w3.eth.account.sign_transaction(tx, private_key)
        
        # ارسال تراکنش
        tx_hash = w3.eth.send_raw_transaction(signed_tx.raw_transaction)
        tx_hash_hex = tx_hash.hex()
        
        logging.info(f"✅ تراکنش BSC ارسال شد: {tx_hash_hex}")
        
        return {
            "success": True,
            "tx_hash": tx_hash_hex,
            "network": "BSC",
            "amount": amount_usdt
        }
        
    except Exception as e:
        logging.error(f"❌ خطا در ارسال USDT BSC: {str(e)}")
        return {
            "success": False,
            "error": str(e),
            "network": "BSC"
        }

# ایجاد instance سرویس‌ها
tron_service = TronGridService()
bsc_service = BSCService()

# بستن سرویس‌ها هنگام shutdown
@app.on_event("shutdown")
async def shutdown_event():
    await tron_service.close()
    await bsc_service.close()

# Models
class UserRegister(BaseModel):
    email: EmailStr
    password: str
    phone_number: Optional[str] = None
    referral_code: Optional[str] = None
    telegram_session: Optional[str] = None  # session_id از تلگرام

class UserLogin(BaseModel):
    email: EmailStr
    password: str

class User(BaseModel):
    user_id: str
    email: str
    phone_number: Optional[str] = None  # شماره تماس
    role: str = "user"  # user or admin
    status: str = "pending"  # pending, active
    referrer_id: Optional[str] = None
    placement_parent_id: Optional[str] = None
    direct_children: List[str] = []
    depth: int = 0
    total_points: float = 0.0
    blocked_direct_reward: bool = False
    blocked_level_reward: bool = False
    is_tree_complete: bool = False  # آیا هرم 7 سطحی کامل شده؟
    created_at: str
    activated_at: Optional[str] = None
    referral_code: str

class RewardSettings(BaseModel):
    activation_fee: float = 100.0  # هزینه ورود به سیستم (دلار)
    seed_initial_payment: float = 100.0  # سرمایه اولیه seed (دلار)
    direct_first: float = 100.0
    direct_second: float = 75.0
    direct_third: float = 50.0
    direct_extra: float = 25.0
    level_3: float = 10.0
    level_5: float = 5.0
    level_7: float = 2.0

class Transaction(BaseModel):
    transaction_id: str
    user_id: str
    from_user_id: str
    reward_type: str  # direct, level_3, level_5, level_7
    amount: float
    timestamp: str
    description: str

class Payment(BaseModel):
    payment_id: str
    user_id: str
    amount: float  # مبلغ پرداختی برای فعال‌سازی
    timestamp: str
    description: str

class TestUserCreate(BaseModel):
    count: int = 1
    auto_activate: bool = True
    referral_code: Optional[str] = None  # تغییر به referral_code

class TreeNode(BaseModel):
    user_id: str
    email: str
    status: str
    depth: int
    total_points: float
    direct_children_count: int
    children: List['TreeNode'] = []

# Initialize default settings and seed user
async def init_db():
    # اتصال به Redis Cache
    await cache.connect()
    
    # Create indexes for performance optimization
    await db.users.create_index("user_id", unique=True)
    await db.users.create_index("email", unique=True)
    await db.users.create_index("referral_code", unique=True)
    await db.users.create_index("status")
    await db.users.create_index("depth")
    await db.users.create_index("referrer_id")
    await db.users.create_index("placement_parent_id")
    await db.users.create_index([("status", 1), ("depth", 1)])
    
    # 🆕 Materialized Path indexes (Critical for 100K+ users)
    await db.users.create_index("placement_path")
    await db.users.create_index([("placement_path", 1), ("status", 1)])
    await db.users.create_index([("placement_path", 1), ("depth", 1)])
    await db.users.create_index([("status", 1), ("direct_children_count", 1), ("depth", 1)])
    
    await db.transactions.create_index("user_id")
    await db.transactions.create_index("from_user_id")
    await db.withdrawals.create_index("user_id")
    await db.withdrawals.create_index("status")
    await db.reward_locks.create_index("lock_id", unique=True)
    
    # Check if settings exist
    settings = await db.settings.find_one({"setting_id": "rewards"})
    if not settings:
        default_settings = {
            "setting_id": "rewards",
            "activation_fee": 100.0,
            "seed_initial_payment": 100.0,
            "direct_first": 100.0,
            "direct_second": 75.0,
            "direct_third": 50.0,
            "direct_extra": 25.0,
            "level_3": 10.0,
            "level_5": 5.0,
            "level_7": 2.0
        }
        await db.settings.insert_one(default_settings)
    
    # Check if crypto settings exist
    crypto_settings = await db.settings.find_one({"setting_id": "crypto"})
    if not crypto_settings:
        default_crypto_settings = {
            "setting_id": "crypto",
            "tron_enabled": True,
            "tron_api_key": "",
            "tron_wallet_address": "TLyqzVGLV1srkB7dToTAEqgDSfPtXRJZYH",
            "bsc_enabled": False,
            "bsc_api_key": "",
            "bsc_wallet_address": "",
            "min_withdrawal_amount": 10.0,
            "withdrawal_fee_type": "fixed",
            "withdrawal_fee_amount": 0.0
        }
        await db.settings.insert_one(default_crypto_settings)
    
    # Check if seed user exists
    seed = await db.users.find_one({"user_id": "seed"})
    if not seed:
        seed_user = {
            "user_id": "seed",
            "email": "admin@mlm.com",
            "phone_number": None,
            "password": hash_password("admin123"),
            "role": "admin",
            "status": "active",
            "referrer_id": None,
            
            # Materialized Path fields
            "placement_path": "seed",
            "placement_parent_id": None,
            "placement_position": None,
            "depth": 0,
            "ancestor_depths": [],
            
            # Tree stats
            "direct_children": [],
            "direct_children_count": 0,
            "total_descendants": 0,
            
            # Rewards
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            
            # Timestamps
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": datetime.now(timezone.utc).isoformat(),
            "referral_code": "SEED",
            
            # Optional
            "telegram_chat_id": None,
            "wallet_address": None,
            "wallet_network": None
        }
        await db.users.insert_one(seed_user)
        logging.info("✅ Seed user created: admin@mlm.com / admin123")
    else:
        # Update existing seed با Materialized Path fields if missing
        updates = {}
        if "placement_path" not in seed:
            updates["placement_path"] = "seed"
        if "direct_children_count" not in seed:
            updates["direct_children_count"] = len(seed.get("direct_children", []))
        if "total_descendants" not in seed:
            updates["total_descendants"] = 0
        if "ancestor_depths" not in seed:
            updates["ancestor_depths"] = []
        
        if updates:
            await db.users.update_one({"user_id": "seed"}, {"$set": updates})
            logging.info("✅ Seed user updated with Materialized Path fields")
    
    logging.info("✅ Database initialized successfully")

@app.on_event("startup")
async def startup_event():
    await init_db()
    
    # 🆕 Initialize Materialized Path Services
    set_tree_placement_db(db)
    set_tree_stats_db(db)
    set_reward_calc_db(db)
    set_reward_dist_db(db)
    set_user_activation_db(db)
    set_user_reg_db(db)
    
    logging.info("✅ Materialized Path services initialized with database")

@app.on_event("shutdown")
async def shutdown_event():
    await cache.disconnect()

# Atomic Placement Helper - جلوگیری از race conditions
async def try_atomic_placement(parent_id: str, child_id: str) -> bool:
    """
    تلاش برای قرار دادن child به صورت atomic
    برمی‌گرداند True اگر موفق باشد، False اگر parent پر باشد
    """
    result = await db.users.update_one(
        {
            "user_id": parent_id,
            "status": "active",
            "$expr": {"$lt": [{"$size": "$direct_children"}, 3]}
        },
        {"$push": {"direct_children": child_id}}
    )
    return result.modified_count > 0

# ==================== OLD FUNCTIONS - REPLACED WITH WRAPPERS ====================
# این wrapperها کد قدیمی را به services جدید redirect می‌کنند

async def find_placement_parent(referrer_id: str) -> str:
    """⚠️ DEPRECATED Wrapper: redirects به PlacementService"""
    parent_info = await PlacementService.find_available_parent(referrer_id)
    if parent_info:
        return parent_info["parent_id"]
    raise HTTPException(status_code=500, detail="Cannot find placement parent")

async def find_placement_parent_atomic(referrer_id: str, new_user_id: str) -> str:
    """⚠️ DEPRECATED Wrapper: redirects به PlacementService"""
    return await find_placement_parent(referrer_id)

async def distribute_rewards(new_user_id: str, referrer_id: str, placement_parent_id: str):
    """⚠️ DEPRECATED Wrapper: redirects به RewardDistributor"""
    await RewardDistributor.distribute_all_rewards(new_user_id)

async def check_tree_completion(user_id: str):
    """⚠️ DEPRECATED Wrapper: استفاده از total_descendants field"""
    # Tree completion check حالا در TreeStatsService.check_tree_completion است
    # که در activation خودکار فراخوانی می‌شود
    pass  # No-op - handled in activation

# ==================== END WRAPPERS ====================
    """
    الگوریتم BFS بهینه شده برای پیدا کردن parent مناسب
    - بدون محدودیت depth (هرم بی‌نهایت است)
    - از bulk queries برای سرعت بیشتر
    ⚠️ هشدار: این تابع non-atomic است و برای ایجاد چندین کاربر همزمان مناسب نیست
    """
    referrer = await db.users.find_one(
        {"user_id": referrer_id, "status": "active"}, 
        {"_id": 0, "direct_children": 1}
    )
    if not referrer:
        return "seed"
    
    # Check if referrer has less than 3 direct children
    if len(referrer.get("direct_children", [])) < 3:
        return referrer_id
    
    # BFS to find first available slot - بهینه شده با batch queries
    queue = deque([referrer_id])
    visited = set([referrer_id])
    batch_size = 50  # تعداد user هایی که یکجا fetch می‌کنیم
    
    while queue:
        # پردازش batch از queue
        batch = []
        for _ in range(min(batch_size, len(queue))):
            if queue:
                batch.append(queue.popleft())
        
        if not batch:
            break
        
        # Fetch all users in batch با یک query
        users = await db.users.find(
            {"user_id": {"$in": batch}, "status": "active"},
            {"_id": 0, "user_id": 1, "direct_children": 1}
        ).to_list(batch_size)
        
        # بررسی هر user
        for current_user in users:
            current_id = current_user.get("user_id")
            children = current_user.get("direct_children", [])
            
            # اگر کمتر از 3 فرزند دارد، اینجا place کن
            if len(children) < 3:
                return current_id
            
            # اضافه کردن فرزندان به queue
            for child_id in children:
                if child_id not in visited:
                    visited.add(child_id)
                    queue.append(child_id)
    
    # اگر جایی پیدا نشد، به seed برگردان
    return "seed"


async def find_placement_parent_atomic(referrer_id: str, new_user_id: str) -> str:
    """
    🔒 الگوریتم BFS با Atomic Placement
    این تابع تضمین می‌کند که هیچ دو کاربری در یک slot قرار نمی‌گیرند
    از find_one_and_update برای atomic operation استفاده می‌کند
    """
    
    # اول سعی می‌کنیم referrer را پیدا کنیم
    referrer = await db.users.find_one(
        {"user_id": referrer_id, "status": "active"}, 
        {"_id": 0, "direct_children": 1, "user_id": 1}
    )
    
    if not referrer:
        # اگر referrer وجود ندارد، به seed متصل کن
        referrer_id = "seed"
        referrer = await db.users.find_one(
            {"user_id": "seed"}, 
            {"_id": 0, "direct_children": 1, "user_id": 1}
        )
    
    # BFS برای پیدا کردن slot خالی و reserve کردن آن به صورت atomic
    queue = deque([referrer_id])
    visited = set([referrer_id])
    batch_size = 50
    
    while queue:
        # پردازش batch از queue
        batch = []
        for _ in range(min(batch_size, len(queue))):
            if queue:
                batch.append(queue.popleft())
        
        if not batch:
            break
        
        # بررسی هر کاربر در batch به ترتیب
        for candidate_id in batch:
            # 🔒 Atomic: سعی کن این کاربر را به عنوان parent انتخاب کن
            # فقط اگر کمتر از 3 فرزند دارد
            result = await db.users.find_one_and_update(
                {
                    "user_id": candidate_id,
                    "status": "active",
                    "$expr": {"$lt": [{"$size": {"$ifNull": ["$direct_children", []]}}, 3]}
                },
                {"$push": {"direct_children": new_user_id}},
                return_document=True
            )
            
            if result:
                # موفقیت! این کاربر parent ماست
                return candidate_id
        
        # اگر هیچکدام از batch موفق نبود، فرزندان را به queue اضافه کن
        users = await db.users.find(
            {"user_id": {"$in": batch}, "status": "active"},
            {"_id": 0, "user_id": 1, "direct_children": 1}
        ).to_list(batch_size)
        
        for user in users:
            for child_id in user.get("direct_children", []):
                if child_id not in visited:
                    visited.add(child_id)
                    queue.append(child_id)
    
    # آخرین تلاش: به seed متصل کن
    result = await db.users.find_one_and_update(
        {
            "user_id": "seed",
            "$expr": {"$lt": [{"$size": {"$ifNull": ["$direct_children", []]}}, 3]}
        },
        {"$push": {"direct_children": new_user_id}},
        return_document=True
    )
    
    if result:
        return "seed"
    
    # اگر seed هم پر است، اولین کاربر با slot خالی را پیدا کن
    any_available = await db.users.find_one_and_update(
        {
            "status": "active",
            "$expr": {"$lt": [{"$size": {"$ifNull": ["$direct_children", []]}}, 3]}
        },
        {"$push": {"direct_children": new_user_id}},
        return_document=True
    )
    
    if any_available:
        return any_available["user_id"]
    
    # در غیر این صورت، خطا
    raise HTTPException(status_code=500, detail="امکان یافتن جایگاه مناسب وجود ندارد")

# Calculate and distribute rewards
async def distribute_rewards(new_user_id: str, referrer_id: str, placement_parent_id: str):
    """
    توزیع پاداش‌ها با جلوگیری از race condition و duplicate rewards
    
    🔒 محافظت‌های امنیتی:
    1. Atomic transaction insert برای جلوگیری از duplicate rewards
    2. شمارش children از دیتابیس (نه از آرایه) برای دقت بیشتر
    3. چک وضعیت کاربر جدید قبل از پاداش‌دهی
    """
    settings = await db.settings.find_one({"setting_id": "rewards"})
    if not settings:
        return
    
    # 🔒 FIX 1: چک کردن وضعیت کاربر جدید - فقط کاربران active پاداش دریافت می‌کنند
    new_user_check = await db.users.find_one({"user_id": new_user_id}, {"_id": 0, "status": 1})
    if not new_user_check or new_user_check.get("status") != "active":
        logging.warning(f"⚠️ کاربر {new_user_id} فعال نیست، پاداش داده نمی‌شود")
        return
    
    # 🔒 FIX 2: Atomic duplicate check با unique index روی transaction
    # ابتدا تلاش می‌کنیم یک transaction placeholder ایجاد کنیم
    # اگر قبلاً وجود داشته باشد، از ادامه جلوگیری می‌شود
    reward_lock_id = f"reward_lock_{new_user_id}_direct"
    try:
        # تلاش برای ایجاد یک lock موقت (این عملیات atomic است)
        lock_result = await db.reward_locks.update_one(
            {"lock_id": reward_lock_id},
            {"$setOnInsert": {
                "lock_id": reward_lock_id,
                "new_user_id": new_user_id,
                "created_at": datetime.now(timezone.utc).isoformat()
            }},
            upsert=True
        )
        
        # اگر modified_count == 0 و upserted_id وجود ندارد، یعنی قبلاً وجود داشته
        if lock_result.upserted_id is None and lock_result.modified_count == 0:
            logging.info(f"⚠️ پاداش قبلاً برای {new_user_id} در حال پردازش است. از ارسال مجدد جلوگیری شد.")
            return
    except Exception as e:
        logging.error(f"❌ خطا در ایجاد lock برای {new_user_id}: {e}")
        # در صورت خطا، با روش قدیمی ادامه می‌دهیم
        existing_reward = await db.transactions.find_one({
            "from_user_id": new_user_id,
            "reward_type": "direct"
        })
        if existing_reward:
            logging.info(f"⚠️ پاداش قبلاً برای {new_user_id} داده شده است. از ارسال مجدد جلوگیری شد.")
            return
    
    referrer = await db.users.find_one({"user_id": referrer_id}, {"_id": 0})
    # اگر referrer وجود نداشته باشد، یا پاداش مستقیم مسدود شده، یا هرمش کامل شده باشد
    if not referrer or referrer.get("blocked_direct_reward") or referrer.get("is_tree_complete"):
        return
    
    # 🔒 FIX 3: شمارش صحیح direct children فعال از دیتابیس (نه از آرایه)
    # فقط کاربران active که referrer_id برابر با referrer_id است و خود کاربر جدید نیست
    direct_count = await db.users.count_documents({
        "referrer_id": referrer_id,
        "status": "active",
        "user_id": {"$ne": new_user_id}
    })
    
    logging.info(f"📊 تعداد direct children فعال {referrer_id}: {direct_count}")
    
    if direct_count == 0:
        reward = settings["direct_first"]
        reward_type = "مستقیم نفر اول"
    elif direct_count == 1:
        reward = settings["direct_second"]
        reward_type = "مستقیم نفر دوم"
    elif direct_count == 2:
        reward = settings["direct_third"]
        reward_type = "مستقیم نفر سوم"
    else:
        reward = settings["direct_extra"]
        reward_type = "مستقیم اضافی"
    
    # Add reward
    await db.users.update_one(
        {"user_id": referrer_id},
        {"$inc": {"total_points": reward}}
    )
    
    # Log transaction
    transaction = {
        "transaction_id": str(uuid.uuid4()),
        "user_id": referrer_id,
        "from_user_id": new_user_id,
        "reward_type": "direct",
        "amount": reward,
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "description": f"پاداش {reward_type}"
    }
    await db.transactions.insert_one(transaction)
    
    logging.info(f"✅ پاداش {reward} به {referrer_id} برای {reward_type} داده شد")
    
    # Level rewards - به ازای ورود هر کاربر، به سرشاخه‌های واجد شرایط پاداش داده می‌شود
    new_user = await db.users.find_one({"user_id": new_user_id}, {"_id": 0})
    new_user_depth = new_user.get("depth", 0)
    
    # پیمایش به بالای درخت و چک کردن فاصله
    current_user = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
    
    while current_user:
        # اگر پاداش سطح مسدود نشده و هرم کامل نشده باشد
        if not current_user.get("blocked_level_reward") and not current_user.get("is_tree_complete"):
            current_depth = current_user.get("depth", 0)
            depth_diff = new_user_depth - current_depth
            
            # محدودیت 7 سطح: فقط به کسانی که حداکثر 7 سطح پایین‌تر هستند پاداش داده می‌شود
            if depth_diff > 7:
                # این کاربر بیش از 7 سطح پایین‌تر است، دیگر پاداشی نمی‌گیرد
                break
            
            # چک کردن آیا این فاصله با سطوح پاداش‌دهنده مطابقت دارد
            # منطق: لول = depth_diff + 1
            # لول 3 = depth_diff 2، لول 5 = depth_diff 4، لول 7 = depth_diff 6
            if depth_diff == 2:
                level_reward = settings["level_3"]
                level_type = "سطح ۳"
                
                # پرداخت پاداش
                await db.users.update_one(
                    {"user_id": current_user["user_id"]},
                    {"$inc": {"total_points": level_reward}}
                )
                
                transaction = {
                    "transaction_id": str(uuid.uuid4()),
                    "user_id": current_user["user_id"],
                    "from_user_id": new_user_id,
                    "reward_type": "level_3",
                    "amount": level_reward,
                    "timestamp": datetime.now(timezone.utc).isoformat(),
                    "description": f"پاداش {level_type} - ورود کاربر در لول ۳ (عمق {new_user_depth})"
                }
                await db.transactions.insert_one(transaction)
            
            elif depth_diff == 4:
                level_reward = settings["level_5"]
                level_type = "سطح ۵"
                
                await db.users.update_one(
                    {"user_id": current_user["user_id"]},
                    {"$inc": {"total_points": level_reward}}
                )
                
                transaction = {
                    "transaction_id": str(uuid.uuid4()),
                    "user_id": current_user["user_id"],
                    "from_user_id": new_user_id,
                    "reward_type": "level_5",
                    "amount": level_reward,
                    "timestamp": datetime.now(timezone.utc).isoformat(),
                    "description": f"پاداش {level_type} - ورود کاربر در لول ۵ (عمق {new_user_depth})"
                }
                await db.transactions.insert_one(transaction)
            
            elif depth_diff == 6:
                level_reward = settings["level_7"]
                level_type = "سطح ۷"
                
                await db.users.update_one(
                    {"user_id": current_user["user_id"]},
                    {"$inc": {"total_points": level_reward}}
                )
                
                transaction = {
                    "transaction_id": str(uuid.uuid4()),
                    "user_id": current_user["user_id"],
                    "from_user_id": new_user_id,
                    "reward_type": "level_7",
                    "amount": level_reward,
                    "timestamp": datetime.now(timezone.utc).isoformat(),
                    "description": f"پاداش {level_type} - ورود کاربر در لول ۷ (عمق {new_user_depth})"
                }
                await db.transactions.insert_one(transaction)
        
        # Move to parent
        parent_id = current_user.get("placement_parent_id")
        if parent_id:
            current_user = await db.users.find_one({"user_id": parent_id}, {"_id": 0})
        else:
            break

# Check if user tree is complete - بهینه شده بدون recursion
async def check_tree_completion(user_id: str):
    """
    بررسی کامل بودن هرم بدون recursion - برای مقیاس‌پذیری
    از MongoDB aggregation برای شمارش descendants استفاده می‌کند
    """
    user = await db.users.find_one({"user_id": user_id}, {"_id": 0, "is_tree_complete": 1, "depth": 1})
    if not user or user.get("is_tree_complete"):
        return
    
    user_depth = user.get("depth", 0)
    
    # شمارش descendants تا 7 سطح پایین‌تر با aggregation
    # این بسیار سریع‌تر از recursion است
    pipeline = [
        {"$match": {"user_id": user_id}},
        {
            "$graphLookup": {
                "from": "users",
                "startWith": "$user_id",
                "connectFromField": "user_id",
                "connectToField": "placement_parent_id",
                "as": "descendants",
                "maxDepth": 6,  # 7 levels (0-6)
                "depthField": "level",
                "restrictSearchWithMatch": {
                    "status": "active",
                    "depth": {"$lte": user_depth + 7}
                }
            }
        },
        {
            "$project": {
                "total_descendants": {"$size": "$descendants"}
            }
        }
    ]
    
    try:
        result = await db.users.aggregate(pipeline).to_list(1)
        if result:
            total_descendants = result[0].get("total_descendants", 0)
            
            # هرم کامل 7 سطحی: 3 + 9 + 27 + 81 + 243 + 729 = 1092
            max_possible = 1092
            
            if total_descendants >= max_possible:
                await db.users.update_one(
                    {"user_id": user_id},
                    {"$set": {"is_tree_complete": True}}
                )
    except Exception as e:
        # در صورت خطا، از fallback استفاده می‌کنیم (simple count)
        # این برای backward compatibility است
        count = await db.users.count_documents({
            "placement_parent_id": user_id,
            "status": "active"
        })
        if count >= 1092:
            await db.users.update_one(
                {"user_id": user_id},
                {"$set": {"is_tree_complete": True}}
            )

# ==================== DIAGNOSTIC ENDPOINTS ====================

@api_router.get("/health")
async def health_check():
    """بررسی سلامت سیستم"""
    result = {
        "status": "ok",
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "mongodb": False,
        "seed_user": False,
        "error": None
    }
    
    try:
        # تست MongoDB
        await db.command('ping')
        result["mongodb"] = True
        
        # تست seed user
        seed = await db.users.find_one({"user_id": "seed"}, {"_id": 0, "email": 1})
        if seed:
            result["seed_user"] = True
            result["seed_email"] = seed.get("email")
            
    except Exception as e:
        result["status"] = "error"
        result["error"] = str(e)
    
    return result

@api_router.get("/metrics")
async def get_system_metrics():
    """دریافت metrics سیستم"""
    try:
        from utils.metrics import get_metrics
        return get_metrics()
    except:
        # اگر metrics موجود نباشد
        users_count = await db.users.count_documents({"status": "active"})
        transactions_count = await db.transactions.count_documents({})
        
        return {
            "users": {"active": users_count},
            "transactions": {"total": transactions_count}
        }



@api_router.get("/debug/test-login/{email}")
async def debug_test_login(email: str):
    """تست دستی برای بررسی کاربر (فقط برای دیباگ)"""
    try:
        email = email.strip().lower()
        user = await db.users.find_one({"email": email}, {"_id": 0, "user_id": 1, "email": 1, "role": 1, "status": 1, "password": 1})
        
        if not user:
            return {"found": False, "message": f"کاربر با ایمیل {email} یافت نشد"}
        
        # بررسی hash password
        has_password = bool(user.get("password"))
        password_hash_prefix = user.get("password", "")[:20] + "..." if has_password else "EMPTY"
        
        return {
            "found": True,
            "user_id": user.get("user_id"),
            "email": user.get("email"),
            "role": user.get("role"),
            "status": user.get("status"),
            "has_password": has_password,
            "password_hash_preview": password_hash_prefix
        }
        
    except Exception as e:
        return {"error": str(e)}


@api_router.post("/debug/reset-admin-password")
async def debug_reset_admin_password():
    """ریست رمز ادمین به admin123 (فقط برای دیباگ)"""
    try:
        new_hash = bcrypt.hashpw("admin123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        result = await db.users.update_one(
            {"user_id": "seed"},
            {"$set": {"password": new_hash}}
        )
        
        if result.modified_count > 0:
            return {"success": True, "message": "رمز ادمین به admin123 تغییر یافت"}
        else:
            return {"success": False, "message": "کاربر seed یافت نشد یا تغییری نکرد"}
            
    except Exception as e:
        return {"error": str(e)}


@api_router.post("/debug/clear-login-attempts")
async def debug_clear_login_attempts():
    """پاک کردن محدودیت تلاش‌های ورود"""
    try:
        login_attempts.clear()
        return {"success": True, "message": "✅ محدودیت‌های ورود پاک شد"}
    except Exception as e:
        return {"error": str(e)}


@api_router.get("/debug/check-admin")
async def debug_check_admin():
    """بررسی وضعیت کاربر ادمین"""
    try:
        admin = await db.users.find_one({"user_id": "seed"}, {"_id": 0})
        if not admin:
            return {"exists": False, "message": "کاربر ادمین وجود ندارد"}
        
        # تست رمز عبور
        test_password = "admin123"
        stored_hash = admin.get("password", "")
        try:
            password_valid = bcrypt.checkpw(test_password.encode('utf-8'), stored_hash.encode('utf-8'))
        except:
            password_valid = False
        
        return {
            "exists": True,
            "email": admin.get("email"),
            "role": admin.get("role"),
            "status": admin.get("status"),
            "password_is_admin123": password_valid,
            "has_password": bool(stored_hash),
            "login_attempts_blocked": len(login_attempts.get("admin@mlm.com", [])) >= MAX_LOGIN_ATTEMPTS
        }
    except Exception as e:
        return {"error": str(e)}


# Auth endpoints
@api_router.post("/auth/register")
async def register(data: UserRegister):
    """
    ثبت‌نام کاربر جدید - نسخه ساده
    """
    try:
        email = data.email.strip().lower()
        password = data.password
        
        logging.info(f"📝 Registration attempt for: {email}")
        
        # اعتبارسنجی ایمیل
        if not validate_email(email):
            raise HTTPException(status_code=400, detail="فرمت ایمیل نامعتبر است")
        
        # اعتبارسنجی رمز عبور
        if len(password) < 6:
            raise HTTPException(status_code=400, detail="رمز عبور باید حداقل 6 کاراکتر باشد")
        
        # بررسی تکراری نبودن ایمیل
        existing = await db.users.find_one({"email": email})
        if existing:
            logging.warning(f"❌ Email already exists: {email}")
            raise HTTPException(status_code=400, detail="این ایمیل قبلاً ثبت شده است")
        
        # پیدا کردن referrer
        referrer_id = "seed"
        if data.referral_code:
            referral_code_clean = data.referral_code.strip().upper()
            referrer = await db.users.find_one({"referral_code": referral_code_clean}, {"_id": 0})
            if referrer:
                referrer_id = referrer["user_id"]
                logging.info(f"✅ Referrer found: {referrer_id}")
        
        # ایجاد کاربر
        user_id = str(uuid.uuid4())
        referral_code = str(uuid.uuid4())[:8].upper()
        
        # هش رمز عبور
        password_hash = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        new_user = {
            "user_id": user_id,
            "email": email,
            "phone_number": data.phone_number if data.phone_number else None,
            "password": password_hash,
            "role": "user",
            "status": "pending",
            "referrer_id": referrer_id,
            "placement_parent_id": None,
            "direct_children": [],
            "depth": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": None,
            "referral_code": referral_code,
            "telegram_chat_id": None
        }
        
        await db.users.insert_one(new_user)
        
        logging.info(f"✅ User registered: {email} with referrer: {referrer_id}")
        
        return {
            "message": "ثبت‌نام موفقیت‌آمیز بود. منتظر فعال‌سازی باشید.",
            "user_id": user_id,
            "referral_code": referral_code
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"❌ Registration error: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطای سرور: {str(e)}")

# Login attempt tracking for brute force prevention
login_attempts = defaultdict(list)
MAX_LOGIN_ATTEMPTS = 5
LOGIN_ATTEMPT_WINDOW = 300  # 5 minutes

@api_router.post("/auth/login")
async def login(data: UserLogin):
    """
    ورود کاربر - نسخه ساده و قابل اعتماد
    """
    try:
        email = data.email.strip().lower()
        password = data.password
        
        logging.info(f"🔐 Login attempt for: {email}")
        
        # بررسی تعداد تلاش‌های ناموفق
        now = time.time()
        login_attempts[email] = [t for t in login_attempts[email] if now - t < LOGIN_ATTEMPT_WINDOW]
        
        if len(login_attempts[email]) >= MAX_LOGIN_ATTEMPTS:
            logging.warning(f"Too many login attempts for: {email}")
            raise HTTPException(
                status_code=429,
                detail="تعداد تلاش‌های ورود بیش از حد است. لطفاً 5 دقیقه صبر کنید."
            )
        
        # پیدا کردن کاربر
        user = await db.users.find_one({"email": email}, {"_id": 0})
        
        if not user:
            logging.warning(f"❌ User not found: {email}")
            login_attempts[email].append(now)
            raise HTTPException(status_code=401, detail="ایمیل یا رمز عبور اشتباه است")
        
        logging.info(f"✅ User found: {email}")
        
        # بررسی رمز عبور
        stored_hash = user.get("password", "")
        
        # چک ساده با bcrypt
        try:
            password_valid = bcrypt.checkpw(
                password.encode('utf-8'), 
                stored_hash.encode('utf-8')
            )
        except Exception as e:
            logging.error(f"❌ Password check error: {e}")
            password_valid = False
        
        if not password_valid:
            logging.warning(f"❌ Wrong password for: {email}")
            login_attempts[email].append(now)
            raise HTTPException(status_code=401, detail="ایمیل یا رمز عبور اشتباه است")
        
        logging.info(f"✅ Password correct for: {email}")
        
        # پاک کردن تلاش‌های ناموفق
        if email in login_attempts:
            del login_attempts[email]
        
        # ساخت توکن
        token = create_access_token({"user_id": user["user_id"], "role": user["role"]})
        
        logging.info(f"✅ Login successful for: {email}")
        
        return {
            "token": token,
            "user": {
                "user_id": user["user_id"],
                "email": user["email"],
                "role": user["role"],
                "status": user["status"],
                "referral_code": user.get("referral_code", "")
            }
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"❌ Login error: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطای سرور: {str(e)}")

# User endpoints
@api_router.get("/user/profile")
async def get_profile(current_user: dict = Depends(get_current_user)):
    user_id = current_user["user_id"]
    
    # بررسی cache
    cache_key = f"profile:{user_id}"
    cached_profile = await cache.get(cache_key)
    if cached_profile:
        return cached_profile
    
    # Get referrer info
    referrer = None
    if current_user.get("referrer_id"):
        ref = await db.users.find_one({"user_id": current_user["referrer_id"]}, {"_id": 0, "email": 1})
        if ref:
            referrer = ref["email"]
    
    # Count direct referrals
    direct_referrals = await db.users.count_documents({"referrer_id": user_id})
    
    # Count pending referrals
    pending_referrals = await db.users.count_documents({
        "referrer_id": user_id,
        "status": "pending"
    })
    
    # Get level_counts from cache or calculate
    level_counts = await get_level_counts_cached(user_id, current_user.get("depth", 0), current_user.get("direct_children", []))
    
    profile = {
        "user_id": user_id,
        "email": current_user["email"],
        "status": current_user["status"],
        "referral_code": current_user["referral_code"],
        "referrer": referrer,
        "total_points": current_user.get("total_points", 0.0),
        "direct_referrals": direct_referrals,
        "pending_referrals": pending_referrals,
        "level_counts": level_counts,
        "depth": current_user["depth"]
    }
    
    # ذخیره در cache (60 ثانیه)
    await cache.set(cache_key, profile, 60)
    
    return profile


async def get_level_counts_cached(user_id: str, current_depth: int, direct_children: list) -> dict:
    """محاسبه level_counts با caching"""
    cache_key = f"level_counts:{user_id}"
    
    # بررسی cache
    cached = await cache.get(cache_key)
    if cached:
        return cached
    
    level_counts = {}
    # Level 1 = self (always 1)
    level_counts["level_1"] = 1
    
    # برای هر سطح، کاربرانی را بشمار
    for level in range(2, 8):  # Levels 2-7
        target_depth = current_depth + (level - 1)
        
        if level == 2:
            # سطح 2 = فرزندان مستقیم
            count = len(direct_children)
        else:
            # روش سریع: شمارش بر اساس depth
            if current_depth == 0:
                count = await db.users.count_documents({
                    "depth": target_depth,
                    "status": "active"
                })
            else:
                # برای کاربران غیر-seed، محاسبه با BFS بهینه شده
                count = await count_descendants_at_level(user_id, level - 1)
        
        level_counts[f"level_{level}"] = count
    
    # ذخیره در cache (5 دقیقه)
    await cache.set(cache_key, level_counts, 300)
    
    return level_counts


async def count_descendants_at_level(user_id: str, target_level: int) -> int:
    """
    شمارش descendants در یک سطح خاص با BFS بهینه شده
    از batch queries استفاده می‌کند
    """
    if target_level <= 0:
        return 1
    
    current_level = [user_id]
    
    for _ in range(target_level):
        if not current_level:
            return 0
        
        # Batch fetch all users at current level
        users = await db.users.find(
            {"user_id": {"$in": current_level}},
            {"_id": 0, "direct_children": 1}
        ).to_list(len(current_level))
        
        # Collect all children
        next_level = []
        for u in users:
            next_level.extend(u.get("direct_children", []))
        
        current_level = next_level
    
    return len(current_level)


@api_router.get("/user/tree")
async def get_user_tree(current_user: dict = Depends(get_current_user)):
    """دریافت درخت کاربر - با استفاده از Materialized Path (بدون recursion)"""
    try:
        tree = await TreeStatsService.get_tree_for_display(
            current_user["user_id"],
            max_depth=7
        )
        return tree
    except Exception as e:
        logger.error(f"Error getting tree: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.get("/user/transactions")
async def get_transactions(current_user: dict = Depends(get_current_user)):
    transactions = await db.transactions.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("timestamp", -1).limit(100).to_list(100)
    
    return transactions

# Admin endpoints
@api_router.get("/admin/users")
async def get_all_users(
    page: int = 1,
    limit: int = 20,
    search: Optional[str] = None,
    search_type: Optional[str] = "email",  # email, phone, referral_code
    status_filter: Optional[str] = None,
    admin_user: dict = Depends(get_admin_user)
):
    """
    دریافت لیست کاربران با pagination و جستجو
    """
    skip = (page - 1) * limit
    query = {}
    
    # فیلتر جستجو
    if search:
        if search_type == "email":
            query["email"] = {"$regex": search, "$options": "i"}
        elif search_type == "phone":
            query["phone_number"] = {"$regex": search, "$options": "i"}
        elif search_type == "referral_code":
            query["referral_code"] = {"$regex": search, "$options": "i"}
    
    # فیلتر status
    if status_filter and status_filter != "all":
        if status_filter == "tree_complete":
            query["is_tree_complete"] = True
        else:
            query["status"] = status_filter
    
    # دریافت کاربران
    users = await db.users.find(query, {"_id": 0, "password": 0}).skip(skip).limit(limit).to_list(limit)
    
    # تعداد کل برای pagination
    total_count = await db.users.count_documents(query)
    
    return {
        "users": users,
        "total": total_count,
        "page": page,
        "limit": limit,
        "total_pages": (total_count + limit - 1) // limit
    }

@api_router.get("/admin/users/{user_id}/report")
async def get_user_report(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """گزارش کامل یک کاربر (کیف پول، تاریخچه برداشت، آمار)"""
    
    # دریافت اطلاعات کاربر
    user = await db.users.find_one({"user_id": user_id}, {"_id": 0, "password": 0})
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # تاریخچه برداشت‌ها
    withdrawals = await db.withdrawals.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("requested_at", -1).to_list(100)
    
    # محاسبه آمار برداشت
    total_withdrawn = sum(w["amount"] for w in withdrawals if w["status"] == "completed")
    pending_withdrawals = sum(w["amount"] for w in withdrawals if w["status"] == "pending_admin")
    
    # تاریخچه پرداخت‌های کریپتو
    crypto_payments = await db.crypto_payments.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("created_at", -1).to_list(10)
    
    # تاریخچه تراکنش‌های امتیاز
    transactions = await db.transactions.find(
        {"$or": [{"user_id": user_id}, {"from_user_id": user_id}]},
        {"_id": 0}
    ).sort("timestamp", -1).limit(20).to_list(20)
    
    return {
        "user": user,
        "wallet_info": {
            "address": user.get("wallet_address"),
            "network": user.get("wallet_network")
        },
        "stats": {
            "total_points": user.get("total_points", 0),
            "total_withdrawn": total_withdrawn,
            "pending_withdrawals": pending_withdrawals,
            "withdrawal_count": len([w for w in withdrawals if w["status"] == "completed"])
        },
        "withdrawals": withdrawals,
        "crypto_payments": crypto_payments,
        "recent_transactions": transactions
    }

@api_router.post("/admin/users/{user_id}/activate")
async def activate_user(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """
    فعال‌سازی کاربر توسط ادمین - Materialized Path
    
    🆕 تغییرات:
    - استفاده از UserActivationService (بدون BFS)
    - Materialized Path برای placement
    - Real-time reward calculation (بدون traversal)
    """
    try:
        logging.info(f"🚀 Starting activation for user {user_id}")
        
        # 🆕 استفاده از Service جدید
        result = await UserActivationService.activate_user(
            user_id=user_id,
            admin_user_id=admin_user["user_id"]
        )
        
        # 📊 Log کامل برای tracking
        logging.info(f"✅ User activated successfully:")
        logging.info(f"   user_id: {result['user_id']}")
        logging.info(f"   email: {result['email']}")
        logging.info(f"   placement_path: {result['placement_path']}")
        logging.info(f"   depth: {result['depth']}")
        logging.info(f"   status: {result['status']}")
        
        # ارسال پیام تلگرام اگر کاربر telegram_chat_id دارد
        try:
            user = await db.users.find_one({"user_id": user_id}, {"_id": 0, "telegram_chat_id": 1, "email": 1, "referral_code": 1})
            if user and user.get("telegram_chat_id"):
                from telegram_bot_simple import send_activation_message
                
                telegram_config = await db.config.find_one({"key": "telegram_bot_settings"}, {"_id": 0})
                
                if telegram_config:
                    import json
                    telegram_settings = json.loads(telegram_config.get("value", "{}"))
                    
                    if telegram_settings.get("enabled"):
                        bot_username = telegram_settings.get("bot_username", "")
                        webapp_url = telegram_settings.get("webapp_url", "")
                        
                        asyncio.create_task(send_activation_message(
                            telegram_user_id=user["telegram_chat_id"],
                            user_email=user["email"],
                            referral_code=user["referral_code"],
                            bot_username=bot_username,
                            webapp_url=webapp_url
                        ))
                        logger.info(f"✅ پیام فعالسازی تلگرام برای {user_id} در صف قرار گرفت")
        except Exception as e:
            logger.error(f"❌ خطا در ارسال پیام تلگرام: {e}")
        
        # پاکسازی cache
        try:
            await cache.invalidate_all_level_counts()
            await cache.invalidate_user_cache(user_id)
            referrer_id = result.get("referrer_id")
            if referrer_id:
                await cache.invalidate_user_cache(referrer_id)
        except Exception as e:
            logger.error(f"⚠️ Cache invalidation error: {e}")
        
        return {
            "success": True,
            "message": "کاربر با موفقیت فعال شد",
            "user": result
        }
        
    except ValueError as e:
        logging.error(f"❌ Validation error: {e}")
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logging.error(f"❌ Activation error: {e}")
        raise HTTPException(status_code=500, detail=f"خطا در فعال‌سازی: {str(e)}")

    # ارسال پیام تلگرام اگر کاربر telegram_chat_id دارد
    try:
        if user.get("telegram_chat_id"):
            from telegram_bot_simple import send_activation_message
            
            # دریافت تنظیمات تلگرام از دیتابیس
            telegram_config = await db.config.find_one({"key": "telegram_bot_settings"}, {"_id": 0})
            
            if telegram_config:
                import json
                telegram_settings = json.loads(telegram_config.get("value", "{}"))
                
                if telegram_settings.get("enabled"):
                    bot_username = telegram_settings.get("bot_username", "")
                    webapp_url = telegram_settings.get("webapp_url", "")
                    
                    asyncio.create_task(send_activation_message(
                        telegram_user_id=user["telegram_chat_id"],
                        user_email=user["email"],
                        referral_code=user["referral_code"],
                        bot_username=bot_username,
                        webapp_url=webapp_url
                    ))
                    logger.info(f"✅ پیام فعالسازی تلگرام برای {user_id} در صف قرار گرفت")
                else:
                    logger.info("ℹ️ ربات تلگرام غیرفعال است")
    except Exception as e:
        logger.error(f"❌ خطا در ارسال پیام تلگرام: {e}")
        # عدم ارسال تلگرام نباید مانع activation شود
    
    # پاکسازی cache
    await cache.invalidate_all_level_counts()
    await cache.invalidate_user_cache(user_id)
    if referrer_id:
        await cache.invalidate_user_cache(referrer_id)
    
    return {"message": "کاربر با موفقیت فعال شد"}

@api_router.post("/admin/users/{user_id}/deactivate")
async def deactivate_user(user_id: str, admin_user: dict = Depends(get_admin_user)):
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {"status": "pending"}}
    )
    return {"message": "کاربر غیرفعال شد"}

@api_router.post("/admin/users/{user_id}/block-rewards")
async def block_rewards(user_id: str, block_direct: bool = False, block_level: bool = False, admin_user: dict = Depends(get_admin_user)):
    update_data = {}
    if block_direct:
        update_data["blocked_direct_reward"] = True
    if block_level:
        update_data["blocked_level_reward"] = True
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": update_data}
    )
    return {"message": "پاداش‌ها مسدود شدند"}

@api_router.post("/admin/users/{user_id}/unblock-rewards")
async def unblock_rewards(user_id: str, admin_user: dict = Depends(get_admin_user)):
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {"blocked_direct_reward": False, "blocked_level_reward": False}}
    )
    return {"message": "پاداش‌ها رفع مسدودیت شدند"}

class ChangePasswordData(BaseModel):
    new_password: str

class ChangeOwnPasswordData(BaseModel):
    current_password: str
    new_password: str

# Ticket System Models
class TicketCreate(BaseModel):
    subject: str
    message: str

class TicketReply(BaseModel):
    message: str

class Ticket(BaseModel):
    ticket_id: str
    user_id: str
    subject: str
    status: str  # open, closed
    created_at: str
    updated_at: str

class TicketMessage(BaseModel):
    message_id: str
    ticket_id: str
    sender_id: str
    sender_role: str  # user, admin
    message: str
    created_at: str

@api_router.put("/user/change-password")
async def change_own_password(data: ChangeOwnPasswordData, current_user: dict = Depends(get_current_user)):
    """کاربر می‌تواند رمز عبور خود را تغییر دهد"""
    # چک کردن پسورد فعلی
    if not verify_password(data.current_password, current_user["password"]):
        raise HTTPException(status_code=400, detail="رمز عبور فعلی اشتباه است")
    
    # تغییر پسورد
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$set": {"password": hash_password(data.new_password)}}
    )
    return {"message": "رمز عبور با موفقیت تغییر یافت"}

@api_router.put("/admin/users/{user_id}/password")
async def change_user_password(user_id: str, data: ChangePasswordData, admin_user: dict = Depends(get_admin_user)):
    """ادمین می‌تواند پسورد کاربر را تغییر دهد"""
    user = await db.users.find_one({"user_id": user_id})
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {"password": hash_password(data.new_password)}}
    )
    return {"message": "رمز عبور با موفقیت تغییر یافت"}

class PointsAdjustment(BaseModel):
    amount: float
    reason: str = "تنظیم دستی توسط ادمین"

@api_router.post("/admin/users/{user_id}/adjust-points")
async def adjust_points(user_id: str, data: PointsAdjustment, admin_user: dict = Depends(get_admin_user)):
    user = await db.users.find_one({"user_id": user_id}, {"_id": 0})
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # Update points
    await db.users.update_one(
        {"user_id": user_id},
        {"$inc": {"total_points": data.amount}}
    )
    
    # Log transaction
    transaction = {
        "transaction_id": str(uuid.uuid4()),
        "user_id": user_id,
        "from_user_id": "admin",
        "reward_type": "manual_adjustment",
        "amount": data.amount,
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "description": f"تنظیم دستی: {data.reason}"
    }
    await db.transactions.insert_one(transaction)
    
    return {"message": f"امتیاز با موفقیت تنظیم شد", "new_total": user["total_points"] + data.amount}

@api_router.get("/admin/settings")
async def get_settings(admin_user: dict = Depends(get_admin_user)):
    settings = await db.settings.find_one({"setting_id": "rewards"}, {"_id": 0})
    return settings

@api_router.put("/admin/settings")
async def update_settings(data: RewardSettings, admin_user: dict = Depends(get_admin_user)):
    # دریافت مقدار قبلی seed_initial_payment
    old_settings = await db.settings.find_one({"setting_id": "rewards"})
    old_seed_payment = old_settings.get("seed_initial_payment", 100.0) if old_settings else 100.0
    
    await db.settings.update_one(
        {"setting_id": "rewards"},
        {"$set": data.model_dump()}
    )
    
    # اگر seed_initial_payment تغییر کرده، payment seed را آپدیت کن
    if data.seed_initial_payment != old_seed_payment:
        await db.payments.update_one(
            {"payment_id": "seed-initial"},
            {"$set": {
                "amount": data.seed_initial_payment,
                "description": f"سرمایه اولیه سیستم (seed) - آپدیت شده"
            }},
            upsert=True
        )
    
    return {"message": "تنظیمات به‌روزرسانی شد"}

# ==================== CRYPTO SETTINGS ====================

class CryptoSettings(BaseModel):
    # TRON
    trongrid_api_key: Optional[str] = None
    tron_deposit_address: Optional[str] = None
    tron_enabled: bool = True
    
    # BSC
    bsc_api_key: Optional[str] = None
    bsc_deposit_address: Optional[str] = None
    bsc_enabled: bool = False
    
    # Legacy (برای سازگاری با کد قدیمی)
    deposit_address: Optional[str] = None
    network: str = "USDT-TRC20"
    
    auto_check_enabled: bool = False
    min_confirmation: int = 19
    
    # تنظیمات برداشت (Withdrawal)
    min_withdrawal_amount: float = 10.0  # حداقل مبلغ برداشت (USDT)
    withdrawal_fee_type: str = "fixed"  # "fixed" یا "percentage"
    withdrawal_fee_amount: float = 0.0  # مبلغ ثابت یا درصد
    
    # Hot Wallet برای auto-transfer
    tron_hot_wallet_address: Optional[str] = None
    tron_hot_wallet_private_key: Optional[str] = None  # رمزگذاری شده
    save_tron_private_key: bool = False
    
    bsc_hot_wallet_address: Optional[str] = None
    bsc_hot_wallet_private_key: Optional[str] = None  # رمزگذاری شده
    save_bsc_private_key: bool = False

@api_router.get("/admin/settings/crypto")
async def get_crypto_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات ارز دیجیتال"""
    settings = await db.settings.find_one({"setting_id": "crypto"}, {"_id": 0})
    if not settings:
        # ایجاد تنظیمات پیش‌فرض
        default_settings = {
            "setting_id": "crypto",
            "trongrid_api_key": None,
            "deposit_address": None,
            "network": "USDT-TRC20",
            "auto_check_enabled": False,
            "min_confirmation": 19
        }
        await db.settings.insert_one(default_settings)
        return default_settings
    return settings

@api_router.put("/admin/settings/crypto")
async def update_crypto_settings(data: CryptoSettings, admin_user: dict = Depends(get_admin_user)):
    """به‌روزرسانی تنظیمات ارز دیجیتال"""
    
    # اعتبارسنجی آدرس TRON (اگر فعال است)
    if data.tron_enabled and data.tron_deposit_address:
        if not data.tron_deposit_address.startswith("T") or len(data.tron_deposit_address) != 34:
            raise HTTPException(status_code=400, detail="فرمت آدرس TRON نامعتبر است (باید با T شروع شود و 34 کاراکتر باشد)")
        
        try:
            is_valid = await tron_service.validate_tron_address(data.tron_deposit_address)
            if not is_valid:
                logging.warning(f"⚠️ TronGrid API: آدرس TRON نامعتبر است")
        except Exception as e:
            logging.warning(f"⚠️ TronGrid API غیرفعال: {str(e)}")
    
    # اعتبارسنجی آدرس BSC (اگر فعال است)
    if data.bsc_enabled and data.bsc_deposit_address:
        if not await bsc_service.validate_bsc_address(data.bsc_deposit_address):
            raise HTTPException(status_code=400, detail="فرمت آدرس BSC نامعتبر است (باید با 0x شروع شود و 42 کاراکتر باشد)")
    
    # بررسی که حداقل یک شبکه فعال باشد
    if not data.tron_enabled and not data.bsc_enabled:
        raise HTTPException(status_code=400, detail="حداقل یک شبکه باید فعال باشد")
    
    # برای سازگاری با کد قدیمی
    if data.tron_enabled and data.tron_deposit_address:
        data.deposit_address = data.tron_deposit_address
        data.network = "USDT-TRC20"
    elif data.bsc_enabled and data.bsc_deposit_address:
        data.deposit_address = data.bsc_deposit_address
        data.network = "USDT-BEP20"
    
    settings_data = {
        "setting_id": "crypto",
        **data.model_dump()
    }
    
    await db.settings.update_one(
        {"setting_id": "crypto"},
        {"$set": settings_data},
        upsert=True
    )
    
    # بستن کلاینت‌ها برای اعمال API keys جدید
    await tron_service.close()
    await bsc_service.close()
    
    logging.info(f"✅ تنظیمات ارز دیجیتال به‌روزرسانی شد - TRON: {data.tron_enabled}, BSC: {data.bsc_enabled}")
    
    return {"message": "✅ تنظیمات ارز دیجیتال به‌روزرسانی شد"}

@api_router.get("/admin/stats")
async def get_stats(admin_user: dict = Depends(get_admin_user)):
    total_users = await db.users.count_documents({})
    active_users = await db.users.count_documents({"status": "active"})
    pending_users = await db.users.count_documents({"status": "pending"})
    tree_complete_users = await db.users.count_documents({"is_tree_complete": True})
    total_transactions = await db.transactions.count_documents({})
    
    # Total points distributed
    pipeline = [
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    result = await db.transactions.aggregate(pipeline).to_list(1)
    total_points = result[0]["total"] if result else 0
    
    return {
        "total_users": total_users,
        "active_users": active_users,
        "pending_users": pending_users,
        "tree_complete_users": tree_complete_users,
        "total_transactions": total_transactions,
        "total_points_distributed": total_points
    }

# Test system endpoints
@api_router.post("/admin/test/create-users")
async def create_test_users(data: TestUserCreate, admin_user: dict = Depends(get_admin_user)):
    """
    ایجاد کاربران تستی - با Materialized Path
    """
    created_users = []
    
    # پیدا کردن referrer
    referrer_id = "seed"
    if data.referral_code:
        referrer = await db.users.find_one({"referral_code": data.referral_code}, {"_id": 0})
        if referrer:
            referrer_id = referrer["user_id"]
        else:
            raise HTTPException(status_code=404, detail=f"کد معرف {data.referral_code} یافت نشد")
    
    for i in range(data.count):
        user_id = str(uuid.uuid4())
        referral_code = str(uuid.uuid4())[:8].upper()
        email = f"test_{user_id[:8]}@test.com"
        password_hash = hash_password("test123")
        
        # Register user
        new_user = {
            "user_id": user_id,
            "email": email,
            "password": password_hash,
            "role": "user",
            "status": "pending",
            "referrer_id": referrer_id,
            "referral_code": referral_code,
            "phone_number": None,
            "placement_path": None,
            "placement_parent_id": None,
            "placement_position": None,
            "depth": None,
            "ancestor_depths": [],
            "direct_children": [],
            "direct_children_count": 0,
            "total_descendants": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": None,
            "telegram_chat_id": None,
            "wallet_address": None,
            "wallet_network": None
        }
        
        await db.users.insert_one(new_user)
        
        # Auto activate با UserActivationService
        if data.auto_activate:
            try:
                result = await UserActivationService.activate_user(user_id, admin_user["user_id"])
                created_users.append({
                    "user_id": user_id,
                    "email": email,
                    "referral_code": referral_code,
                    "status": "active",
                    "placement_path": result.get("placement_path"),
                    "depth": result.get("depth")
                })
            except Exception as e:
                logging.error(f"Failed to activate test user {user_id}: {e}")
                created_users.append({
                    "user_id": user_id,
                    "email": email,
                    "referral_code": referral_code,
                    "status": "pending",
                    "error": str(e)
                })
        else:
            created_users.append({
                "user_id": user_id,
                "email": email,
                "referral_code": referral_code,
                "status": "pending",
            "depth": new_depth
        })
    
    return {
        "message": f"{data.count} کاربر تست ایجاد شد با کد معرف {data.referral_code or 'SEED'}",
        "users": created_users
    }

@api_router.delete("/admin/test/clear-test-users")
async def clear_test_users(admin_user: dict = Depends(get_admin_user)):
    result = await db.users.delete_many({"email": {"$regex": "^test_.*@test\.com$"}})
    await db.transactions.delete_many({})
    
    return {"message": f"{result.deleted_count} کاربر تست حذف شد"}

@api_router.delete("/admin/users/clear-all")
async def clear_all_users(admin_user: dict = Depends(get_admin_user)):
    """حذف کامل همه داده‌ها به جز ادمین و تنظیمات"""
    # حذف همه کاربران به جز seed و admin
    result = await db.users.delete_many({
        "user_id": {"$nin": ["seed"]},
        "role": {"$ne": "admin"}
    })
    
    # حذف تمام تراکنش‌ها، پرداخت‌ها، crypto payments، withdrawals
    trans_result = await db.transactions.delete_many({})
    pay_result = await db.payments.delete_many({})
    crypto_result = await db.crypto_payments.delete_many({})
    withdrawal_result = await db.withdrawals.delete_many({})
    
    # حذف تیکت‌ها و اعلانات (اگر وجود دارد)
    await db.tickets.delete_many({})
    await db.announcements.delete_many({})
    
    # ریست کردن direct_children و امتیاز seed
    await db.users.update_one(
        {"user_id": "seed"},
        {"$set": {
            "direct_children": [],
            "total_points": 0.0,
            "status": "active",
            "is_tree_complete": False
        }}
    )
    
    logging.info(f"🧹 پاکسازی کامل انجام شد: {result.deleted_count} کاربر، {trans_result.deleted_count} تراکنش، {crypto_result.deleted_count} پرداخت کریپتو، {withdrawal_result.deleted_count} برداشت")
    
    return {
        "message": "✅ سیستم کاملاً پاک شد",
        "deleted": {
            "users": result.deleted_count,
            "transactions": trans_result.deleted_count,
            "payments": pay_result.deleted_count,
            "crypto_payments": crypto_result.deleted_count,
            "withdrawals": withdrawal_result.deleted_count
        }
    }

@api_router.post("/admin/test/fill-tree")
async def fill_tree_test(referral_code: str, levels: int = 3, admin_user: dict = Depends(get_admin_user)):
    """Fill tree under a user for testing - با رعایت کامل قوانین هرم"""
    
    # پیدا کردن کاربر بر اساس کد معرف
    parent_user = await db.users.find_one({"referral_code": referral_code}, {"_id": 0})
    if not parent_user:
        raise HTTPException(status_code=404, detail=f"کاربر با کد معرف {referral_code} یافت نشد")
    
    parent_id = parent_user["user_id"]
    
    # دریافت تنظیمات یکبار در ابتدا
    settings = await db.settings.find_one({"setting_id": "rewards"})
    activation_fee = settings.get("activation_fee", 100.0) if settings else 100.0
    
    async def create_children(current_parent_id: str, remaining_levels: int):
        """
        🔒 ایجاد فرزندان تستی با استفاده از Atomic Operations
        این تابع از find_one_and_update برای جلوگیری از race condition استفاده می‌کند
        """
        if remaining_levels <= 0:
            return
        
        from collections import deque
        
        # BFS برای پر کردن سطح به سطح
        queue = deque([current_parent_id])
        current_level = 0
        
        while queue and current_level < remaining_levels:
            level_size = len(queue)
            current_level += 1
            next_level_queue = []
            
            for _ in range(level_size):
                p_id = queue.popleft()
                
                # خواندن تازه والد از دیتابیس
                parent = await db.users.find_one({"user_id": p_id}, {"_id": 0})
                if not parent:
                    continue
                
                current_children = parent.get("direct_children", [])
                slots_available = 3 - len(current_children)
                parent_depth = parent.get("depth", 0)
                
                # ایجاد فرزندان برای این والد با atomic operation
                for slot in range(slots_available):
                    test_user_id = str(uuid.uuid4())
                    email = f"test_{test_user_id[:8]}@test.com"
                    referral_code_new = str(uuid.uuid4())[:8].upper()
                    new_depth = parent_depth + 1
                    
                    # 🔒 Atomic: ابتدا slot را reserve کن
                    reserve_result = await db.users.find_one_and_update(
                        {
                            "user_id": p_id,
                            "$expr": {"$lt": [{"$size": {"$ifNull": ["$direct_children", []]}}, 3]}
                        },
                        {"$push": {"direct_children": test_user_id}},
                        return_document=True
                    )
                    
                    if not reserve_result:
                        # slot پر شده، به کاربر بعدی برو
                        break
                    
                    new_user = {
                        "user_id": test_user_id,
                        "email": email,
                        "password": hash_password("test123"),
                        "role": "user",
                        "status": "active",
                        "referrer_id": p_id,
                        "placement_parent_id": p_id,
                        "direct_children": [],
                        "depth": new_depth,
                        "total_points": 0.0,
                        "blocked_direct_reward": False,
                        "blocked_level_reward": False,
                        "is_tree_complete": False,
                        "created_at": datetime.now(timezone.utc).isoformat(),
                        "activated_at": datetime.now(timezone.utc).isoformat(),
                        "referral_code": referral_code_new
                    }
                    
                    # ثبت پرداخت
                    payment = {
                        "payment_id": str(uuid.uuid4()),
                        "user_id": test_user_id,
                        "amount": activation_fee,
                        "timestamp": datetime.now(timezone.utc).isoformat(),
                        "description": f"پرداخت تستی - {email}"
                    }
                    await db.payments.insert_one(payment)
                    
                    await db.users.insert_one(new_user)
                    
                    await distribute_rewards(test_user_id, p_id, p_id)
                    
                    # اضافه به queue برای سطح بعد
                    if current_level < remaining_levels:
                        next_level_queue.append(test_user_id)
                
                # اضافه کردن فرزندان موجود به queue برای ادامه
                for child_id in current_children:
                    if current_level < remaining_levels:
                        next_level_queue.append(child_id)
            
            # آماده کردن queue برای سطح بعدی
            queue.extend(next_level_queue)
    
    await create_children(parent_id, levels)
    
    return {"message": f"درخت تا {levels} سطح با رعایت قوانین هرم (حداکثر 3 شاخه) پر شد"}

# ==================== TICKET SYSTEM ====================

@api_router.post("/user/tickets")
async def create_ticket(data: TicketCreate, current_user: dict = Depends(get_current_user)):
    """ایجاد تیکت جدید توسط کاربر"""
    ticket_id = str(uuid.uuid4())
    message_id = str(uuid.uuid4())
    
    ticket = {
        "ticket_id": ticket_id,
        "user_id": current_user["user_id"],
        "user_email": current_user["email"],
        "subject": data.subject,
        "status": "open",
        "created_at": datetime.now(timezone.utc).isoformat(),
        "updated_at": datetime.now(timezone.utc).isoformat()
    }
    
    message = {
        "message_id": message_id,
        "ticket_id": ticket_id,
        "sender_id": current_user["user_id"],
        "sender_role": "user",
        "message": data.message,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.tickets.insert_one(ticket)
    await db.ticket_messages.insert_one(message)
    
    return {"message": "تیکت با موفقیت ایجاد شد", "ticket_id": ticket_id}

@api_router.get("/user/tickets")
async def get_my_tickets(current_user: dict = Depends(get_current_user)):
    """دریافت تیکت‌های کاربر"""
    tickets = await db.tickets.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("created_at", -1).to_list(100)
    return tickets

@api_router.get("/user/tickets/{ticket_id}/messages")
async def get_ticket_messages(ticket_id: str, current_user: dict = Depends(get_current_user)):
    """دریافت پیام‌های یک تیکت"""
    ticket = await db.tickets.find_one({"ticket_id": ticket_id, "user_id": current_user["user_id"]})
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    messages = await db.ticket_messages.find(
        {"ticket_id": ticket_id},
        {"_id": 0}
    ).sort("created_at", 1).to_list(100)
    
    return {"ticket": ticket, "messages": messages}

@api_router.post("/user/tickets/{ticket_id}/reply")
async def reply_to_ticket(ticket_id: str, data: TicketReply, current_user: dict = Depends(get_current_user)):
    """پاسخ کاربر به تیکت"""
    ticket = await db.tickets.find_one({"ticket_id": ticket_id, "user_id": current_user["user_id"]})
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    message_id = str(uuid.uuid4())
    message = {
        "message_id": message_id,
        "ticket_id": ticket_id,
        "sender_id": current_user["user_id"],
        "sender_role": "user",
        "message": data.message,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.ticket_messages.insert_one(message)
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {"$set": {"updated_at": datetime.now(timezone.utc).isoformat()}}
    )
    
    return {"message": "پاسخ ارسال شد"}

# ==================== ADMIN TICKET MANAGEMENT ====================

@api_router.get("/admin/tickets")
async def get_all_tickets(status: Optional[str] = None, admin_user: dict = Depends(get_admin_user)):
    """دریافت همه تیکت‌ها"""
    query = {}
    if status:
        query["status"] = status
    
    tickets = await db.tickets.find(query, {"_id": 0}).sort("updated_at", -1).to_list(1000)
    return tickets

@api_router.post("/admin/tickets/{ticket_id}/reply")
async def admin_reply_ticket(ticket_id: str, data: TicketReply, admin_user: dict = Depends(get_admin_user)):
    """پاسخ ادمین به تیکت"""
    ticket = await db.tickets.find_one({"ticket_id": ticket_id})
    if not ticket:
        raise HTTPException(status_code=404, detail="تیکت یافت نشد")
    
    message_id = str(uuid.uuid4())
    message = {
        "message_id": message_id,
        "ticket_id": ticket_id,
        "sender_id": admin_user["user_id"],
        "sender_role": "admin",
        "message": data.message,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.ticket_messages.insert_one(message)
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {"$set": {"updated_at": datetime.now(timezone.utc).isoformat()}}
    )
    
    return {"message": "پاسخ ارسال شد"}

@api_router.put("/admin/tickets/{ticket_id}/close")
async def close_ticket(ticket_id: str, admin_user: dict = Depends(get_admin_user)):
    """بستن تیکت"""
    await db.tickets.update_one(
        {"ticket_id": ticket_id},
        {"$set": {"status": "closed", "updated_at": datetime.now(timezone.utc).isoformat()}}
    )
    return {"message": "تیکت بسته شد"}

# ==================== CRYPTO PAYMENT SYSTEM ====================

async def get_enabled_networks():
    """دریافت لیست شبکه‌های فعال"""
    settings = await db.settings.find_one({"setting_id": "crypto"})
    if not settings:
        return []
    
    networks = []
    
    # TRON
    if settings.get("tron_enabled", True) and settings.get("tron_deposit_address"):
        networks.append({
            "network": "TRON",
            "code": "USDT-TRC20",
            "address": settings["tron_deposit_address"],
            "contract": USDT_CONTRACTS["TRON"]
        })
    
    # BSC
    if settings.get("bsc_enabled", False) and settings.get("bsc_deposit_address"):
        networks.append({
            "network": "BSC",
            "code": "USDT-BEP20",
            "address": settings["bsc_deposit_address"],
            "contract": USDT_CONTRACTS["BSC"]
        })
    
    return networks

class CryptoPaymentRequest(BaseModel):
    amount: Optional[float] = None  # مبلغ از settings خوانده می‌شود
    network: Optional[str] = None  # TRON or BSC

@api_router.post("/user/crypto-payment/create")
async def create_crypto_payment(data: CryptoPaymentRequest, current_user: dict = Depends(get_current_user)):
    """ایجاد درخواست پرداخت ارز دیجیتال"""
    # دریافت شبکه‌های فعال
    enabled_networks = await get_enabled_networks()
    if not enabled_networks:
        raise HTTPException(status_code=500, detail="هیچ شبکه‌ای فعال نیست. لطفاً با مدیر تماس بگیرید.")
    
    # اگر کاربر شبکه مشخص کرده
    selected_network = None
    if data.network:
        selected_network = next((n for n in enabled_networks if n["network"] == data.network), None)
        if not selected_network:
            raise HTTPException(status_code=400, detail="شبکه انتخابی فعال نیست")
    else:
        # اولین شبکه فعال
        selected_network = enabled_networks[0]
    
    # دریافت تنظیمات rewards برای activation_fee
    reward_settings = await db.settings.find_one({"setting_id": "rewards"})
    if not reward_settings:
        raise HTTPException(status_code=500, detail="تنظیمات سیستم یافت نشد")
    
    current_address = selected_network["address"]
    current_network = selected_network["code"]
    current_blockchain = selected_network["network"]
    current_amount = reward_settings.get("activation_fee", 100.0)
    
    # بررسی اگر قبلاً pending payment دارد
    existing = await db.crypto_payments.find_one({
        "user_id": current_user["user_id"],
        "status": "pending"
    })
    
    if existing:
        # چک کردن آیا آدرس، مبلغ یا شبکه تغییر کرده
        needs_update = False
        update_fields = {}
        
        if existing.get("crypto_address") != current_address:
            logging.info(f"🔄 Address changed for payment {existing['payment_id']}")
            update_fields["crypto_address"] = current_address
            update_fields["network"] = current_network
            update_fields["blockchain"] = current_blockchain
            needs_update = True
        
        if existing.get("amount") != current_amount:
            logging.info(f"🔄 Amount changed for payment {existing['payment_id']}: {existing.get('amount')} → {current_amount}")
            update_fields["amount"] = current_amount
            # اگر مبلغ تغییر کرد، unique_amount را دوباره محاسبه کن
            new_unique_amount = generate_unique_amount(current_amount, existing["payment_id"])
            update_fields["unique_amount"] = new_unique_amount
            needs_update = True
        
        # اگر unique_amount وجود ندارد (برای payment های قدیمی)، اضافه کن
        if "unique_amount" not in existing:
            unique_amount = generate_unique_amount(existing.get("amount", current_amount), existing["payment_id"])
            update_fields["unique_amount"] = unique_amount
            needs_update = True
        
        if needs_update:
            update_fields["updated_at"] = datetime.now(timezone.utc).isoformat()
            await db.crypto_payments.update_one(
                {"payment_id": existing["payment_id"]},
                {"$set": update_fields}
            )
        
        # دریافت unique_amount (یا از update یا از existing)
        final_unique_amount = update_fields.get("unique_amount", existing.get("unique_amount"))
        
        return {
            "payment_id": existing["payment_id"],
            "crypto_address": current_address,
            "amount": current_amount,
            "unique_amount": final_unique_amount,
            "network": current_network,
            "blockchain": current_blockchain,
            "enabled_networks": enabled_networks
        }
    
    # ایجاد payment جدید
    payment_id = str(uuid.uuid4())
    
    # تولید مبلغ unique برای این payment
    unique_amount = generate_unique_amount(current_amount, payment_id)
    
    crypto_payment = {
        "payment_id": payment_id,
        "user_id": current_user["user_id"],
        "user_email": current_user["email"],
        "amount": current_amount,  # مبلغ اصلی
        "unique_amount": unique_amount,  # مبلغ unique که کاربر باید دقیقاً واریز کند
        "crypto_address": current_address,
        "network": current_network,
        "blockchain": current_blockchain,  # TRON or BSC
        "status": "pending",
        "transaction_hash": None,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "expires_at": (datetime.now(timezone.utc) + timedelta(hours=24)).isoformat()
    }
    
    await db.crypto_payments.insert_one(crypto_payment)
    
    return {
        "payment_id": payment_id,
        "crypto_address": crypto_payment["crypto_address"],
        "amount": crypto_payment["amount"],
        "unique_amount": unique_amount,  # ارسال به frontend
        "network": crypto_payment["network"],
        "blockchain": crypto_payment["blockchain"],
        "expires_at": crypto_payment["expires_at"],
        "enabled_networks": enabled_networks
    }

@api_router.get("/user/crypto-payment/networks")
async def get_available_networks():
    """دریافت لیست شبکه‌های فعال برای پرداخت و برداشت"""
    networks = await get_enabled_networks()
    
    # فرمت مناسب برای UI
    formatted_networks = []
    for net in networks:
        formatted_networks.append({
            "value": net["code"],  # USDT-TRC20 or USDT-BEP20
            "label": f"{net['code']} ({net['network']})",
            "name": net["network"],
            "address": net["address"]
        })
    
    return formatted_networks

@api_router.get("/user/withdrawal-settings")
async def get_withdrawal_settings():
    """دریافت تنظیمات برداشت برای کاربر"""
    settings = await db.settings.find_one({"setting_id": "crypto"})
    
    return {
        "min_withdrawal_amount": settings.get("min_withdrawal_amount", 10.0) if settings else 10.0,
        "withdrawal_fee_type": settings.get("withdrawal_fee_type", "fixed") if settings else "fixed",
        "withdrawal_fee_amount": settings.get("withdrawal_fee_amount", 0.0) if settings else 0.0
    }

@api_router.get("/user/crypto-payment/status")
async def get_payment_status(current_user: dict = Depends(get_current_user)):
    """وضعیت پرداخت کاربر"""
    payment = await db.crypto_payments.find_one(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    )
    
    if not payment:
        return {"status": "none"}
    
    # اگر payment در وضعیت pending است، تنظیمات جدید را بگیریم
    if payment.get("status") == "pending":
        crypto_settings = await db.settings.find_one({"setting_id": "crypto"})
        reward_settings = await db.settings.find_one({"setting_id": "rewards"})
        
        if crypto_settings and reward_settings:
            # تشخیص blockchain از payment
            blockchain = payment.get("blockchain", "TRON")
            
            # دریافت آدرس و شبکه بر اساس blockchain
            if blockchain == "TRON":
                new_address = crypto_settings.get("tron_deposit_address") or crypto_settings.get("deposit_address")
                new_network = "USDT-TRC20"
            elif blockchain == "BSC":
                new_address = crypto_settings.get("bsc_deposit_address")
                new_network = "USDT-BEP20"
            else:
                new_address = None
                new_network = None
            
            new_amount = reward_settings.get("activation_fee", 100.0)
            
            # اگر تنظیمات تغییر کرده، payment را update کنیم
            needs_update = False
            update_fields = {}
            
            if new_address and payment.get("crypto_address") != new_address:
                logging.info(f"🔄 Updating payment {payment['payment_id']} with new address: {new_address}")
                update_fields["crypto_address"] = new_address
                update_fields["network"] = new_network
                payment["crypto_address"] = new_address
                payment["network"] = new_network
                needs_update = True
            
            if payment.get("amount") != new_amount:
                logging.info(f"🔄 Updating payment {payment['payment_id']} with new amount: {payment.get('amount')} → {new_amount}")
                update_fields["amount"] = new_amount
                payment["amount"] = new_amount
                needs_update = True
            
            if needs_update:
                update_fields["updated_at"] = datetime.now(timezone.utc).isoformat()
                await db.crypto_payments.update_one(
                    {"payment_id": payment["payment_id"]},
                    {"$set": update_fields}
                )
    
    return payment

@api_router.post("/user/crypto-payment/check-my-transaction")
async def check_my_transaction(current_user: dict = Depends(get_current_user)):
    """بررسی تراکنش من توسط خود کاربر"""
    try:
        # پیدا کردن payment این کاربر
        payment = await db.crypto_payments.find_one({
            "user_id": current_user["user_id"],
            "status": "pending"
        })
        
        if not payment:
            return {
                "success": False,
                "message": "درخواست پرداختی یافت نشد",
                "status": "none"
            }
        
        # دریافت تنظیمات
        settings = await db.settings.find_one({"setting_id": "crypto"})
        if not settings or not settings.get("deposit_address"):
            return {
                "success": False,
                "message": "تنظیمات سیستم ناقص است",
                "status": "error"
            }
        
        expected_amount = payment["amount"]
        blockchain = payment.get("blockchain", "TRON")  # Default TRON برای سازگاری
        
        # تشخیص شبکه و دریافت آدرس
        if blockchain == "TRON":
            deposit_address = settings.get("tron_deposit_address") or settings.get("deposit_address")
        elif blockchain == "BSC":
            deposit_address = settings.get("bsc_deposit_address")
        else:
            return {
                "success": False,
                "message": "شبکه نامعتبر",
                "status": "error"
            }
        
        if not deposit_address:
            return {
                "success": False,
                "message": f"آدرس دریافت برای شبکه {blockchain} تنظیم نشده است",
                "status": "error"
            }
        
        # دریافت تراکنش‌ها بسته به شبکه
        try:
            usdt_transactions = []
            
            if blockchain == "TRON":
                transactions = await tron_service.get_trc20_transactions(
                    address=deposit_address,
                    limit=50,
                    only_to=True
                )
                usdt_transactions = await tron_service.filter_usdt_transactions(transactions)
                
            elif blockchain == "BSC":
                usdt_transactions = await bsc_service.get_bep20_transactions(
                    address=deposit_address,
                    contract_address=USDT_CONTRACTS["BSC"],
                    limit=50
                )
            
            logging.info(f"🔍 Checking {len(usdt_transactions)} {blockchain} USDT transactions for user {current_user['email']}")
            
            # جستجو برای تراکنش مطابق
            for tx in usdt_transactions:
                if blockchain == "TRON":
                    tx_amount = await tron_service.parse_transaction_amount(tx)
                    tx_hash = tx.get("transaction_id")
                    tx_to = tx.get("to")
                else:  # BSC
                    tx_amount = await bsc_service.parse_transaction_amount(tx)
                    tx_hash = tx.get("hash")
                    tx_to = tx.get("to")
                
                # چک کردن آیا تراکنش مطابقت دارد
                if (
                    abs(tx_amount - expected_amount) < 0.01 and  # تلرانس 0.01 USDT
                    tx_to.lower() == deposit_address.lower()
                ):
                    logging.info(f"✅ Found matching {blockchain} transaction: {tx_hash} - {tx_amount} USDT")
                    
                    # تایید پرداخت
                    await db.crypto_payments.update_one(
                        {"payment_id": payment["payment_id"]},
                        {"$set": {
                            "status": "confirmed",
                            "transaction_hash": tx_hash,
                            "confirmed_at": datetime.now(timezone.utc).isoformat()
                        }}
                    )
                    
                    # فعال‌سازی کاربر
                    user = await db.users.find_one({"user_id": current_user["user_id"]})
                    if user and user["status"] == "pending":
                        # ثبت در جدول payments
                        payment_record = {
                            "payment_id": str(uuid.uuid4()),
                            "user_id": current_user["user_id"],
                            "amount": payment["amount"],
                            "timestamp": datetime.now(timezone.utc).isoformat(),
                            "description": f"پرداخت خودکار ارز دیجیتال - {current_user['email']}"
                        }
                        await db.payments.insert_one(payment_record)
                        
                        # فعال‌سازی کاربر با UserActivationService
                        try:
                            activation_result = await UserActivationService.activate_user(
                                current_user["user_id"],
                                "system_auto"
                            )
                            logging.info(f"✅ User {current_user['email']} activated successfully!")
                        except Exception as e:
                            logging.error(f"❌ Auto-activation failed: {e}")
                            # Update status anyway
                            await db.users.update_one(
                                {"user_id": current_user["user_id"]},
                                {"$set": {"status": "pending"}}
                            )
                        
                        return {
                            "success": True,
                            "message": "🎉 پرداخت شما تایید و حساب شما فعال شد!",
                            "status": "confirmed",
                            "transaction_hash": tx_hash
                        }
                    else:
                        return {
                            "success": True,
                            "message": "پرداخت تایید شد",
                            "status": "confirmed"
                        }
            
            # اگر تراکنشی پیدا نشد
            logging.info(f"❌ No matching transaction found for user {current_user['email']}")
            return {
                "success": False,
                "message": "تراکنش شما هنوز در بلاکچین تایید نشده است. لطفاً چند دقیقه صبر کنید و دوباره تلاش کنید.",
                "status": "pending",
                "checked_transactions": len(usdt_transactions)
            }
            
        except Exception as e:
            logging.error(f"❌ Error checking blockchain: {str(e)}")
            return {
                "success": False,
                "message": f"خطا در بررسی تراکنش: {str(e)}",
                "status": "error"
            }
            
    except Exception as e:
        logging.error(f"❌ Error in check_my_transaction: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطا در بررسی تراکنش: {str(e)}")

@api_router.post("/admin/crypto-payment/{payment_id}/confirm")
async def confirm_crypto_payment(payment_id: str, transaction_hash: str, admin_user: dict = Depends(get_admin_user)):
    """
    تایید دستی پرداخت توسط ادمین با محافظت در برابر double activation
    
    🔒 محافظت‌های امنیتی:
    1. چک وضعیت پرداخت قبل از تایید
    2. Atomic status update برای جلوگیری از double confirm
    3. چک وضعیت کاربر قبل از فعال‌سازی
    """
    payment = await db.crypto_payments.find_one({"payment_id": payment_id})
    if not payment:
        raise HTTPException(status_code=404, detail="پرداخت یافت نشد")
    
    # 🔒 FIX: چک وضعیت پرداخت - فقط pending قابل تایید است
    if payment.get("status") in ["confirmed", "completed"]:
        raise HTTPException(status_code=400, detail="این پرداخت قبلاً تایید شده است")
    
    # 🔒 Atomic status update - فقط اگر هنوز pending باشد
    update_result = await db.crypto_payments.update_one(
        {"payment_id": payment_id, "status": "pending"},
        {"$set": {
            "status": "confirmed",
            "transaction_hash": transaction_hash,
            "confirmed_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    # اگر update موفق نبود، یعنی وضعیت تغییر کرده
    if update_result.modified_count == 0:
        raise HTTPException(status_code=400, detail="وضعیت پرداخت تغییر کرده است. لطفاً صفحه را رفرش کنید.")
    
    # فعال‌سازی کاربر
    user = await db.users.find_one({"user_id": payment["user_id"]})
    
    # 🔒 FIX: فقط کاربران pending فعال می‌شوند
    if not user:
        logging.warning(f"⚠️ کاربر {payment['user_id']} یافت نشد")
        return {"message": "پرداخت تایید شد اما کاربر یافت نشد"}
    
    if user["status"] != "pending":
        logging.info(f"ℹ️ کاربر {payment['user_id']} قبلاً فعال شده است (status: {user['status']})")
        return {"message": "پرداخت تایید شد. کاربر قبلاً فعال شده بود."}
    
    # ثبت در جدول payments
    payment_record = {
        "payment_id": str(uuid.uuid4()),
        "user_id": payment["user_id"],
        "amount": payment["amount"],
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "description": f"پرداخت ارز دیجیتال - {payment['user_email']}"
    }
    await db.payments.insert_one(payment_record)
    
    # فعال‌سازی کاربر
    referrer_id = user.get("referrer_id")
    if referrer_id:
        placement_parent_id = await find_placement_parent(referrer_id)
    else:
        placement_parent_id = "seed"
    
    parent = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
    parent_depth = parent.get("depth", 0) if parent else 0
    new_depth = parent_depth + 1
    
    # 🔒 Atomic activation - فقط اگر هنوز pending باشد
    activation_result = await db.users.update_one(
        {"user_id": payment["user_id"], "status": "pending"},
        {"$set": {
            "status": "active",
            "placement_parent_id": placement_parent_id,
            "depth": new_depth,
            "activated_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    # اگر activation موفق نبود، کاربر قبلاً فعال شده
    if activation_result.modified_count == 0:
        logging.info(f"ℹ️ کاربر {payment['user_id']} قبلاً توسط عملیات دیگری فعال شده است")
        return {"message": "پرداخت تایید شد. کاربر توسط عملیات دیگری فعال شده بود."}
    
    result = await db.users.update_one(
        {
            "user_id": placement_parent_id,
            "$expr": {"$lt": [{"$size": "$direct_children"}, 3]}
        },
        {"$push": {"direct_children": payment["user_id"]}}
    )
    
    if result.modified_count == 0:
        placement_parent_id = await find_placement_parent(referrer_id)
        parent = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
        parent_depth = parent.get("depth", 0) if parent else 0
        new_depth = parent_depth + 1
        
        await db.users.update_one(
            {"user_id": payment["user_id"]},
            {"$set": {"placement_parent_id": placement_parent_id, "depth": new_depth}}
        )
        
        await db.users.update_one(
            {
                "user_id": placement_parent_id,
                "$expr": {"$lt": [{"$size": "$direct_children"}, 3]}
            },
            {"$push": {"direct_children": payment["user_id"]}}
        )
        
        await distribute_rewards(payment["user_id"], referrer_id, placement_parent_id)
        
        current = parent
        while current:
            await check_tree_completion(current["user_id"])
            if current.get("placement_parent_id"):
                current = await db.users.find_one({"user_id": current["placement_parent_id"]}, {"_id": 0})
            else:
                break
    
    return {"message": "پرداخت تایید و کاربر فعال شد"}

@api_router.get("/admin/crypto-payments")
async def get_all_crypto_payments(status: Optional[str] = None, admin_user: dict = Depends(get_admin_user)):
    """لیست پرداخت‌های ارز دیجیتال"""
    query = {}
    if status:
        query["status"] = status
    
    payments = await db.crypto_payments.find(query, {"_id": 0}).sort("created_at", -1).limit(100).to_list(100)
    return payments

@api_router.post("/admin/crypto-payments/check-transactions")
async def check_pending_payments(admin_user: dict = Depends(get_admin_user)):
    """بررسی خودکار تراکنش‌های pending"""
    try:
        # دریافت تنظیمات
        settings = await db.settings.find_one({"setting_id": "crypto"})
        if not settings or not settings.get("deposit_address"):
            raise HTTPException(status_code=500, detail="آدرس دریافت تنظیم نشده است")
        
        deposit_address = settings["deposit_address"]
        
        # دریافت تراکنش‌های TRC20 از بلاکچین
        transactions = await tron_service.get_trc20_transactions(
            address=deposit_address,
            limit=50,
            only_to=True
        )
        
        # فیلتر کردن تراکنش‌های USDT
        usdt_transactions = await tron_service.filter_usdt_transactions(transactions)
        
        # دریافت pending payments
        pending_payments = await db.crypto_payments.find(
            {"status": "pending"},
            {"_id": 0}
        ).to_list(100)
        
        confirmed_count = 0
        
        # مقایسه و تایید خودکار
        for payment in pending_payments:
            # استفاده از unique_amount اگر وجود دارد، در غیر این صورت amount معمولی
            expected_amount = payment.get("unique_amount", payment["amount"])
            
            for tx in usdt_transactions:
                tx_amount = await tron_service.parse_transaction_amount(tx)
                tx_hash = tx.get("transaction_id")
                tx_to = tx.get("to")
                
                # چک کردن آیا تراکنش مطابقت دارد
                # اگر unique_amount وجود دارد، باید دقیقاً مطابقت کند (تلرانس 0.0001 برای 4 رقم اعشار)
                # در غیر این صورت، تلرانس 0.01 مثل قبل
                tolerance = 0.0001 if "unique_amount" in payment else 0.01
                
                if (
                    abs(tx_amount - expected_amount) < tolerance and
                    tx_to == deposit_address
                ):
                    # تایید پرداخت
                    await db.crypto_payments.update_one(
                        {"payment_id": payment["payment_id"]},
                        {"$set": {
                            "status": "confirmed",
                            "transaction_hash": tx_hash,
                            "confirmed_at": datetime.now(timezone.utc).isoformat()
                        }}
                    )
                    
                    # فعال‌سازی کاربر
                    user = await db.users.find_one({"user_id": payment["user_id"]})
                    if user and user["status"] == "pending":
                        # ثبت در جدول payments
                        payment_record = {
                            "payment_id": str(uuid.uuid4()),
                            "user_id": payment["user_id"],
                            "amount": payment["amount"],
                            "timestamp": datetime.now(timezone.utc).isoformat(),
                            "description": f"پرداخت خودکار ارز دیجیتال - {payment['user_email']}"
                        }
                        await db.payments.insert_one(payment_record)
                        
                        # فعال‌سازی کاربر
                        referrer_id = user.get("referrer_id")
                        if referrer_id:
                            placement_parent_id = await find_placement_parent(referrer_id)
                        else:
                            placement_parent_id = "seed"
                        
                        parent = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
                        parent_depth = parent.get("depth", 0) if parent else 0
                        new_depth = parent_depth + 1
                        
                        await db.users.update_one(
                            {"user_id": payment["user_id"]},
                            {"$set": {
                                "status": "active",
                                "placement_parent_id": placement_parent_id,
                                "depth": new_depth,
                                "activated_at": datetime.now(timezone.utc).isoformat()
                            }}
                        )
                        
                        result = await db.users.update_one(
                            {
                                "user_id": placement_parent_id,
                                "$expr": {"$lt": [{"$size": "$direct_children"}, 3]}
                            },
                            {"$push": {"direct_children": payment["user_id"]}}
                        )
                        
                        if result.modified_count == 0:
                            placement_parent_id = await find_placement_parent(referrer_id)
                            parent = await db.users.find_one({"user_id": placement_parent_id}, {"_id": 0})
                            parent_depth = parent.get("depth", 0) if parent else 0
                            new_depth = parent_depth + 1
                            
                            await db.users.update_one(
                                {"user_id": payment["user_id"]},
                                {"$set": {"placement_parent_id": placement_parent_id, "depth": new_depth}}
                            )
                            
                            await db.users.update_one(
                                {
                                    "user_id": placement_parent_id,
                                    "$expr": {"$lt": [{"$size": "$direct_children"}, 3]}
                                },
                                {"$push": {"direct_children": payment["user_id"]}}
                            )
                        
                        await distribute_rewards(payment["user_id"], referrer_id, placement_parent_id)
                        
                        current = parent
                        while current:
                            await check_tree_completion(current["user_id"])
                            if current.get("placement_parent_id"):
                                current = await db.users.find_one({"user_id": current["placement_parent_id"]}, {"_id": 0})
                            else:
                                break
                    
                    confirmed_count += 1
                    break
        
        return {
            "message": "بررسی تراکنش‌ها انجام شد",
            "checked_transactions": len(usdt_transactions),
            "confirmed_payments": confirmed_count
        }
        
    except Exception as e:
        logging.error(f"❌ خطا در بررسی تراکنش‌ها: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطا در بررسی تراکنش‌ها: {str(e)}")

# ==================== WITHDRAWAL SYSTEM ====================

class WithdrawalRequest(BaseModel):
    amount: float
    crypto_address: str
    network: str = "USDT-TRC20"

class WithdrawalApprove(BaseModel):
    transaction_hash: str

# ==================== MESSAGING MODELS ====================

class MessageCreate(BaseModel):
    subject: str
    content: str
    recipient_id: Optional[str] = None  # None برای پیام عمومی

class MessageReply(BaseModel):
    content: str

@api_router.post("/user/withdrawals")
async def create_withdrawal(data: WithdrawalRequest, current_user: dict = Depends(get_current_user)):
    """
    درخواست برداشت با محافظت در برابر race condition
    
    🔒 محافظت‌های امنیتی:
    1. Atomic check-and-decrement برای جلوگیری از برداشت بیش از حد
    2. Double validation موجودی
    3. اعتبارسنجی آدرس کیف پول
    """
    # دریافت تنظیمات برداشت
    settings = await db.settings.find_one({"setting_id": "crypto"})
    min_withdrawal = settings.get("min_withdrawal_amount", 10.0) if settings else 10.0
    
    # حداقل برداشت
    if data.amount < min_withdrawal:
        raise HTTPException(status_code=400, detail=f"حداقل مبلغ برداشت {min_withdrawal} USDT است")
    
    # بررسی آدرس
    if data.network == "USDT-TRC20" and not data.crypto_address.startswith("T"):
        raise HTTPException(status_code=400, detail="آدرس نامعتبر برای شبکه TRC20")
    
    if data.network == "USDT-BEP20" and not data.crypto_address.startswith("0x"):
        raise HTTPException(status_code=400, detail="آدرس نامعتبر برای شبکه BEP20")
    
    withdrawal_id = str(uuid.uuid4())
    
    # 🔒 FIX 3: Atomic check-and-decrement برای جلوگیری از race condition
    # این عملیات در یک query انجام می‌شود و atomic است
    # فقط اگر موجودی >= مبلغ درخواستی باشد، کسر انجام می‌شود
    result = await db.users.update_one(
        {
            "user_id": current_user["user_id"],
            "total_points": {"$gte": data.amount}  # شرط موجودی کافی
        },
        {"$inc": {"total_points": -data.amount}}
    )
    
    # اگر update موفق نبود، یعنی موجودی کافی نیست
    if result.modified_count == 0:
        # دریافت موجودی فعلی برای پیام خطای دقیق‌تر
        fresh_user = await db.users.find_one({"user_id": current_user["user_id"]}, {"_id": 0, "total_points": 1})
        current_balance = fresh_user.get("total_points", 0) if fresh_user else 0
        raise HTTPException(
            status_code=400, 
            detail=f"موجودی کافی نیست. موجودی فعلی: {current_balance:.2f} USDT"
        )
    
    # ایجاد رکورد برداشت
    withdrawal = {
        "withdrawal_id": withdrawal_id,
        "user_id": current_user["user_id"],
        "user_email": current_user["email"],
        "amount": data.amount,
        "crypto_address": data.crypto_address,
        "network": data.network,
        "status": "pending_admin",  # pending_admin, approved, rejected, completed
        "admin_note": "",
        "transaction_hash": None,
        "requested_at": datetime.now(timezone.utc).isoformat(),
        "completed_at": None
    }
    
    await db.withdrawals.insert_one(withdrawal)
    
    logging.info(f"✅ درخواست برداشت {withdrawal_id} برای {current_user['email']} با مبلغ {data.amount} ثبت شد")
    
    return {"message": "درخواست برداشت ثبت شد", "withdrawal_id": withdrawal_id}

@api_router.get("/user/withdrawals")
async def get_my_withdrawals(current_user: dict = Depends(get_current_user)):
    """تاریخچه برداشت‌های کاربر"""
    withdrawals = await db.withdrawals.find(
        {"user_id": current_user["user_id"]},
        {"_id": 0}
    ).sort("requested_at", -1).to_list(100)
    return withdrawals

@api_router.put("/user/wallet-address")
async def update_wallet_address(crypto_address: str, network: str, current_user: dict = Depends(get_current_user)):
    """به‌روزرسانی آدرس کیف پول کاربر"""
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$set": {
            "wallet_address": crypto_address,
            "wallet_network": network
        }}
    )
    return {"message": "آدرس کیف پول به‌روزرسانی شد"}

# ==================== ADMIN WITHDRAWAL MANAGEMENT ====================

@api_router.get("/admin/withdrawals")
async def get_all_withdrawals(status: Optional[str] = None, admin_user: dict = Depends(get_admin_user)):
    """لیست درخواست‌های برداشت"""
    query = {}
    if status:
        query["status"] = status
    
    withdrawals = await db.withdrawals.find(query, {"_id": 0}).sort("requested_at", -1).to_list(1000)
    return withdrawals

@api_router.put("/admin/withdrawals/{withdrawal_id}/approve")
async def approve_withdrawal(withdrawal_id: str, data: WithdrawalApprove, admin_user: dict = Depends(get_admin_user)):
    """تایید و تکمیل برداشت"""
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "completed",
            "transaction_hash": data.transaction_hash,
            "completed_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "برداشت تایید شد"}

@api_router.put("/admin/withdrawals/{withdrawal_id}/reject")
async def reject_withdrawal(withdrawal_id: str, reason: str, admin_user: dict = Depends(get_admin_user)):
    """رد برداشت"""
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "rejected",
            "admin_note": reason,
            "completed_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    # برگرداندن موجودی
    await db.users.update_one(
        {"user_id": withdrawal["user_id"]},
        {"$inc": {"total_points": withdrawal["amount"]}}
    )
    
    return {"message": "درخواست رد شد و موجودی برگشت"}

# ==================== AUTO-TRANSFER ENDPOINTS ====================

@api_router.post("/admin/withdrawals/{withdrawal_id}/validate")
async def validate_withdrawal(withdrawal_id: str, admin_user: dict = Depends(get_admin_user)):
    """بررسی اعتبار درخواست برداشت (چک امتیاز واقعی کاربر)"""
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    # دریافت امتیاز واقعی کاربر
    user = await db.users.find_one({"user_id": withdrawal["user_id"]})
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # امتیاز واقعی + مبلغ برداشت (چون قبلاً کسر شده)
    actual_points = user["total_points"] + withdrawal["amount"]
    
    # بررسی validation
    is_valid = actual_points >= withdrawal["amount"]
    
    # دریافت تنظیمات کارمزد
    settings = await db.settings.find_one({"setting_id": "crypto"})
    fee_type = settings.get("withdrawal_fee_type", "fixed") if settings else "fixed"
    fee_amount = settings.get("withdrawal_fee_amount", 0.0) if settings else 0.0
    
    # محاسبه کارمزد
    if fee_type == "fixed":
        fee = fee_amount
    else:  # percentage
        fee = (withdrawal["amount"] * fee_amount) / 100
    
    final_amount = withdrawal["amount"] - fee
    
    # آپدیت withdrawal با اطلاعات validation
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "validation_status": "valid" if is_valid else "invalid",
            "actual_user_points": actual_points,
            "calculated_fee": fee,
            "final_transfer_amount": final_amount,
            "validated_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {
        "valid": is_valid,
        "user_email": withdrawal["user_email"],
        "requested_amount": withdrawal["amount"],
        "actual_user_points": actual_points,
        "fee_type": fee_type,
        "fee_amount": fee,
        "final_transfer_amount": final_amount,
        "message": "✅ درخواست معتبر است" if is_valid else "❌ امتیاز کاربر کافی نیست"
    }

class AutoTransferRequest(BaseModel):
    private_key: str
    save_key: bool = False

@api_router.post("/admin/withdrawals/{withdrawal_id}/auto-transfer")
async def auto_transfer_withdrawal(
    withdrawal_id: str, 
    data: AutoTransferRequest,
    admin_user: dict = Depends(get_admin_user)
):
    """ارسال خودکار تراکنش به blockchain"""
    
    logging.info(f"🔄 شروع auto-transfer برای {withdrawal_id}")
    
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        logging.error(f"❌ درخواست {withdrawal_id} یافت نشد")
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    logging.info(f"📝 وضعیت: {withdrawal['status']}, Validation: {withdrawal.get('validation_status')}")
    
    if withdrawal["status"] != "pending_admin":
        logging.error(f"❌ درخواست قبلاً پردازش شده: {withdrawal['status']}")
        raise HTTPException(status_code=400, detail=f"این درخواست در وضعیت '{withdrawal['status']}' است")
    
    # چک validation
    if not withdrawal.get("validation_status"):
        logging.error("❌ درخواست Validate نشده")
        raise HTTPException(status_code=400, detail="ابتدا باید درخواست را Validate کنید")
    
    if withdrawal.get("validation_status") != "valid":
        logging.error(f"❌ درخواست نامعتبر: {withdrawal.get('validation_status')}")
        raise HTTPException(status_code=400, detail="این درخواست معتبر نیست")
    
    # دریافت مبلغ نهایی
    final_amount = withdrawal.get("final_transfer_amount", withdrawal["amount"])
    to_address = withdrawal["crypto_address"]
    network = withdrawal["network"]
    
    logging.info(f"💰 انتقال {final_amount} USDT به {to_address} در شبکه {network}")
    
    # ارسال تراکنش بر اساس شبکه
    if network == "USDT-TRC20" or network == "TRON":
        logging.info("🔵 استفاده از شبکه TRON")
        result = await send_tron_usdt(data.private_key, to_address, final_amount)
    elif network == "USDT-BEP20" or network == "BSC":
        logging.info("🟡 استفاده از شبکه BSC")
        result = await send_bsc_usdt(data.private_key, to_address, final_amount)
    else:
        logging.error(f"❌ شبکه ناشناخته: {network}")
        raise HTTPException(status_code=400, detail="شبکه پشتیبانی نمی‌شود")
    
    if not result["success"]:
        # اگر ارسال ناموفق بود، موجودی را برگردان
        logging.error(f"❌ خطا در ارسال: {result.get('error')}")
        await db.users.update_one(
            {"user_id": withdrawal["user_id"]},
            {"$inc": {"total_points": withdrawal["amount"]}}
        )
        
        raise HTTPException(status_code=500, detail=f"خطا در ارسال: {result.get('error')}")
    
    # آپدیت وضعیت برداشت
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "completed",
            "transaction_hash": result["tx_hash"],
            "completed_at": datetime.now(timezone.utc).isoformat(),
            "transfer_method": "auto"
        }}
    )
    
    # ذخیره private key (اگر درخواست شده باشد) - رمزگذاری شده
    if data.save_key:
        from cryptography.fernet import Fernet
        # استفاده از یک کلید ساده برای رمزگذاری (در production باید از secret بهتری استفاده شود)
        key = Fernet.generate_key()
        cipher = Fernet(key)
        encrypted_key = cipher.encrypt(data.private_key.encode())
        
        # ذخیره در settings بر اساس شبکه
        if network == "USDT-TRC20" or network == "TRON":
            await db.settings.update_one(
                {"setting_id": "crypto"},
                {"$set": {"tron_hot_wallet_private_key": encrypted_key.decode()}}
            )
        elif network == "USDT-BEP20" or network == "BSC":
            await db.settings.update_one(
                {"setting_id": "crypto"},
                {"$set": {"bsc_hot_wallet_private_key": encrypted_key.decode()}}
            )
    
    return {
        "success": True,
        "message": "✅ تراکنش با موفقیت ارسال شد",
        "tx_hash": result["tx_hash"],
        "network": result["network"],
        "amount_transferred": final_amount
    }

class BatchTransferRequest(BaseModel):
    withdrawal_ids: List[str]
    private_key: str
    save_key: bool = False

@api_router.post("/admin/withdrawals/batch-transfer")
async def batch_auto_transfer(
    data: BatchTransferRequest,
    admin_user: dict = Depends(get_admin_user)
):
    """ارسال دسته‌جمعی چندین برداشت"""
    
    results = []
    success_count = 0
    fail_count = 0
    
    for withdrawal_id in data.withdrawal_ids:
        try:
            # فراخوانی auto_transfer برای هر withdrawal
            result = await auto_transfer_withdrawal(
                withdrawal_id,
                AutoTransferRequest(
                    private_key=data.private_key,
                    save_key=data.save_key if success_count == 0 else False  # فقط اولین بار ذخیره کند
                ),
                admin_user
            )
            
            results.append({
                "withdrawal_id": withdrawal_id,
                "success": True,
                "tx_hash": result["tx_hash"]
            })
            success_count += 1
            
        except Exception as e:
            results.append({
                "withdrawal_id": withdrawal_id,
                "success": False,
                "error": str(e)
            })
            fail_count += 1
    
    return {
        "total": len(data.withdrawal_ids),
        "success": success_count,
        "failed": fail_count,
        "results": results
    }

# چک موجودی کیف پول
class CheckBalanceRequest(BaseModel):
    private_key: str
    network: str  # "TRON" or "BSC"

@api_router.post("/admin/wallet/check-balance")
async def check_wallet_balance(
    data: CheckBalanceRequest,
    admin_user: dict = Depends(get_admin_user)
):
    """بررسی موجودی کیف پول"""
    
    if data.network == "TRON" or data.network == "USDT-TRC20":
        result = await get_tron_wallet_balance(data.private_key)
    elif data.network == "BSC" or data.network == "USDT-BEP20":
        result = await get_bsc_wallet_balance(data.private_key)
    else:
        raise HTTPException(status_code=400, detail="شبکه نامعتبر")
    
    if not result["success"]:
        raise HTTPException(status_code=500, detail=result.get("error", "خطا در دریافت موجودی"))
    
    return result

# تست انتقال
class TestTransferRequest(BaseModel):
    to_address: str
    amount: float
    network: str  # "TRON" or "BSC"

@api_router.post("/admin/test-transfer")
async def test_transfer(
    data: TestTransferRequest,
    admin_user: dict = Depends(get_admin_user)
):
    """تست سیستم انتقال - برای debug"""
    
    logging.info(f"🧪 تست انتقال: {data.amount} USDT به {data.to_address} در شبکه {data.network}")
    
    # دریافت Private Key از تنظیمات
    settings = await db.settings.find_one({"setting_id": "crypto"})
    
    if not settings:
        raise HTTPException(status_code=400, detail="تنظیمات یافت نشد")
    
    if data.network == "TRON":
        private_key = settings.get("tron_hot_wallet_private_key")
        if not private_key:
            raise HTTPException(status_code=400, detail="Private Key TRON ذخیره نشده")
        
        result = await send_tron_usdt(private_key, data.to_address, data.amount)
        
    elif data.network == "BSC":
        private_key = settings.get("bsc_hot_wallet_private_key")
        if not private_key:
            raise HTTPException(status_code=400, detail="Private Key BSC ذخیره نشده")
        
        result = await send_bsc_usdt(private_key, data.to_address, data.amount)
        
    else:
        raise HTTPException(status_code=400, detail="شبکه نامعتبر")
    
    if not result["success"]:
        logging.error(f"❌ تست ناموفق: {result.get('error')}")
        raise HTTPException(status_code=500, detail=result.get("error", "خطا در ارسال"))
    
    logging.info(f"✅ تست موفق: {result['tx_hash']}")
    
    return {
        "success": True,
        "tx_hash": result["tx_hash"],
        "network": data.network,
        "amount": data.amount,
        "to_address": data.to_address
    }

# رد کردن درخواست برداشت
@api_router.put("/admin/withdrawals/{withdrawal_id}/reject")
async def reject_withdrawal_with_reason(
    withdrawal_id: str,
    reason: str = "رد شده توسط ادمین",
    admin_user: dict = Depends(get_admin_user)
):
    """رد کردن درخواست برداشت و بازگشت موجودی"""
    
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    if withdrawal["status"] != "pending_admin":
        raise HTTPException(status_code=400, detail="این درخواست قبلاً پردازش شده است")
    
    # بازگشت موجودی به کاربر
    await db.users.update_one(
        {"user_id": withdrawal["user_id"]},
        {"$inc": {"total_points": withdrawal["amount"]}}
    )
    
    # آپدیت وضعیت برداشت
    await db.withdrawals.update_one(
        {"withdrawal_id": withdrawal_id},
        {"$set": {
            "status": "rejected",
            "admin_note": reason,
            "rejected_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    logging.info(f"✅ درخواست برداشت {withdrawal_id} رد شد - دلیل: {reason}")
    
    return {
        "success": True,
        "message": "✅ درخواست رد شد و موجودی به کاربر بازگشت",
        "returned_amount": withdrawal["amount"]
    }

# حذف/لغو درخواست برداشت
@api_router.delete("/admin/withdrawals/{withdrawal_id}")
async def cancel_withdrawal(
    withdrawal_id: str,
    admin_user: dict = Depends(get_admin_user)
):
    """حذف/لغو درخواست برداشت (فقط برای درخواست‌های pending)"""
    
    withdrawal = await db.withdrawals.find_one({"withdrawal_id": withdrawal_id})
    if not withdrawal:
        raise HTTPException(status_code=404, detail="درخواست یافت نشد")
    
    if withdrawal["status"] != "pending_admin":
        raise HTTPException(status_code=400, detail="فقط درخواست‌های در انتظار قابل حذف هستند")
    
    # بازگشت موجودی به کاربر
    await db.users.update_one(
        {"user_id": withdrawal["user_id"]},
        {"$inc": {"total_points": withdrawal["amount"]}}
    )
    
    # حذف درخواست
    await db.withdrawals.delete_one({"withdrawal_id": withdrawal_id})
    
    logging.info(f"🗑️ درخواست برداشت {withdrawal_id} حذف شد")
    
    return {
        "success": True,
        "message": "✅ درخواست حذف شد و موجودی به کاربر بازگشت"
    }

# ==================== HOT WALLET MANAGEMENT ====================

class HotWalletCreate(BaseModel):
    wallet_address: str
    network: str = "USDT-TRC20"
    initial_balance: float = 0.0
    daily_limit: float = 1000.0

@api_router.post("/admin/hot-wallet")
async def create_hot_wallet(data: HotWalletCreate, admin_user: dict = Depends(get_admin_user)):
    """افزودن کیف پول برداشت"""
    # غیرفعال کردن کیف پول قبلی
    await db.hot_wallets.update_many(
        {"status": "active"},
        {"$set": {"status": "inactive"}}
    )
    
    wallet_id = str(uuid.uuid4())
    hot_wallet = {
        "wallet_id": wallet_id,
        "wallet_address": data.wallet_address,
        "network": data.network,
        "balance": data.initial_balance,
        "daily_limit": data.daily_limit,
        "status": "active",
        "created_by": admin_user["user_id"],
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.hot_wallets.insert_one(hot_wallet)
    
    return {"message": "کیف پول اضافه شد", "wallet_id": wallet_id}

@api_router.get("/admin/hot-wallet/active")
async def get_active_hot_wallet(admin_user: dict = Depends(get_admin_user)):
    """دریافت کیف پول فعال"""
    wallet = await db.hot_wallets.find_one({"status": "active"}, {"_id": 0})
    return wallet if wallet else {}

@api_router.get("/admin/hot-wallets")
async def get_all_hot_wallets(admin_user: dict = Depends(get_admin_user)):
    """تاریخچه کیف پول‌ها"""
    wallets = await db.hot_wallets.find({}, {"_id": 0}).sort("created_at", -1).to_list(100)
    return wallets

@api_router.put("/admin/hot-wallet/{wallet_id}/balance")
async def update_wallet_balance(wallet_id: str, balance: float, admin_user: dict = Depends(get_admin_user)):
    """به‌روزرسانی موجودی"""
    await db.hot_wallets.update_one(
        {"wallet_id": wallet_id},
        {"$set": {"balance": balance}}
    )
    return {"message": "موجودی به‌روزرسانی شد"}

@api_router.put("/admin/hot-wallet/{wallet_id}/deactivate")
async def deactivate_wallet(wallet_id: str, admin_user: dict = Depends(get_admin_user)):
    """غیرفعال کردن کیف پول"""
    await db.hot_wallets.update_one(
        {"wallet_id": wallet_id},
        {"$set": {"status": "inactive"}}
    )
    return {"message": "کیف پول غیرفعال شد"}

# Financial Stats Endpoint
@api_router.get("/admin/financial-stats")
async def get_financial_stats(admin_user: dict = Depends(get_admin_user)):
    """محاسبات دقیق مالی سیستم"""
    
    # کل پرداخت‌های دریافتی (activation fees)
    total_payments_count = await db.payments.count_documents({})
    payments_pipeline = [
        {"$group": {"_id": None, "total": {"$sum": "$amount"}}}
    ]
    payments_result = await db.payments.aggregate(payments_pipeline).to_list(1)
    total_revenue = payments_result[0]["total"] if payments_result else 0.0
    
    # کل جوایز پرداختی - تفکیک شده
    # Direct rewards
    direct_pipeline = [
        {"$match": {"reward_type": "direct"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}, "count": {"$sum": 1}}}
    ]
    direct_result = await db.transactions.aggregate(direct_pipeline).to_list(1)
    total_direct_rewards = direct_result[0]["total"] if direct_result else 0.0
    direct_rewards_count = direct_result[0]["count"] if direct_result else 0
    
    # Level 3 rewards
    level3_pipeline = [
        {"$match": {"reward_type": "level_3"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}, "count": {"$sum": 1}}}
    ]
    level3_result = await db.transactions.aggregate(level3_pipeline).to_list(1)
    total_level3_rewards = level3_result[0]["total"] if level3_result else 0.0
    level3_rewards_count = level3_result[0]["count"] if level3_result else 0
    
    # Level 5 rewards
    level5_pipeline = [
        {"$match": {"reward_type": "level_5"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}, "count": {"$sum": 1}}}
    ]
    level5_result = await db.transactions.aggregate(level5_pipeline).to_list(1)
    total_level5_rewards = level5_result[0]["total"] if level5_result else 0.0
    level5_rewards_count = level5_result[0]["count"] if level5_result else 0
    
    # Level 7 rewards
    level7_pipeline = [
        {"$match": {"reward_type": "level_7"}},
        {"$group": {"_id": None, "total": {"$sum": "$amount"}, "count": {"$sum": 1}}}
    ]
    level7_result = await db.transactions.aggregate(level7_pipeline).to_list(1)
    total_level7_rewards = level7_result[0]["total"] if level7_result else 0.0
    level7_rewards_count = level7_result[0]["count"] if level7_result else 0
    
    # محاسبات کلی
    total_paid = total_direct_rewards + total_level3_rewards + total_level5_rewards + total_level7_rewards
    profit = total_revenue - total_paid
    profit_percentage = (profit / total_revenue * 100) if total_revenue > 0 else 0
    
    # تعداد کاربران جایزه‌گرفته (unique)
    unique_users_with_rewards = await db.transactions.distinct("user_id")
    
    return {
        "total_revenue": round(total_revenue, 2),
        "total_payments_count": total_payments_count,
        "total_paid": round(total_paid, 2),
        "profit": round(profit, 2),
        "profit_percentage": round(profit_percentage, 2),
        "rewards_breakdown": {
            "direct": {
                "total": round(total_direct_rewards, 2),
                "count": direct_rewards_count
            },
            "level_3": {
                "total": round(total_level3_rewards, 2),
                "count": level3_rewards_count
            },
            "level_5": {
                "total": round(total_level5_rewards, 2),
                "count": level5_rewards_count
            },
            "level_7": {
                "total": round(total_level7_rewards, 2),
                "count": level7_rewards_count
            }
        },
        "unique_users_with_rewards": len(unique_users_with_rewards)
    }

# ==================== SECURITY MIDDLEWARE ====================

# Rate Limiting
from collections import defaultdict
import time

class SimpleRateLimiter:
    """محدود کردن تعداد درخواست‌ها - جلوگیری از DDoS و brute force"""
    def __init__(self):
        self.requests = defaultdict(list)
        self.max_requests = int(os.environ.get('RATE_LIMIT_CALLS', '100'))
        self.window = int(os.environ.get('RATE_LIMIT_PERIOD', '60'))
    
    def is_allowed(self, ip: str) -> bool:
        now = time.time()
        # پاک کردن درخواست‌های قدیمی
        self.requests[ip] = [req for req in self.requests[ip] if now - req < self.window]
        
        if len(self.requests[ip]) >= self.max_requests:
            return False
        
        self.requests[ip].append(now)
        return True

rate_limiter = SimpleRateLimiter()

@app.middleware("http")
async def rate_limit_middleware(request, call_next):
    """Rate limiting middleware"""
    # Skip برای health check
    if request.url.path == "/api/health":
        return await call_next(request)
    
    client_ip = request.client.host
    
    if not rate_limiter.is_allowed(client_ip):
        return JSONResponse(
            status_code=429,
            content={"detail": "تعداد درخواست‌های شما بیش از حد مجاز است. لطفاً چند لحظه صبر کنید."}
        )
    
    response = await call_next(request)
    response.headers["X-RateLimit-Limit"] = str(rate_limiter.max_requests)
    return response

@app.middleware("http")
async def security_headers_middleware(request, call_next):
    """اضافه کردن Security Headers"""
    response = await call_next(request)
    
    # Security Headers
    response.headers["X-Content-Type-Options"] = "nosniff"
    response.headers["X-Frame-Options"] = "DENY"
    response.headers["X-XSS-Protection"] = "1; mode=block"
    response.headers["Referrer-Policy"] = "strict-origin-when-cross-origin"
    response.headers["Permissions-Policy"] = "geolocation=(), microphone=(), camera=()"
    
    # Content Security Policy
    response.headers["Content-Security-Policy"] = (
        "default-src 'self'; "
        "script-src 'self' 'unsafe-inline' 'unsafe-eval'; "
        "style-src 'self' 'unsafe-inline'; "
        "img-src 'self' data: https:; "
        "font-src 'self' data:; "
        "connect-src 'self' https:;"
    )
    
    return response

# CORS Middleware
app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)

# Import JSONResponse
from fastapi.responses import JSONResponse

# ==================== HEALTH CHECK ====================

@api_router.get("/health")
async def health_check():
    """
    Health check endpoint برای monitoring
    بررسی وضعیت سرویس و اتصال به دیتابیس
    """
    try:
        # بررسی اتصال به MongoDB
        await db.command('ping')
        db_status = "healthy"
    except Exception as e:
        db_status = f"unhealthy: {str(e)}"
    
    # بررسی تعداد کاربران active
    try:
        active_users = await db.users.count_documents({"status": "active"})
    except:
        active_users = -1
    
    return {
        "status": "healthy" if db_status == "healthy" else "degraded",
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "database": db_status,
        "active_users": active_users,
        "version": "2.0.0"
    }

@api_router.get("/metrics")
async def get_metrics(admin_user: dict = Depends(get_admin_user)):
    """
    System metrics برای admin
    آمار کلی سیستم
    """
    try:
        total_users = await db.users.count_documents({})
        active_users = await db.users.count_documents({"status": "active"})
        pending_users = await db.users.count_documents({"status": "pending"})
        total_transactions = await db.transactions.count_documents({})
        pending_withdrawals = await db.withdrawals.count_documents({"status": "pending_admin"})
        
        # آمار depth
        depth_stats = {}
        for d in range(0, 10):
            count = await db.users.count_documents({"depth": d, "status": "active"})
            if count > 0:
                depth_stats[f"depth_{d}"] = count
        
        return {
            "timestamp": datetime.now(timezone.utc).isoformat(),
            "users": {
                "total": total_users,
                "active": active_users,
                "pending": pending_users
            },
            "transactions": {
                "total": total_transactions
            },
            "withdrawals": {
                "pending": pending_withdrawals
            },
            "tree_structure": depth_stats
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))
# ==================== MESSAGING SYSTEM ====================

@api_router.post("/user/messages")
async def create_message(data: MessageCreate, current_user: dict = Depends(get_current_user)):
    """ارسال پیام از کاربر به ادمین"""
    
    message_id = str(uuid.uuid4())
    message = {
        "message_id": message_id,
        "sender_id": current_user["user_id"],
        "sender_email": current_user["email"],
        "sender_type": "user",
        "recipient_id": "admin",
        "recipient_type": "admin",
        "subject": data.subject,
        "content": data.content,
        "status": "unread",
        "created_at": datetime.now(timezone.utc).isoformat(),
        "read_at": None,
        "replies": []
    }
    
    await db.messages.insert_one(message)
    
    return {"message": "پیام ارسال شد", "message_id": message_id}

@api_router.get("/user/messages")
async def get_user_messages(current_user: dict = Depends(get_current_user)):
    """دریافت پیام‌های کاربر"""
    
    messages = await db.messages.find(
        {"$or": [
            {"sender_id": current_user["user_id"]},
            {"recipient_id": current_user["user_id"]},
            {"recipient_type": "broadcast"}  # پیام‌های عمومی
        ]},
        {"_id": 0}
    ).sort("created_at", -1).to_list(100)
    
    return messages

@api_router.get("/admin/messages")
async def get_admin_messages(
    status: Optional[str] = None,
    admin_user: dict = Depends(get_admin_user)
):
    """دریافت تمام پیام‌ها برای ادمین"""
    
    query = {"recipient_type": "admin"}
    if status:
        query["status"] = status
    
    messages = await db.messages.find(query, {"_id": 0}).sort("created_at", -1).to_list(500)
    
    return messages

@api_router.post("/admin/messages")
async def admin_send_message(data: MessageCreate, admin_user: dict = Depends(get_admin_user)):
    """ارسال پیام از ادمین"""
    
    message_id = str(uuid.uuid4())
    
    # اگر recipient_id وجود نداشت، پیام عمومی است
    if data.recipient_id:
        # پیام به کاربر خاص
        recipient_type = "user"
        recipient_email = None
        user = await db.users.find_one({"user_id": data.recipient_id})
        if user:
            recipient_email = user["email"]
    else:
        # پیام عمومی (Broadcast)
        recipient_type = "broadcast"
        recipient_email = None
    
    message = {
        "message_id": message_id,
        "sender_id": admin_user["user_id"],
        "sender_email": admin_user["email"],
        "sender_type": "admin",
        "recipient_id": data.recipient_id or "all",
        "recipient_email": recipient_email,
        "recipient_type": recipient_type,
        "subject": data.subject,
        "content": data.content,
        "status": "unread",
        "created_at": datetime.now(timezone.utc).isoformat(),
        "read_at": None,
        "replies": []
    }
    
    await db.messages.insert_one(message)
    
    return {"message": "پیام ارسال شد", "message_id": message_id}

@api_router.post("/messages/{message_id}/reply")
async def reply_to_message(
    message_id: str,
    data: MessageReply,
    current_user: dict = Depends(get_current_user)
):
    """پاسخ به یک پیام"""
    
    message = await db.messages.find_one({"message_id": message_id})
    if not message:
        raise HTTPException(status_code=404, detail="پیام یافت نشد")
    
    reply = {
        "reply_id": str(uuid.uuid4()),
        "sender_id": current_user["user_id"],
        "sender_email": current_user["email"],
        "content": data.content,
        "created_at": datetime.now(timezone.utc).isoformat()
    }
    
    await db.messages.update_one(
        {"message_id": message_id},
        {"$push": {"replies": reply}}
    )
    
    return {"message": "پاسخ ارسال شد"}

@api_router.put("/messages/{message_id}/read")
async def mark_as_read(message_id: str, current_user: dict = Depends(get_current_user)):
    """علامت‌گذاری پیام به عنوان خوانده شده"""
    
    await db.messages.update_one(
        {"message_id": message_id},
        {"$set": {
            "status": "read",
            "read_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "پیام به عنوان خوانده شده علامت‌گذاری شد"}

@api_router.get("/user/messages/unread-count")
async def get_unread_count(current_user: dict = Depends(get_current_user)):
    """تعداد پیام‌های خوانده نشده"""
    
    count = await db.messages.count_documents({
        "$or": [
            {"recipient_id": current_user["user_id"], "status": "unread"},
            {"recipient_type": "broadcast", "status": "unread"}
        ]
    })
    
    return {"unread_count": count}

# ==================== ADMIN DASHBOARD STATS ====================

@api_router.get("/admin/dashboard/stats")
async def get_dashboard_stats(admin_user: dict = Depends(get_admin_user)):
    """آمار کلی برای داشبورد ادمین"""
    
    # تعداد کاربران
    total_users = await db.users.count_documents({})
    active_users = await db.users.count_documents({"status": "active"})
    pending_users = await db.users.count_documents({"status": "pending"})
    
    # مجموع امتیازات
    pipeline = [
        {"$group": {
            "_id": None,
            "total_points": {"$sum": "$total_points"}
        }}
    ]
    result = await db.users.aggregate(pipeline).to_list(1)
    total_points = result[0]["total_points"] if result else 0
    
    # برداشت‌ها
    pending_withdrawals = await db.withdrawals.count_documents({"status": "pending_admin"})
    completed_withdrawals = await db.withdrawals.count_documents({"status": "completed"})
    
    # مجموع مبلغ برداشت‌ها
    pipeline = [
        {"$match": {"status": "completed"}},
        {"$group": {
            "_id": None,
            "total_withdrawn": {"$sum": "$amount"}
        }}
    ]
    result = await db.withdrawals.aggregate(pipeline).to_list(1)
    total_withdrawn = result[0]["total_withdrawn"] if result else 0
    
    # پیام‌های خوانده نشده
    unread_messages = await db.messages.count_documents({
        "recipient_type": "admin",
        "status": "unread"
    })
    
    # پرداخت‌های crypto pending
    pending_crypto_payments = await db.crypto_payments.count_documents({"status": "pending"})
    
    return {
        "users": {
            "total": total_users,
            "active": active_users,
            "pending": pending_users
        },
        "points": {
            "total": total_points
        },
        "withdrawals": {
            "pending": pending_withdrawals,
            "completed": completed_withdrawals,
            "total_amount": total_withdrawn
        },
        "messages": {
            "unread": unread_messages
        },
        "crypto_payments": {
            "pending": pending_crypto_payments
        }
    }

# ===== Announcements & Slider System =====
class AnnouncementCreate(BaseModel):
    title: str
    content: str
    link: Optional[str] = None
    is_modal: bool = False  # True = modal popup, False = slider

@api_router.post("/admin/announcements")
async def create_announcement(data: AnnouncementCreate, admin_user: dict = Depends(get_admin_user)):
    """ایجاد اطلاعیه یا اعلان"""
    announcement = {
        "announcement_id": str(uuid.uuid4()),
        "title": data.title,
        "content": data.content,
        "link": data.link,
        "is_modal": data.is_modal,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "active": True
    }
    await db.announcements.insert_one(announcement)
    return {"message": "اطلاعیه ایجاد شد", "announcement_id": announcement["announcement_id"]}

@api_router.get("/admin/announcements")
async def get_all_announcements(admin_user: dict = Depends(get_admin_user)):
    """دریافت همه اطلاعیه‌ها"""
    announcements = await db.announcements.find({}, {"_id": 0}).sort("created_at", -1).to_list(100)
    return announcements

@api_router.get("/user/announcements/slider")
async def get_slider_announcements():
    """دریافت اطلاعیه‌های اسلایدر"""
    announcements = await db.announcements.find(
        {"is_modal": False, "active": True},
        {"_id": 0}
    ).sort("created_at", -1).to_list(10)
    return announcements

@api_router.get("/user/announcements/modal")
async def get_modal_announcement(current_user: dict = Depends(get_current_user)):
    """دریافت اعلان مودال"""
    # پیدا کردن آخرین اعلان فعال
    modal = await db.announcements.find_one(
        {"is_modal": True, "active": True},
        {"_id": 0},
        sort=[("created_at", -1)]
    )
    
    if not modal:
        return {"has_announcement": False}
    
    # بررسی آیا کاربر قبلاً این اعلان را دیده
    seen = await db.announcement_views.find_one({
        "user_id": current_user["user_id"],
        "announcement_id": modal["announcement_id"]
    })
    
    if seen:
        return {"has_announcement": False}
    
    return {"has_announcement": True, "announcement": modal}

@api_router.post("/user/announcements/{announcement_id}/mark-seen")
async def mark_announcement_seen(announcement_id: str, current_user: dict = Depends(get_current_user)):
    """علامت‌گذاری اعلان به عنوان دیده شده"""
    await db.announcement_views.insert_one({
        "user_id": current_user["user_id"],
        "announcement_id": announcement_id,
        "seen_at": datetime.now(timezone.utc).isoformat()
    })
    return {"message": "ثبت شد"}

@api_router.delete("/admin/announcements/{announcement_id}")
async def delete_announcement(announcement_id: str, admin_user: dict = Depends(get_admin_user)):
    """حذف اطلاعیه"""
    await db.announcements.delete_one({"announcement_id": announcement_id})
    return {"message": "اطلاعیه حذف شد"}

# ===== Admin Configuration Settings =====
class ConfigUpdate(BaseModel):
    key: str
    value: str

@api_router.put("/admin/config")
async def update_config(data: ConfigUpdate, admin_user: dict = Depends(get_admin_user)):
    """به‌روزرسانی تنظیمات"""
    await db.config.update_one(
        {"key": data.key},
        {"$set": {"value": data.value, "updated_at": datetime.now(timezone.utc).isoformat()}},
        upsert=True
    )
    return {"message": "تنظیمات به‌روز شد"}

@api_router.get("/admin/config/{key}")
async def get_config(key: str, admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات"""
    config = await db.config.find_one({"key": key}, {"_id": 0})
    return config if config else {"key": key, "value": ""}

@api_router.get("/user/config/public")
async def get_public_config():
    """دریافت تنظیمات عمومی شامل تب‌های login"""
    keys = ["tutorial_url", "faq_url", "telegram_support", "login_tab_1", "login_tab_2", "login_tab_3"]
    configs = await db.config.find({"key": {"$in": keys}}, {"_id": 0}).to_list(20)
    return {c["key"]: c.get("value", "") for c in configs}


@api_router.delete("/admin/messages/{message_id}")
async def delete_message(message_id: str, admin_user: dict = Depends(get_admin_user)):
    """حذف پیام توسط ادمین"""
    await db.messages.delete_one({"message_id": message_id})
    return {"message": "پیام حذف شد"}


# ===== Login Page Tabs =====
@api_router.get("/public/login-tabs")
async def get_login_tabs():
    """دریافت تب‌های صفحه ورود"""
    tabs = []
    for i in range(1, 4):
        config = await db.config.find_one({"key": f"login_tab_{i}"}, {"_id": 0})
        if config and config.get("value"):
            try:
                import json
                tab_data = json.loads(config["value"])
                if tab_data.get("enabled"):
                    tabs.append(tab_data)
            except:
                pass
    return tabs

@api_router.put("/admin/login-tabs/{tab_number}")
async def update_login_tab(tab_number: int, data: dict, admin_user: dict = Depends(get_admin_user)):
    """به‌روزرسانی تب صفحه ورود"""
    import json
    await db.config.update_one(
        {"key": f"login_tab_{tab_number}"},
        {"$set": {"value": json.dumps(data, ensure_ascii=False)}},
        upsert=True
    )
    return {"message": "تب به‌روز شد"}

# ===== Telegram Integration =====
class TelegramNotificationRequest(BaseModel):
    user_id: str
    telegram_chat_id: int

class TelegramSettings(BaseModel):
    enabled: bool = False
    bot_token: str = ""
    bot_username: str = ""
    webapp_url: str = ""
    welcome_message: str = "🎉 سلام {user_name} عزیز!\n\nبه سیستم MLM ما خوش آمدید!\n\nبرای ورود یا ثبت‌نام، روی دکمه زیر کلیک کنید:\n👇"
    welcome_with_ref_message: str = "🎉 سلام {user_name} عزیز!\n\nشما با کد معرف **{referral_code}** دعوت شده‌اید.\n\nبرای ثبت‌نام و شروع کار، روی دکمه زیر کلیک کنید:\n👇"

async def get_telegram_settings():
    """دریافت تنظیمات تلگرام از دیتابیس"""
    config = await db.config.find_one({"key": "telegram_settings"}, {"_id": 0})
    if config and config.get("value"):
        import json
        return json.loads(config["value"])
    return {
        "enabled": False,
        "bot_token": os.getenv("TELEGRAM_BOT_TOKEN", ""),
        "bot_username": os.getenv("TELEGRAM_BOT_USERNAME", ""),
        "webapp_url": os.getenv("WEBAPP_URL", ""),
        "welcome_message": "🎉 سلام {user_name} عزیز!\n\nبه سیستم MLM ما خوش آمدید!\n\nبرای ورود یا ثبت‌نام، روی دکمه زیر کلیک کنید:\n👇",
        "welcome_with_ref_message": "🎉 سلام {user_name} عزیز!\n\nشما با کد معرف **{referral_code}** دعوت شده‌اید.\n\nبرای ثبت‌نام و شروع کار، روی دکمه زیر کلیک کنید:\n👇"
    }

@api_router.get("/admin/telegram/settings")
async def get_admin_telegram_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات تلگرام برای ادمین"""
    settings = await get_telegram_settings()
    return settings

@api_router.put("/admin/telegram/settings")
async def update_telegram_settings(settings: TelegramSettings, admin_user: dict = Depends(get_admin_user)):
    """به‌روزرسانی تنظیمات تلگرام"""
    import json
    
    await db.config.update_one(
        {"key": "telegram_settings"},
        {"$set": {"value": json.dumps(settings.dict(), ensure_ascii=False)}},
        upsert=True
    )
    
    return {"success": True, "message": "تنظیمات ذخیره شد"}

@api_router.post("/admin/telegram/test")
async def test_telegram_bot(admin_user: dict = Depends(get_admin_user)):
    """تست اتصال به Bot"""
    try:
        settings = await get_telegram_settings()
        
        if not settings.get("bot_token"):
            return {"success": False, "message": "Bot Token تنظیم نشده است"}
        
        from telegram import Bot
        bot = Bot(token=settings["bot_token"])
        me = await bot.get_me()
        
        return {
            "success": True,
            "bot_id": me.id,
            "bot_name": me.first_name,
            "bot_username": me.username
        }
    except Exception as e:
        logger.error(f"Error testing telegram bot: {e}")
        return {"success": False, "message": str(e)}

@api_router.post("/admin/telegram/restart")
async def restart_telegram_bot(admin_user: dict = Depends(get_admin_user)):
    """راه‌اندازی مجدد Bot"""
    try:
        import subprocess
        # Kill old bot
        subprocess.run(["pkill", "-9", "-f", "telegram_bot"], check=False)
        
        # Wait a moment
        import time
        time.sleep(1)
        
        # Start new bot with logging
        log_file = open("/tmp/telegram_bot_live.log", "a")
        subprocess.Popen(
            ["/root/.venv/bin/python3", "/app/backend/telegram_bot_simple.py"],
            stdout=log_file,
            stderr=log_file,
            start_new_session=True,
            cwd="/app/backend"
        )
        return {"success": True, "message": "Bot با موفقیت راه‌اندازی مجدد شد"}
    except Exception as e:
        logger.error(f"Error restarting telegram bot: {e}")
        return {"success": False, "message": str(e)}

@api_router.post("/admin/telegram/broadcast")
async def broadcast_telegram_message(
    data: dict,
    admin_user: dict = Depends(get_admin_user)
):
    """
    ارسال پیام عمومی به تمام کاربران تلگرام
    """
    try:
        message = data.get("message", "")
        if not message:
            return {"success": False, "message": "پیام نمی‌تواند خالی باشد"}
        
        # دریافت تنظیمات تلگرام
        telegram_config = await db.config.find_one({"key": "telegram_bot_settings"}, {"_id": 0})
        
        if not telegram_config:
            return {"success": False, "message": "تنظیمات تلگرام یافت نشد"}
        
        import json
        telegram_settings = json.loads(telegram_config.get("value", "{}"))
        
        if not telegram_settings.get("enabled"):
            return {"success": False, "message": "ربات تلگرام غیرفعال است"}
        
        bot_token = telegram_settings.get("bot_token", "")
        if not bot_token:
            return {"success": False, "message": "توکن ربات یافت نشد"}
        
        # دریافت تمام chat_id هایی که /start زده‌اند
        telegram_chats = await db.telegram_chats.find(
            {},
            {"_id": 0, "chat_id": 1}
        ).to_list(100000)
        
        if not telegram_chats:
            return {"success": False, "message": "هیچ کاربر تلگرامی یافت نشد (کسی /start نزده)"}
        
        # استخراج chat_id ها
        user_ids = [chat["chat_id"] for chat in telegram_chats]
        
        logger.info(f"📢 ارسال پیام به {len(user_ids)} کاربر تلگرام")
        
        # ارسال پیام‌ها
        from telegram_bot_simple import send_broadcast_message
        result = await send_broadcast_message(user_ids, message, bot_token)
        
        logger.info(f"📢 Broadcast: {result['success']} موفق، {result['failed']} ناموفق")
        
        return {
            "success": True,
            "message": f"پیام به {result['success']} کاربر ارسال شد. {result['failed']} ناموفق.",
            "stats": result
        }
        
    except Exception as e:
        logger.error(f"Error broadcasting telegram message: {e}")
        return {"success": False, "message": str(e)}


# Telegram Settings - Simple Version
@api_router.get("/admin/telegram/config")
async def get_telegram_config(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات تلگرام - نسخه ساده"""
    try:
        from telegram_settings import get_settings
        settings = await get_settings()
        logger.info(f"📤 Sending settings: enabled={settings.get('enabled')}")
        return settings
    except Exception as e:
        logger.error(f"Error getting telegram config: {e}")
        return {"enabled": False, "bot_token": "", "bot_username": "", "webapp_url": "", "welcome_message": "", "welcome_with_ref_message": ""}

@api_router.post("/admin/telegram/config")
async def save_telegram_config(settings: dict, admin_user: dict = Depends(get_admin_user)):
    """ذخیره تنظیمات تلگرام - نسخه ساده"""
    try:
        from telegram_settings import save_settings
        success = await save_settings(settings)
        if success:
            logger.info(f"💾 Settings saved: enabled={settings.get('enabled')}")
            return {"success": True, "message": "تنظیمات با موفقیت ذخیره شد"}
        else:
            return {"success": False, "message": "خطا در ذخیره"}
    except Exception as e:
        logger.error(f"Error saving telegram config: {e}")
        return {"success": False, "message": str(e)}
    
@api_router.post("/user/telegram/send-activation")
async def send_telegram_activation(current_user: dict = Depends(get_current_user)):
    """
    ارسال پیام فعالسازی به تلگرام
    این endpoint بعد از فعالسازی کاربر صدا زده می‌شود
    """
    try:
        from telegram_bot import send_activation_message
        
        # دریافت اطلاعات کاربر
        user = await db.users.find_one({"user_id": current_user["user_id"]}, {"_id": 0})
        if not user:
            raise HTTPException(status_code=404, detail="کاربر یافت نشد")
        
        if user.get("status") != "active":
            raise HTTPException(status_code=400, detail="کاربر هنوز فعال نشده است")
        
        # دریافت telegram_chat_id از کاربر (اگر ذخیره شده باشد)
        telegram_chat_id = user.get("telegram_chat_id")
        if not telegram_chat_id:
            return {"success": False, "message": "کاربر از طریق تلگرام ثبت‌نام نکرده است"}
        
        # دریافت bot username از env
        bot_username = os.getenv("TELEGRAM_BOT_USERNAME", "YourBotUsername")
        
        # ارسال پیام
        success = await send_activation_message(
            telegram_user_id=telegram_chat_id,
            user_email=user["email"],
            referral_code=user["referral_code"],
            bot_username=bot_username
        )
        
        if success:
            return {"success": True, "message": "پیام ارسال شد"}
        else:
            return {"success": False, "message": "خطا در ارسال پیام"}
            
    except Exception as e:
        logger.error(f"Error in send_telegram_activation: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.post("/user/telegram/save-chat-id")
async def save_telegram_chat_id(telegram_chat_id: int, current_user: dict = Depends(get_current_user)):
    """ذخیره telegram chat ID برای کاربر"""
    try:
        await db.users.update_one(
            {"user_id": current_user["user_id"]},
            {"$set": {"telegram_chat_id": telegram_chat_id}}
        )
        return {"success": True, "message": "Telegram chat ID ذخیره شد"}
    except Exception as e:
        logger.error(f"Error saving telegram chat ID: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@api_router.get("/user/telegram/bot-info")
async def get_telegram_bot_info():
    """دریافت اطلاعات عمومی Bot برای نمایش لینک‌ها"""
    try:
        settings = await get_telegram_settings()
        return {
            "enabled": settings.get("enabled", False),
            "bot_username": settings.get("bot_username", "YourBotUsername")
        }
    except Exception as e:
        logger.error(f"Error fetching telegram bot info: {e}")
        return {"enabled": False, "bot_username": "YourBotUsername"}

# ==================== SETUP ENDPOINTS ====================
# این endpoint ها برای تست و راه‌اندازی اولیه هستند

@api_router.get("/setup/test")
async def setup_test():
    """تست اتصال به دیتابیس"""
    try:
        # تست MongoDB
        await db.command("ping")
        
        # شمارش کاربران
        users_count = await db.users.count_documents({})
        
        # بررسی seed user
        seed = await db.users.find_one({"user_id": "seed"}, {"_id": 0, "email": 1})
        
        return {
            "status": "ok",
            "database": "connected",
            "users_count": users_count,
            "seed_exists": seed is not None,
            "seed_email": seed["email"] if seed else None,
            "message": "✅ سیستم آماده است" if seed else "⚠️ نیاز به اجرای /api/setup/init"
        }
    except Exception as e:
        return {
            "status": "error",
            "database": "disconnected",
            "error": str(e),
            "message": "❌ خطا در اتصال به دیتابیس"
        }

@api_router.post("/setup/init")
async def setup_init():
    """ایجاد seed user و تنظیمات اولیه"""
    try:
        # بررسی seed user
        seed = await db.users.find_one({"user_id": "seed"})
        
        if seed:
            return {
                "status": "exists",
                "message": "✅ کاربر ادمین قبلاً وجود دارد",
                "email": "admin@mlm.com",
                "password": "admin123"
            }
        
        # ایجاد seed user
        seed_user = {
            "user_id": "seed",
            "email": "admin@mlm.com",
            "phone_number": None,
            "password": hash_password("admin123"),
            "role": "admin",
            "status": "active",
            "referrer_id": None,
            "placement_parent_id": None,
            "direct_children": [],
            "depth": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": datetime.now(timezone.utc).isoformat(),
            "referral_code": "SEED"
        }
        await db.users.insert_one(seed_user)
        
        # ایجاد تنظیمات
        settings = await db.settings.find_one({"setting_id": "rewards"})
        if not settings:
            await db.settings.insert_one({
                "setting_id": "rewards",
                "activation_fee": 100.0,
                "direct_first": 100.0,
                "direct_second": 75.0,
                "direct_third": 50.0,
                "direct_extra": 25.0,
                "level_3": 10.0,
                "level_5": 5.0,
                "level_7": 2.0
            })
        
        return {
            "status": "created",
            "message": "✅ کاربر ادمین ایجاد شد",
            "email": "admin@mlm.com",
            "password": "admin123"
        }
    except Exception as e:
        return {
            "status": "error",
            "message": f"❌ خطا: {str(e)}"
        }

@api_router.post("/setup/reset-admin")
async def setup_reset_admin():
    """ریست رمز ادمین به admin123"""
    try:
        result = await db.users.update_one(
            {"user_id": "seed"},
            {"$set": {"password": hash_password("admin123")}}
        )
        
        if result.modified_count > 0:
            return {
                "status": "ok",
                "message": "✅ رمز ادمین ریست شد",
                "email": "admin@mlm.com",
                "password": "admin123"
            }
        else:
            return {
                "status": "not_found",
                "message": "⚠️ کاربر ادمین یافت نشد - اول /api/setup/init را اجرا کنید"
            }
    except Exception as e:
        return {
            "status": "error",
            "message": f"❌ خطا: {str(e)}"
        }

# Include the router in the main app - باید در آخر باشد بعد از تمام endpoint ها
app.include_router(api_router)

logger = logging.getLogger(__name__)

@app.on_event("shutdown")
async def shutdown_db_client():
    client.close()

