"""Telegram Routes - اندپوینت‌های تلگرام"""

import logging
from fastapi import APIRouter, HTTPException, Depends

from models.telegram import TelegramSettings
from services.auth_service import get_current_user, get_admin_user
from database import get_db

router = APIRouter()


@router.get("/admin/telegram/settings")
async def get_telegram_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات تلگرام"""
    db = await get_db()
    settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
    return settings or {}


@router.put("/admin/telegram/settings")
async def update_telegram_settings(settings: TelegramSettings, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی تنظیمات تلگرام"""
    db = await get_db()
    
    settings_dict = settings.model_dump()
    settings_dict["setting_id"] = "telegram"
    
    await db.settings.update_one(
        {"setting_id": "telegram"},
        {"$set": settings_dict},
        upsert=True
    )
    
    return {"message": "تنظیمات تلگرام بروزرسانی شد"}


@router.post("/admin/telegram/test")
async def test_telegram_bot(admin_user: dict = Depends(get_admin_user)):
    """تست ربات تلگرام"""
    db = await get_db()
    
    settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
    
    if not settings or not settings.get("bot_token"):
        raise HTTPException(status_code=400, detail="توکن ربات تنظیم نشده")
    
    # تست اتصال
    import httpx
    try:
        async with httpx.AsyncClient() as client:
            response = await client.get(
                f"https://api.telegram.org/bot{settings['bot_token']}/getMe",
                timeout=10
            )
            data = response.json()
            
            if data.get("ok"):
                return {
                    "success": True,
                    "bot": data.get("result", {})
                }
            else:
                return {
                    "success": False,
                    "error": data.get("description", "خطا")
                }
    except Exception as e:
        return {
            "success": False,
            "error": str(e)
        }


@router.post("/admin/telegram/broadcast")
async def broadcast_telegram_message(message: str, admin_user: dict = Depends(get_admin_user)):
    """ارسال پیام عمومی تلگرام"""
    db = await get_db()
    
    settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
    
    if not settings or not settings.get("bot_token"):
        raise HTTPException(status_code=400, detail="توکن ربات تنظیم نشده")
    
    # دریافت کاربران با telegram_chat_id
    users = await db.users.find(
        {"telegram_chat_id": {"$ne": None}},
        {"_id": 0, "telegram_chat_id": 1}
    ).to_list(10000)
    
    sent = 0
    failed = 0
    
    import httpx
    async with httpx.AsyncClient() as client:
        for user in users:
            try:
                chat_id = user.get("telegram_chat_id")
                if chat_id:
                    await client.post(
                        f"https://api.telegram.org/bot{settings['bot_token']}/sendMessage",
                        json={"chat_id": chat_id, "text": message},
                        timeout=5
                    )
                    sent += 1
            except:
                failed += 1
    
    return {
        "sent": sent,
        "failed": failed,
        "total": len(users)
    }


@router.get("/user/telegram/bot-info")
async def get_telegram_bot_info():
    """دریافت اطلاعات ربات"""
    db = await get_db()
    
    settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
    
    if not settings or not settings.get("enabled"):
        return {"enabled": False}
    
    return {
        "enabled": True,
        "bot_username": settings.get("bot_username"),
        "webapp_url": settings.get("webapp_url")
    }


@router.post("/user/telegram/save-chat-id")
async def save_telegram_chat_id(telegram_chat_id: int, current_user: dict = Depends(get_current_user)):
    """ذخیره chat_id تلگرام"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": current_user["user_id"]},
        {"$set": {"telegram_chat_id": telegram_chat_id}}
    )
    
    return {"message": "chat_id ذخیره شد"}


@router.get("/public/login-tabs")
async def get_login_tabs():
    """دریافت تنظیمات تب‌های ورود"""
    db = await get_db()
    
    config = await db.config.find_one({"key": "login_tabs"}, {"_id": 0})
    
    return config.get("value", {"tab1": True, "tab2": True}) if config else {"tab1": True, "tab2": True}


@router.put("/admin/login-tabs/{tab_number}")
async def update_login_tab(tab_number: int, data: dict, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی تنظیمات تب ورود"""
    db = await get_db()
    
    current = await db.config.find_one({"key": "login_tabs"}, {"_id": 0})
    tabs = current.get("value", {}) if current else {}
    tabs[f"tab{tab_number}"] = data.get("enabled", True)
    
    await db.config.update_one(
        {"key": "login_tabs"},
        {"$set": {"value": tabs}},
        upsert=True
    )
    
    return {"message": "تنظیمات بروزرسانی شد"}
