import { useState, useEffect } from 'react';
import axios from 'axios';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { MessageCircle, ExternalLink } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function TelegramSupport() {
  const [telegramId, setTelegramId] = useState('');
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchConfig();
  }, []);

  const fetchConfig = async () => {
    try {
      const res = await axios.get(`${API}/user/config/public`);
      setTelegramId(res.data.telegram_support || '');
    } catch (error) {
      console.error('Error fetching config:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleOpenTelegram = () => {
    if (telegramId) {
      const cleanId = telegramId.replace('@', '');
      window.open(`https://t.me/${cleanId}`, '_blank');
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <Card className="max-w-2xl mx-auto">
      <CardHeader className="bg-gradient-to-r from-blue-500 to-cyan-600 text-white">
        <CardTitle className="text-2xl flex items-center gap-2">
          <MessageCircle className="w-6 h-6" />
          پشتیبانی تلگرام
        </CardTitle>
        <CardDescription className="text-blue-50">
          برای دریافت پشتیبانی سریع با ما در تلگرام در ارتباط باشید
        </CardDescription>
      </CardHeader>
      <CardContent className="p-8 space-y-6">
        <div className="text-center space-y-4">
          <div className="w-24 h-24 bg-gradient-to-br from-blue-500 to-cyan-500 rounded-full mx-auto flex items-center justify-center">
            <MessageCircle className="w-12 h-12 text-white" />
          </div>
          
          {telegramId ? (
            <>
              <h3 className="text-xl font-semibold text-gray-800">
                آیدی تلگرام پشتیبانی
              </h3>
              <p className="text-2xl font-bold text-blue-600">
                {telegramId}
              </p>
              
              <Button
                onClick={handleOpenTelegram}
                className="w-full max-w-sm bg-gradient-to-r from-blue-500 to-cyan-500 hover:from-blue-600 hover:to-cyan-600 text-lg py-6"
              >
                <ExternalLink className="w-5 h-5 ml-2" />
                باز کردن در تلگرام
              </Button>

            </>
          ) : (
            <div className="text-gray-500">
              <p>اطلاعات پشتیبانی هنوز تنظیم نشده است</p>
              <p className="text-sm mt-2">لطفاً با مدیر سیستم تماس بگیرید</p>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}
