import { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Switch } from './ui/switch';
import { Save, ExternalLink } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

export default function AdminLoginTabs() {
  const [tabs, setTabs] = useState([
    { number: 1, title: '', url: '', description: '', button_text: '', enabled: false },
    { number: 2, title: '', url: '', description: '', button_text: '', enabled: false },
    { number: 3, title: '', url: '', description: '', button_text: '', enabled: false }
  ]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    fetchTabs();
  }, []);

  const fetchTabs = async () => {
    try {
      const token = localStorage.getItem('token');
      
      // Fetch each tab individually with error handling
      const fetchTab = async (tabNumber) => {
        try {
          const response = await axios.get(`${API}/admin/config/login_tab_${tabNumber}`, { 
            headers: { Authorization: `Bearer ${token}` } 
          });
          
          if (response.data?.value) {
            try {
              const data = JSON.parse(response.data.value);
              return { 
                number: tabNumber, 
                title: data.title || '',
                url: data.url || '',
                description: data.description || '',
                button_text: data.button_text || '',
                enabled: data.enabled || false
              };
            } catch {
              return { number: tabNumber, title: '', url: '', description: '', button_text: '', enabled: false };
            }
          }
          return { number: tabNumber, title: '', url: '', description: '', button_text: '', enabled: false };
        } catch (err) {
          // Return default if config doesn't exist yet (404)
          return { number: tabNumber, title: '', url: '', description: '', button_text: '', enabled: false };
        }
      };

      const results = await Promise.all([
        fetchTab(1),
        fetchTab(2),
        fetchTab(3)
      ]);

      setTabs(results);
    } catch (error) {
      console.error('Error fetching tabs:', error);
    }
  };

  const handleSave = async (tabNumber) => {
    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      const tab = tabs.find(t => t.number === tabNumber);
      
      await axios.put(
        `${API}/admin/login-tabs/${tabNumber}`,
        {
          title: tab.title,
          url: tab.url,
          description: tab.description,
          button_text: tab.button_text,
          enabled: tab.enabled
        },
        { headers: { Authorization: `Bearer ${token}` } }
      );

      toast.success(`✅ تب ${tabNumber} ذخیره شد`);
    } catch (error) {
      toast.error('خطا در ذخیره');
    } finally {
      setLoading(false);
    }
  };

  const updateTab = (number, field, value) => {
    setTabs(tabs.map(t => 
      t.number === number ? { ...t, [field]: value } : t
    ));
  };

  return (
    <div className="max-w-4xl mx-auto space-y-6">
      <Card>
        <CardHeader className="bg-gradient-to-r from-cyan-500 to-blue-600 text-white">
          <CardTitle className="text-2xl">🔗 تب‌های صفحه ورود</CardTitle>
          <CardDescription className="text-cyan-50">
            سه تب قابل تنظیم در بالای صفحه ورود
          </CardDescription>
        </CardHeader>
        <CardContent className="p-6">
          <div className="space-y-6">
            {tabs.map((tab) => (
              <div key={tab.number} className={`border-2 rounded-xl p-6 space-y-4 transition-all ${tab.enabled ? 'bg-gradient-to-br from-blue-50 to-purple-50 border-blue-300 shadow-lg' : 'bg-gray-50 border-gray-200'}`}>
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-xl font-bold text-gray-800 flex items-center gap-2">
                    <span className={`w-8 h-8 rounded-full flex items-center justify-center ${tab.enabled ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white' : 'bg-gray-300 text-gray-600'}`}>
                      {tab.number}
                    </span>
                    تب {tab.number}
                  </h3>
                  <div className="flex items-center gap-3 bg-white px-4 py-2 rounded-lg shadow-sm">
                    <Label htmlFor={`enabled-${tab.number}`} className="font-semibold text-gray-700">فعال</Label>
                    <Switch
                      id={`enabled-${tab.number}`}
                      checked={tab.enabled}
                      onCheckedChange={(checked) => updateTab(tab.number, 'enabled', checked)}
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor={`title-${tab.number}`}>عنوان تب</Label>
                  <Input
                    id={`title-${tab.number}`}
                    value={tab.title}
                    onChange={(e) => updateTab(tab.number, 'title', e.target.value)}
                    placeholder="مثال: کانال تلگرام"
                    disabled={!tab.enabled}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor={`url-${tab.number}`}>لینک</Label>
                  <Input
                    id={`url-${tab.number}`}
                    value={tab.url}
                    onChange={(e) => updateTab(tab.number, 'url', e.target.value)}
                    placeholder="https://t.me/yourchannel"
                    disabled={!tab.enabled}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor={`description-${tab.number}`}>متن توضیحات</Label>
                  <Input
                    id={`description-${tab.number}`}
                    value={tab.description}
                    onChange={(e) => updateTab(tab.number, 'description', e.target.value)}
                    placeholder="برای مشاهده محتوا، روی دکمه زیر کلیک کنید"
                    disabled={!tab.enabled}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor={`button-text-${tab.number}`}>متن دکمه</Label>
                  <Input
                    id={`button-text-${tab.number}`}
                    value={tab.button_text}
                    onChange={(e) => updateTab(tab.number, 'button_text', e.target.value)}
                    placeholder="مشاهده"
                    disabled={!tab.enabled}
                  />
                </div>

                <Button
                  onClick={() => handleSave(tab.number)}
                  disabled={loading || !tab.enabled}
                  className="w-full h-12 text-base font-bold bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 shadow-lg hover:shadow-xl transition-all"
                >
                  <Save className="w-5 h-5 ml-2" />
                  ذخیره تب {tab.number}
                </Button>
              </div>
            ))}
          </div>

          <div className="bg-gradient-to-r from-blue-50 to-purple-50 border-2 border-blue-300 p-6 rounded-xl mt-8 shadow-md">
            <h4 className="font-bold text-blue-900 mb-3 text-lg flex items-center gap-2">
              💡 راهنمای استفاده
            </h4>
            <ul className="text-sm text-blue-800 space-y-2">
              <li className="flex items-start gap-2">
                <span className="text-blue-600 font-bold">•</span>
                <span>تب‌ها در پایین فرم ورود نمایش داده می‌شوند</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-blue-600 font-bold">•</span>
                <span>می‌توانید لینک کانال، گروه، وب‌سایت و... قرار دهید</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-blue-600 font-bold">•</span>
                <span>متن توضیحات و متن دکمه را می‌توانید شخصی‌سازی کنید</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-blue-600 font-bold">•</span>
                <span>با کلیک روی دکمه، کاربر به لینک مربوطه هدایت می‌شود</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-blue-600 font-bold">•</span>
                <span>فقط تب‌های فعال نمایش داده می‌شوند</span>
              </li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
