<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>داشبورد - سیستم MLM</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            direction: rtl;
        }
        
        .container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar */
        .sidebar {
            width: 250px;
            background: #2c3e50;
            color: white;
            padding: 20px 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
        }
        
        .sidebar-header {
            padding: 0 20px 20px;
            border-bottom: 1px solid #34495e;
        }
        
        .sidebar-header h2 {
            font-size: 18px;
            color: #ecf0f1;
        }
        
        .user-info {
            font-size: 12px;
            color: #95a5a6;
            margin-top: 5px;
        }
        
        .sidebar-menu {
            list-style: none;
            padding: 20px 0;
        }
        
        .sidebar-menu li {
            margin: 5px 0;
        }
        
        .sidebar-menu a {
            display: block;
            padding: 12px 20px;
            color: #ecf0f1;
            text-decoration: none;
            transition: background 0.3s;
        }
        
        .sidebar-menu a:hover,
        .sidebar-menu a.active {
            background: #34495e;
        }
        
        /* Main Content */
        .main-content {
            flex: 1;
            margin-right: 250px;
            padding: 30px;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            color: #2c3e50;
            font-size: 24px;
        }
        
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .status-badge.active {
            background: #d4edda;
            color: #155724;
        }
        
        .status-badge.pending {
            background: #fff3cd;
            color: #856404;
        }
        
        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .stat-card h3 {
            color: #7f8c8d;
            font-size: 14px;
            margin-bottom: 10px;
        }
        
        .stat-card .value {
            font-size: 32px;
            font-weight: bold;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        
        .stat-card .label {
            font-size: 12px;
            color: #95a5a6;
        }
        
        .stat-card.blue { border-right: 4px solid #3498db; }
        .stat-card.green { border-right: 4px solid #27ae60; }
        .stat-card.orange { border-right: 4px solid #e67e22; }
        .stat-card.purple { border-right: 4px solid #9b59b6; }
        
        /* Cards */
        .card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        
        .card h2 {
            color: #2c3e50;
            margin-bottom: 20px;
            font-size: 18px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: bold;
            text-decoration: none;
            display: inline-block;
            transition: transform 0.2s;
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-success {
            background: #27ae60;
            color: white;
        }
        
        .btn-warning {
            background: #f39c12;
            color: white;
        }
        
        .referral-box {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .referral-box label {
            display: block;
            font-weight: bold;
            color: #1976d2;
            margin-bottom: 10px;
        }
        
        .referral-input {
            display: flex;
            gap: 10px;
        }
        
        .referral-input input {
            flex: 1;
            padding: 10px;
            border: 2px solid #90caf9;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        table th {
            background: #f8f9fa;
            padding: 12px;
            text-align: right;
            font-weight: 600;
            color: #2c3e50;
            border-bottom: 2px solid #dee2e6;
        }
        
        table td {
            padding: 12px;
            border-bottom: 1px solid #dee2e6;
        }
        
        .badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
        }
        
        .badge.success { background: #d4edda; color: #155724; }
        .badge.warning { background: #fff3cd; color: #856404; }
        .badge.info { background: #d1ecf1; color: #0c5460; }
    </style>
</head>
<body>
    <div class="container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>🎯 سیستم MLM</h2>
                <div class="user-info" id="userEmail">در حال بارگذاری...</div>
            </div>
            <ul class="sidebar-menu">
                <li><a href="index.php" class="active">📊 داشبورد</a></li>
                <li><a href="tree.php">🌳 درخت من</a></li>
                <li><a href="payment.php">💳 پرداخت</a></li>
                <li><a href="withdrawal.php">💸 برداشت</a></li>
                <li><a href="transactions.php">📝 تراکنش‌ها</a></li>
                <li><a href="profile.php">👤 پروفایل</a></li>
                <li><a href="#" onclick="logout()">🚪 خروج</a></li>
            </ul>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h1>📊 داشبورد</h1>
                <span class="status-badge" id="userStatus">در حال بارگذاری...</span>
            </div>
            
            <div id="loading" class="loading">
                <div class="spinner"></div>
                <p style="color: #7f8c8d; margin-top: 10px;">در حال بارگذاری...</p>
            </div>
            
            <div id="content" style="display: none;">
                <!-- Stats Cards -->
                <div class="stats-grid">
                    <div class="stat-card blue">
                        <h3>موجودی کل</h3>
                        <div class="value">$<span id="totalBalance">0</span></div>
                        <div class="label">قابل برداشت</div>
                    </div>
                    
                    <div class="stat-card green">
                        <h3>کل پاداش‌ها</h3>
                        <div class="value">$<span id="totalRewards">0</span></div>
                        <div class="label">دریافت شده</div>
                    </div>
                    
                    <div class="stat-card orange">
                        <h3>زیرمجموعه‌ها</h3>
                        <div class="value"><span id="totalDesc">0</span></div>
                        <div class="label">نفر</div>
                    </div>
                    
                    <div class="stat-card purple">
                        <h3>درخت</h3>
                        <div class="value"><span id="treeStatus">ناقص</span></div>
                        <div class="label">وضعیت</div>
                    </div>
                </div>
                
                <!-- Referral Link -->
                <div class="card">
                    <h2>🔗 لینک معرفی شما</h2>
                    <div class="referral-box">
                        <label>لینک معرفی:</label>
                        <div class="referral-input">
                            <input type="text" id="referralLink" readonly>
                            <button class="btn btn-primary" onclick="copyReferralLink()">کپی</button>
                        </div>
                        <p style="margin-top: 10px; font-size: 12px; color: #666;">
                            کد معرفی: <strong id="referralCode">-</strong>
                        </p>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="card">
                    <h2>⚡ عملیات سریع</h2>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <a href="payment.php" class="btn btn-primary" id="paymentBtn">💳 فعال‌سازی حساب</a>
                        <a href="tree.php" class="btn btn-success">🌳 مشاهده درخت</a>
                        <a href="withdrawal.php" class="btn btn-warning" id="withdrawalBtn">💸 درخواست برداشت</a>
                    </div>
                </div>
                
                <!-- Recent Transactions -->
                <div class="card">
                    <h2>📝 آخرین تراکنش‌ها</h2>
                    <div id="recentTransactions">
                        <p style="text-align: center; padding: 20px; color: #7f8c8d;">در حال بارگذاری...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        const API_BASE = window.location.origin + '/api';
        const userId = localStorage.getItem('user_id');
        
        if (!userId) {
            window.location.href = '../auth/login.php';
        }
        
        async function loadDashboard() {
            try {
                // Load user info
                const userResponse = await fetch(`${API_BASE}/user/${userId}`);
                const userData = await userResponse.json();
                
                if (!userData.success) {
                    throw new Error('کاربر یافت نشد');
                }
                
                const user = userData.data;
                
                // Update UI
                document.getElementById('userEmail').textContent = user.email;
                
                const statusBadge = document.getElementById('userStatus');
                statusBadge.textContent = user.status === 'active' ? '✅ فعال' : '⏳ در انتظار فعال‌سازی';
                statusBadge.className = `status-badge ${user.status}`;
                
                // Stats
                document.getElementById('totalBalance').textContent = parseFloat(user.total_points || 0).toFixed(2);
                document.getElementById('totalDesc').textContent = user.total_descendants || 0;
                document.getElementById('treeStatus').textContent = user.is_tree_complete ? 'کامل ✅' : 'ناقص';
                
                // Referral
                document.getElementById('referralCode').textContent = user.referral_code;
                const refLink = `${window.location.origin}/pages/auth/register.php?ref=${user.referral_code}`;
                document.getElementById('referralLink').value = refLink;
                
                // Button states
                if (user.status === 'active') {
                    document.getElementById('paymentBtn').style.display = 'none';
                    document.getElementById('withdrawalBtn').style.display = 'inline-block';
                } else {
                    document.getElementById('paymentBtn').style.display = 'inline-block';
                    document.getElementById('withdrawalBtn').style.display = 'none';
                }
                
                // Load reward stats
                const rewardResponse = await fetch(`${API_BASE}/reward-stats/${userId}`);
                const rewardData = await rewardResponse.json();
                
                if (rewardData.success) {
                    document.getElementById('totalRewards').textContent = 
                        parseFloat(rewardData.data.grand_total || 0).toFixed(2);
                }
                
                // Load recent transactions
                const txResponse = await fetch(`${API_BASE}/transactions/${userId}?limit=5`);
                const txData = await txResponse.json();
                
                if (txData.success && txData.data.length > 0) {
                    displayTransactions(txData.data);
                } else {
                    document.getElementById('recentTransactions').innerHTML = 
                        '<p style="text-align: center; padding: 20px; color: #7f8c8d;">هنوز تراکنشی ندارید</p>';
                }
                
                // Show content
                document.getElementById('loading').style.display = 'none';
                document.getElementById('content').style.display = 'block';
                
            } catch (error) {
                console.error(error);
                alert('خطا در بارگذاری اطلاعات');
            }
        }
        
        function displayTransactions(transactions) {
            const html = `
                <table>
                    <thead>
                        <tr>
                            <th>نوع</th>
                            <th>مبلغ</th>
                            <th>وضعیت</th>
                            <th>تاریخ</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${transactions.map(tx => `
                            <tr>
                                <td>${getTypeName(tx.transaction_type)}</td>
                                <td>$${parseFloat(tx.amount).toFixed(2)}</td>
                                <td><span class="badge ${tx.status === 'completed' ? 'success' : 'warning'}">${tx.status}</span></td>
                                <td>${new Date(tx.created_at).toLocaleDateString('fa-IR')}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            `;
            document.getElementById('recentTransactions').innerHTML = html;
        }
        
        function getTypeName(type) {
            const types = {
                'direct': 'پاداش مستقیم',
                'level_3': 'پاداش سطح 3',
                'level_5': 'پاداش سطح 5',
                'level_7': 'پاداش سطح 7',
                'withdrawal': 'برداشت'
            };
            return types[type] || type;
        }
        
        function copyReferralLink() {
            const input = document.getElementById('referralLink');
            input.select();
            document.execCommand('copy');
            alert('لینک معرفی کپی شد!');
        }
        
        function logout() {
            localStorage.clear();
            window.location.href = '../auth/login.php';
        }
        
        // Load on page load
        loadDashboard();
    </script>
</body>
</html>
