<?php
/**
 * سرویس مدیریت برداشت (Withdrawal)
 */

require_once __DIR__ . '/Database.php';

class WithdrawalService {
    private $db;
    private $minWithdrawal = 10; // حداقل برداشت
    private $maxWithdrawal = 10000; // حداکثر برداشت
    private $withdrawalFee = 0.02; // 2% کارمزد
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->loadSettings();
    }
    
    /**
     * بارگذاری تنظیمات
     */
    private function loadSettings() {
        $settings = $this->db->fetchAll(
            "SELECT setting_key, setting_value 
             FROM settings 
             WHERE setting_key LIKE 'withdrawal.%'"
        );
        
        foreach ($settings as $setting) {
            $key = str_replace('withdrawal.', '', $setting['setting_key']);
            $this->$key = (float)$setting['setting_value'];
        }
    }
    
    /**
     * 💸 درخواست برداشت
     */
    public function requestWithdrawal(string $userId, float $amount, string $walletAddress, string $network = 'BSC'): array {
        try {
            // بررسی کاربر
            $user = $this->db->fetchOne(
                "SELECT user_id, total_points, pending_withdrawal, status 
                 FROM users 
                 WHERE user_id = ?",
                [$userId]
            );
            
            if (!$user) {
                throw new Exception('User not found');
            }
            
            if ($user['status'] !== 'active') {
                throw new Exception('User is not active');
            }
            
            // بررسی حداقل/حداکثر
            if ($amount < $this->minWithdrawal) {
                throw new Exception("Minimum withdrawal is {$this->minWithdrawal} USDT");
            }
            
            if ($amount > $this->maxWithdrawal) {
                throw new Exception("Maximum withdrawal is {$this->maxWithdrawal} USDT");
            }
            
            // محاسبه موجودی قابل برداشت
            $availableBalance = $user['total_points'] - $user['pending_withdrawal'];
            
            if ($amount > $availableBalance) {
                throw new Exception('Insufficient balance');
            }
            
            // محاسبه کارمزد
            $fee = $amount * $this->withdrawalFee;
            $netAmount = $amount - $fee;
            
            $this->db->beginTransaction();
            
            // ثبت درخواست برداشت
            $withdrawalId = $this->generateWithdrawalId();
            
            $this->db->insert('withdrawals', [
                'withdrawal_id' => $withdrawalId,
                'user_id' => $userId,
                'amount' => $amount,
                'fee' => $fee,
                'net_amount' => $netAmount,
                'wallet_address' => $walletAddress,
                'network' => $network,
                'status' => 'pending'
            ]);
            
            // به‌روزرسانی موجودی کاربر
            $this->db->query(
                "UPDATE users 
                 SET pending_withdrawal = pending_withdrawal + ? 
                 WHERE user_id = ?",
                [$amount, $userId]
            );
            
            $this->db->commit();
            
            return [
                'success' => true,
                'withdrawal_id' => $withdrawalId,
                'amount' => $amount,
                'fee' => $fee,
                'net_amount' => $netAmount,
                'status' => 'pending',
                'message' => 'Withdrawal request submitted. Waiting for admin approval.'
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * ✅ تایید برداشت توسط ادمین
     */
    public function approveWithdrawal(string $withdrawalId, string $txHash = null): array {
        try {
            // دریافت اطلاعات برداشت
            $withdrawal = $this->db->fetchOne(
                "SELECT * FROM withdrawals WHERE withdrawal_id = ?",
                [$withdrawalId]
            );
            
            if (!$withdrawal) {
                throw new Exception('Withdrawal not found');
            }
            
            if ($withdrawal['status'] !== 'pending') {
                throw new Exception('Withdrawal already processed');
            }
            
            $this->db->beginTransaction();
            
            // به‌روزرسانی وضعیت برداشت
            $this->db->update(
                'withdrawals',
                [
                    'status' => 'completed',
                    'tx_hash' => $txHash,
                    'processed_at' => date('Y-m-d H:i:s')
                ],
                'withdrawal_id = ?',
                [$withdrawalId]
            );
            
            // به‌روزرسانی موجودی کاربر
            $this->db->query(
                "UPDATE users 
                 SET pending_withdrawal = pending_withdrawal - ?,
                     total_withdrawn = total_withdrawn + ?,
                     total_points = total_points - ?
                 WHERE user_id = ?",
                [
                    $withdrawal['amount'],
                    $withdrawal['amount'],
                    $withdrawal['amount'],
                    $withdrawal['user_id']
                ]
            );
            
            $this->db->commit();
            
            return [
                'success' => true,
                'message' => 'Withdrawal approved successfully'
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * ❌ رد برداشت
     */
    public function rejectWithdrawal(string $withdrawalId, string $reason): array {
        try {
            $withdrawal = $this->db->fetchOne(
                "SELECT * FROM withdrawals WHERE withdrawal_id = ?",
                [$withdrawalId]
            );
            
            if (!$withdrawal) {
                throw new Exception('Withdrawal not found');
            }
            
            if ($withdrawal['status'] !== 'pending') {
                throw new Exception('Withdrawal already processed');
            }
            
            $this->db->beginTransaction();
            
            // به‌روزرسانی وضعیت
            $this->db->update(
                'withdrawals',
                [
                    'status' => 'rejected',
                    'rejection_reason' => $reason,
                    'processed_at' => date('Y-m-d H:i:s')
                ],
                'withdrawal_id = ?',
                [$withdrawalId]
            );
            
            // بازگرداندن موجودی
            $this->db->query(
                "UPDATE users 
                 SET pending_withdrawal = pending_withdrawal - ? 
                 WHERE user_id = ?",
                [$withdrawal['amount'], $withdrawal['user_id']]
            );
            
            $this->db->commit();
            
            return [
                'success' => true,
                'message' => 'Withdrawal rejected'
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * 📋 دریافت لیست برداشت‌های یک کاربر
     */
    public function getUserWithdrawals(string $userId, int $page = 1, int $limit = 20): array {
        $offset = ($page - 1) * $limit;
        
        $withdrawals = $this->db->fetchAll(
            "SELECT 
                withdrawal_id,
                amount,
                fee,
                net_amount,
                wallet_address,
                network,
                status,
                tx_hash,
                rejection_reason,
                created_at,
                processed_at
             FROM withdrawals
             WHERE user_id = ?
             ORDER BY created_at DESC
             LIMIT ? OFFSET ?",
            [$userId, $limit, $offset]
        );
        
        $total = $this->db->fetchOne(
            "SELECT COUNT(*) as count FROM withdrawals WHERE user_id = ?",
            [$userId]
        )['count'];
        
        return [
            'withdrawals' => $withdrawals,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ];
    }
    
    /**
     * 📋 دریافت برداشت‌های pending (برای ادمین)
     */
    public function getPendingWithdrawals(): array {
        return $this->db->fetchAll(
            "SELECT 
                w.*,
                u.email,
                u.total_points
             FROM withdrawals w
             JOIN users u ON w.user_id = u.user_id
             WHERE w.status = 'pending'
             ORDER BY w.created_at ASC"
        );
    }
    
    /**
     * 📊 آمار برداشت‌ها
     */
    public function getWithdrawalStats(): array {
        $stats = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
                SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_withdrawn
             FROM withdrawals"
        );
        
        return [
            'total_requests' => (int)$stats['total'],
            'pending_requests' => (int)$stats['pending'],
            'completed_requests' => (int)$stats['completed'],
            'rejected_requests' => (int)$stats['rejected'],
            'total_withdrawn' => (float)$stats['total_withdrawn']
        ];
    }
    
    /**
     * تولید Withdrawal ID
     */
    private function generateWithdrawalId(): string {
        return 'WD_' . strtoupper(uniqid() . substr(bin2hex(random_bytes(4)), 0, 8));
    }
}
