<?php
/**
 * کلاس مدیریت کاربران MLM
 * ثبت‌نام، فعال‌سازی، authentication
 */

require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/TreeService.php';
require_once __DIR__ . '/RewardService.php';

class UserService {
    private $db;
    private $treeService;
    private $rewardService;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->treeService = new TreeService();
        $this->rewardService = new RewardService();
    }
    
    /**
     * 📝 ثبت‌نام کاربر جدید (Status = Pending)
     * 
     * @param array $data اطلاعات کاربر
     * @return array نتیجه ثبت‌نام
     */
    public function register(array $data): array {
        // Validation
        $errors = $this->validateRegisterData($data);
        if (!empty($errors)) {
            return [
                'success' => false,
                'errors' => $errors
            ];
        }
        
        // بررسی تکراری نبودن email
        $existingUser = $this->db->fetchOne(
            "SELECT user_id FROM users WHERE email = ?",
            [$data['email']]
        );
        
        if ($existingUser) {
            return [
                'success' => false,
                'errors' => ['email' => 'Email already registered']
            ];
        }
        
        // بررسی وجود referrer
        $referrerId = $data['referrer_id'] ?? 'seed';
        $referrer = $this->db->fetchOne(
            "SELECT user_id, status FROM users WHERE user_id = ? OR referral_code = ?",
            [$referrerId, $referrerId]
        );
        
        if (!$referrer) {
            $referrerId = 'seed'; // fallback به seed
        } else {
            $referrerId = $referrer['user_id'];
        }
        
        try {
            $this->db->beginTransaction();
            
            // تولید user_id و referral_code
            $userId = $this->generateUserId();
            $referralCode = $this->generateReferralCode();
            
            // پیدا کردن placement parent با BFS
            $placementParentId = $this->treeService->findPlacementParent($referrerId, $userId);
            
            // دریافت position
            $position = $this->treeService->getNextPosition($placementParentId);
            
            // Hash password
            $passwordHash = password_hash($data['password'], PASSWORD_BCRYPT, ['cost' => 10]);
            
            // Insert user
            $this->db->insert('users', [
                'user_id' => $userId,
                'email' => $data['email'],
                'password_hash' => $passwordHash,
                'referrer_id' => $referrerId,
                'placement_parent_id' => $placementParentId,
                'placement_position' => $position,
                'referral_code' => $referralCode,
                'status' => 'pending', // ✅ وضعیت اولیه pending
                'is_active' => 0,
                'phone_number' => $data['phone_number'] ?? null,
                'wallet_address' => $data['wallet_address'] ?? null,
                'wallet_network' => $data['wallet_network'] ?? null
            ]);
            
            $this->db->commit();
            
            return [
                'success' => true,
                'user_id' => $userId,
                'referral_code' => $referralCode,
                'placement_parent' => $placementParentId,
                'message' => 'Registration successful. Please make payment to activate.'
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'errors' => ['database' => $e->getMessage()]
            ];
        }
    }
    
    /**
     * 🎯 فعال‌سازی کاربر (پس از تایید پرداخت)
     * شامل: تغییر status + توزیع پاداش
     * 
     * @param string $userId شناسه کاربر
     * @param string $paymentId شناسه پرداخت
     * @return array نتیجه فعال‌سازی
     */

    /**
     * فعال‌سازی توسط ادمین (بدون نیاز به payment)
     */
    public function activateUser(string $userId): array {
        try {
            $this->db->beginTransaction();
            
            $user = $this->db->fetchOne(
                "SELECT * FROM users WHERE user_id = ?",
                [$userId]
            );
            
            if (!$user) {
                throw new Exception('کاربر یافت نشد');
            }
            
            if ($user['status'] === 'active') {
                throw new Exception('کاربر قبلاً فعال شده است');
            }
            
            // Place in tree
            $referrerId = $user['referrer_id'] ?: 'seed';
            $placement = $this->treeService->placeInTree($userId, $referrerId);
            
            if (!$placement['success']) {
                throw new Exception($placement['error']);
            }
            
            // Update status
            $this->db->update(
                'users',
                [
                    'status' => 'active',
                    'activated_at' => date('Y-m-d H:i:s')
                ],
                'user_id = ?',
                [$userId]
            );
            
            $this->db->commit();
            
            // پردازش پاداش‌ها
            $this->rewardService->processActivationRewards($userId, $referrerId);
            
            return [
                'success' => true,
                'message' => 'کاربر با موفقیت فعال شد'
            ];
            
        } catch (Exception $e) {
            $this->db->rollback();
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function activate(string $userId, string $paymentId): array {
        try {
            $this->db->beginTransaction();
            
            // بررسی کاربر
            $user = $this->db->fetchOne(
                "SELECT user_id, status, referrer_id, placement_parent_id, email 
                 FROM users 
                 WHERE user_id = ?",
                [$userId]
            );
            
            if (!$user) {
                throw new Exception('User not found');
            }
            
            if ($user['status'] === 'active') {
                throw new Exception('User already activated');
            }
            
            // بررسی پرداخت
            $payment = $this->db->fetchOne(
                "SELECT payment_id, status 
                 FROM payments 
                 WHERE payment_id = ? AND user_id = ?",
                [$paymentId, $userId]
            );
            
            if (!$payment || $payment['status'] !== 'confirmed') {
                throw new Exception('Payment not confirmed');
            }
            
            // 1. تغییر وضعیت کاربر به active
            $this->db->update(
                'users',
                [
                    'status' => 'active',
                    'is_active' => 1,
                    'activated_at' => date('Y-m-d H:i:s')
                ],
                'user_id = ?',
                [$userId]
            );
            
            $this->db->commit();
            
            // 2. ثبت event برای پردازش deferred
            $this->createRewardEvent($userId);
            
            // 3. پردازش پاداش‌ها (می‌توان در Cron انجام داد)
            // $this->processActivationRewards($userId, $user['referrer_id']);
            
            return [
                'success' => true,
                'user_id' => $userId,
                'message' => 'Activation successful. Rewards will be processed shortly.'
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * 🔄 پردازش پاداش‌های فعال‌سازی (می‌تواند deferred باشد)
     */
    private function processActivationRewards(string $userId, string $referrerId): array {
        return $this->rewardService->processActivationRewards($userId, $referrerId);
    }
    
    /**
     * ثبت event برای پردازش deferred توسط Cron
     */
    private function createRewardEvent(string $userId): void {
        $eventId = 'EVT_' . strtoupper(uniqid());
        
        $this->db->insert('reward_events', [
            'event_id' => $eventId,
            'user_id' => $userId,
            'event_type' => 'activation',
            'status' => 'pending'
        ]);
    }
    
    /**
     * 🔐 Login کاربر
     */
    public function login(string $email, string $password): array {
        $user = $this->db->fetchOne(
            "SELECT user_id, email, password_hash, status, role 
             FROM users 
             WHERE email = ?",
            [$email]
        );
        
        if (!$user) {
            return [
                'success' => false,
                'error' => 'Invalid credentials'
            ];
        }
        
        if (!password_verify($password, $user['password_hash'])) {
            return [
                'success' => false,
                'error' => 'Invalid credentials'
            ];
        }
        
        // ساخت JWT token یا session
        $token = $this->generateToken($user);
        
        return [
            'success' => true,
            'user_id' => $user['user_id'],
            'email' => $user['email'],
            'status' => $user['status'],
            'role' => $user['role'],
            'token' => $token
        ];
    }
    
    /**
     * دریافت اطلاعات کاربر
     */
    public function getUserInfo(string $userId): ?array {
        $user = $this->db->fetchOne(
            "SELECT 
                user_id,
                email,
                referrer_id,
                placement_parent_id,
                placement_path,
                depth,
                direct_children_count,
                total_descendants,
                total_points,
                pending_withdrawal,
                total_withdrawn,
                status,
                is_active,
                is_tree_complete,
                referral_code,
                wallet_address,
                wallet_network,
                phone_number,
                created_at,
                activated_at,
                role
             FROM users
             WHERE user_id = ?",
            [$userId]
        );
        
        if (!$user) {
            return null;
        }
        
        // اضافه کردن آمار درخت
        $user['tree_stats'] = $this->treeService->getTreeStats($userId);
        
        // اضافه کردن آمار پاداش
        $user['reward_stats'] = $this->rewardService->getUserRewardStats($userId);
        
        return $user;
    }
    
    /**
     * Validation داده‌های ثبت‌نام
     */
    private function validateRegisterData(array $data): array {
        $errors = [];
        
        if (empty($data['email']) || !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Valid email is required';
        }
        
        if (empty($data['password']) || strlen($data['password']) < 8) {
            $errors['password'] = 'Password must be at least 8 characters';
        }
        
        return $errors;
    }
    
    /**
     * تولید user_id یکتا
     */
    private function generateUserId(): string {
        return 'USR_' . strtoupper(uniqid() . substr(bin2hex(random_bytes(4)), 0, 8));
    }
    
    /**
     * تولید کد معرفی یکتا
     */
    private function generateReferralCode(): string {
        do {
            $code = strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 10));
            
            $exists = $this->db->fetchOne(
                "SELECT referral_code FROM users WHERE referral_code = ?",
                [$code]
            );
        } while ($exists);
        
        return $code;
    }
    
    /**
     * تولید JWT Token
     */
    private function generateToken(array $user): string {
        $secretKey = $_ENV['JWT_SECRET'] ?? 'mlm-secret-key-change-in-production';
        $issuedAt = time();
        $expire = $issuedAt + 86400; // 24 hours
        
        $payload = [
            'iat' => $issuedAt,
            'exp' => $expire,
            'user_id' => $user['user_id'],
            'email' => $user['email'],
            'role' => $user['role']
        ];
        
        // Simple JWT implementation (بهتر است از کتابخانه استفاده کنید)
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload = json_encode($payload);
        
        $base64UrlHeader = $this->base64UrlEncode($header);
        $base64UrlPayload = $this->base64UrlEncode($payload);
        
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, $secretKey, true);
        $base64UrlSignature = $this->base64UrlEncode($signature);
        
        return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
    }
    
    private function base64UrlEncode(string $text): string {
        return str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($text));
    }
}
