<?php
/**
 * Optimized Tree Service for 100K+ Users
 * Uses caching, optimized queries, and batch processing
 */

require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/CacheService.php';

class TreeService {
    private $db;
    private $cache;
    private $maxChildren = 3;
    private $maxDepth = 7;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->cache = CacheService::getInstance();
    }
    
    /**
     * Place user in tree - OPTIMIZED for scale
     */
    public function placeInTree($userId, $referrerId) {
        // Start transaction
        $this->db->beginTransaction();
        
        try {
            // Lock row for update
            $referrer = $this->db->fetchOne(
                "SELECT * FROM users WHERE user_id = ? FOR UPDATE",
                [$referrerId]
            );
            
            if (!$referrer) {
                throw new Exception("Referrer not found");
            }
            
            // Find available position using optimized query
            $position = $this->findAvailablePosition($referrerId);
            
            if ($position === null) {
                throw new Exception("No available position in tree");
            }
            
            // Calculate path and depth
            $path = $this->buildPath($position['parent_id'], $position['position']);
            $depth = $position['depth'];
            
            // Update user with SINGLE query
            $this->db->update('users', [
                'referrer_id' => $position['parent_id'],
                'path' => $path,
                'depth' => $depth,
                'position' => $position['position']
            ], ['user_id' => $userId]);
            
            // Invalidate caches
            $this->cache->invalidateUser($userId);
            $this->cache->invalidateUser($position['parent_id']);
            
            $this->db->commit();
            
            return [
                'success' => true,
                'parent_id' => $position['parent_id'],
                'position' => $position['position'],
                'depth' => $depth
            ];
            
        } catch (Exception $e) {
            $this->db->rollback();
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Find available position - OPTIMIZED
     * Uses single query instead of BFS traversal
     */
    private function findAvailablePosition($referrerId) {
        // Direct query to find first available spot
        $result = $this->db->fetchOne(
            "SELECT 
                u.user_id as parent_id,
                u.depth + 1 as depth,
                COALESCE(
                    (SELECT MIN(pos.num) 
                     FROM (SELECT 1 as num UNION SELECT 2 UNION SELECT 3) pos
                     WHERE pos.num NOT IN (
                         SELECT position 
                         FROM users 
                         WHERE referrer_id = u.user_id
                     )),
                    NULL
                ) as position
             FROM users u
             WHERE u.path LIKE CONCAT(
                 (SELECT path FROM users WHERE user_id = ?), '%'
             )
             AND u.status = 'active'
             AND u.depth < ?
             AND (
                 SELECT COUNT(*) 
                 FROM users 
                 WHERE referrer_id = u.user_id
             ) < ?
             ORDER BY u.depth ASC, u.created_at ASC
             LIMIT 1",
            [$referrerId, $this->maxDepth, $this->maxChildren]
        );
        
        if ($result && $result['position'] !== null) {
            return $result;
        }
        
        return null;
    }
    
    /**
     * Build path - optimized
     */
    private function buildPath($parentId, $position) {
        if ($parentId === 'seed') {
            return "seed.{$position}";
        }
        
        $parent = $this->db->fetchOne(
            "SELECT path FROM users WHERE user_id = ?",
            [$parentId]
        );
        
        return $parent['path'] . ".{$position}";
    }
    
    /**
     * Get tree for user - WITH CACHING
     */
    public function getTree($userId, $maxDepth = 3) {
        // Try cache first
        $cached = $this->cache->getUserTree($userId);
        if ($cached) {
            return $cached;
        }
        
        // Build tree with optimized query
        $tree = $this->buildTreeOptimized($userId, $maxDepth);
        
        // Cache it
        $this->cache->cacheUserTree($userId, $tree);
        
        return $tree;
    }
    
    /**
     * Build tree with single query - MUCH FASTER
     */
    private function buildTreeOptimized($userId, $maxDepth) {
        $user = $this->db->fetchOne(
            "SELECT * FROM users WHERE user_id = ?",
            [$userId]
        );
        
        if (!$user) {
            return null;
        }
        
        // Get all descendants in ONE query
        $descendants = $this->db->fetchAll(
            "SELECT * FROM users 
             WHERE path LIKE CONCAT(?, '%')
             AND depth <= ?
             ORDER BY depth ASC, position ASC",
            [$user['path'], $user['depth'] + $maxDepth]
        );
        
        // Build tree structure in memory
        return $this->buildTreeFromFlat($user, $descendants);
    }
    
    /**
     * Build tree from flat array
     */
    private function buildTreeFromFlat($root, $allNodes) {
        $tree = [
            'user_id' => $root['user_id'],
            'email' => $root['email'],
            'status' => $root['status'],
            'depth' => $root['depth'],
            'position' => $root['position'],
            'children' => []
        ];
        
        foreach ($allNodes as $node) {
            if ($node['referrer_id'] === $root['user_id']) {
                $tree['children'][] = $this->buildTreeFromFlat($node, $allNodes);
            }
        }
        
        return $tree;
    }
    
    /**
     * Get tree stats - CACHED
     */
    public function getTreeStats($userId) {
        return $this->cache->remember(
            "tree_stats:{$userId}",
            function() use ($userId) {
                return $this->calculateTreeStats($userId);
            },
            600 // 10 minutes
        );
    }
    
    /**
     * Calculate tree stats with optimized query
     */
    private function calculateTreeStats($userId) {
        $user = $this->db->fetchOne(
            "SELECT path, total_descendants, direct_referrals, is_tree_complete 
             FROM users WHERE user_id = ?",
            [$userId]
        );
        
        // Single query for all stats
        $stats = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_descendants,
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_descendants,
                MAX(depth) - MIN(depth) as max_depth
             FROM users
             WHERE path LIKE CONCAT(?, '%') AND user_id != ?",
            [$user['path'], $userId]
        );
        
        return [
            'total_descendants' => $stats['total_descendants'] ?? 0,
            'active_descendants' => $stats['active_descendants'] ?? 0,
            'direct_referrals' => $user['direct_referrals'] ?? 0,
            'is_tree_complete' => (bool)$user['is_tree_complete'],
            'max_depth' => $stats['max_depth'] ?? 0
        ];
    }
    
    /**
     * Batch update tree completeness - for cron
     */
    public function updateTreeCompleteness($limit = 1000) {
        // Mark trees as complete in batch
        $this->db->query(
            "UPDATE users u
             SET is_tree_complete = TRUE
             WHERE is_tree_complete = FALSE
             AND (
                 SELECT COUNT(*) 
                 FROM users d 
                 WHERE d.path LIKE CONCAT(u.path, '%')
                 AND d.status = 'active'
             ) >= ?
             LIMIT ?",
            [pow($this->maxChildren, $this->maxDepth) - 1, $limit]
        );
    }
}
