<?php
/**
 * کلاس مدیریت پاداش‌های MLM
 * محاسبه و توزیع پاداش مستقیم و سطحی
 */

require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/TreeService.php';

class RewardService {
    private $db;
    private $treeService;
    
    // مقادیر پیش‌فرض پاداش (از settings خوانده می‌شود)
    private $settings = [];
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->treeService = new TreeService();
        $this->loadSettings();
    }
    
    /**
     * بارگذاری تنظیمات پاداش از database
     */
    private function loadSettings() {
        $settingsRows = $this->db->fetchAll(
            "SELECT setting_key, setting_value 
             FROM settings 
             WHERE setting_key LIKE 'reward.%'"
        );
        
        foreach ($settingsRows as $row) {
            $key = str_replace('reward.', '', $row['setting_key']);
            $this->settings[$key] = (float)$row['setting_value'];
        }
        
        // مقادیر پیش‌فرض اگر setting نبود
        $defaults = [
            'direct_first' => 100.0,
            'direct_second' => 75.0,
            'direct_third' => 50.0,
            'direct_extra' => 25.0,
            'level_3' => 10.0,
            'level_5' => 5.0,
            'level_7' => 2.0
        ];
        
        $this->settings = array_merge($defaults, $this->settings);
    }
    
    /**
     * 💰 محاسبه و توزیع پاداش مستقیم (Direct Reward)
     * به معرف (referrer) پرداخت می‌شود
     * 
     * @param string $referrerId شناسه معرف
     * @param string $newUserId شناسه کاربر فعال شده
     * @return array نتیجه پرداخت
     */
    public function distributeDirectReward(string $referrerId, string $newUserId): array {
        // بررسی referrer
        $referrer = $this->db->fetchOne(
            "SELECT user_id, status, blocked_direct_reward 
             FROM users 
             WHERE user_id = ?",
            [$referrerId]
        );
        
        if (!$referrer || $referrer['status'] !== 'active') {
            return [
                'success' => false,
                'reason' => 'Referrer is not active',
                'amount' => 0
            ];
        }
        
        if ($referrer['blocked_direct_reward']) {
            return [
                'success' => false,
                'reason' => 'Direct reward blocked',
                'amount' => 0
            ];
        }
        
        // شمارش زیرمجموعه‌های فعال قبلی (غیر از کاربر جدید)
        $directCount = $this->db->fetchOne(
            "SELECT COUNT(*) as count 
             FROM users 
             WHERE referrer_id = ? 
               AND status = 'active' 
               AND user_id != ?",
            [$referrerId, $newUserId]
        )['count'];
        
        // محاسبه مبلغ پاداش بر اساس ترتیب
        $rewardAmount = $this->getDirectRewardAmount($directCount);
        
        // ثبت تراکنش
        $transactionId = $this->generateTransactionId();
        try {
            $this->db->beginTransaction();
            
            // Insert transaction
            $this->db->insert('transactions', [
                'transaction_id' => $transactionId,
                'user_id' => $referrerId,
                'source_user_id' => $newUserId,
                'transaction_type' => 'direct',
                'amount' => $rewardAmount,
                'status' => 'completed',
                'description' => "Direct reward for activation of {$newUserId}",
                'processed_at' => date('Y-m-d H:i:s')
            ]);
            
            // به‌روزرسانی موجودی کاربر
            $this->db->query(
                "UPDATE users 
                 SET total_points = total_points + ? 
                 WHERE user_id = ?",
                [$rewardAmount, $referrerId]
            );
            
            $this->db->commit();
            
            return [
                'success' => true,
                'amount' => $rewardAmount,
                'transaction_id' => $transactionId,
                'recipient' => $referrerId
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'reason' => $e->getMessage(),
                'amount' => 0
            ];
        }
    }
    
    /**
     * محاسبه مبلغ پاداش مستقیم بر اساس ترتیب
     */
    private function getDirectRewardAmount(int $directCount): float {
        switch ($directCount) {
            case 0:
                return $this->settings['direct_first'];  // اولین: $100
            case 1:
                return $this->settings['direct_second']; // دومین: $75
            case 2:
                return $this->settings['direct_third'];  // سومین: $50
            default:
                return $this->settings['direct_extra'];  // بقیه: $25
        }
    }
    
    /**
     * 🎯 محاسبه و توزیع پاداش سطحی (Level Rewards)
     * به ancestors در placement line پرداخت می‌شود
     * 
     * Level 3: depth_diff = 2 → $10
     * Level 5: depth_diff = 4 → $5
     * Level 7: depth_diff = 6 → $2
     * 
     * @param string $newUserId شناسه کاربر فعال شده
     * @return array لیست پاداش‌های پرداخت شده
     */
    public function distributeLevelRewards(string $newUserId): array {
        // دریافت اطلاعات کاربر جدید
        $newUser = $this->db->fetchOne(
            "SELECT user_id, depth FROM users WHERE user_id = ?",
            [$newUserId]
        );
        
        if (!$newUser) {
            return [
                'success' => false,
                'rewards' => [],
                'reason' => 'User not found'
            ];
        }
        
        $newUserDepth = (int)$newUser['depth'];
        
        // دریافت ancestors
        $ancestors = $this->treeService->getAncestors($newUserId);
        
        $rewards = [];
        $transactions = [];
        
        try {
            $this->db->beginTransaction();
            
            foreach ($ancestors as $ancestor) {
                $ancestorDepth = (int)$ancestor['depth'];
                $depthDiff = $newUserDepth - $ancestorDepth;
                
                // بررسی محدودیت‌ها
                if ($ancestor['status'] !== 'active' || $ancestor['blocked_level_reward']) {
                    continue;
                }
                
                // محاسبه پاداش بر اساس depth_diff
                $rewardInfo = $this->getLevelRewardAmount($depthDiff);
                
                if ($rewardInfo['amount'] > 0) {
                    $transactionId = $this->generateTransactionId();
                    
                    // Insert transaction
                    $this->db->insert('transactions', [
                        'transaction_id' => $transactionId,
                        'user_id' => $ancestor['user_id'],
                        'source_user_id' => $newUserId,
                        'transaction_type' => $rewardInfo['type'],
                        'amount' => $rewardInfo['amount'],
                        'status' => 'completed',
                        'level' => $rewardInfo['level'],
                        'depth_diff' => $depthDiff,
                        'description' => "Level {$rewardInfo['level']} reward for activation of {$newUserId}",
                        'processed_at' => date('Y-m-d H:i:s')
                    ]);
                    
                    // به‌روزرسانی موجودی
                    $this->db->query(
                        "UPDATE users 
                         SET total_points = total_points + ? 
                         WHERE user_id = ?",
                        [$rewardInfo['amount'], $ancestor['user_id']]
                    );
                    
                    $rewards[] = [
                        'user_id' => $ancestor['user_id'],
                        'level' => $rewardInfo['level'],
                        'depth_diff' => $depthDiff,
                        'amount' => $rewardInfo['amount'],
                        'transaction_id' => $transactionId
                    ];
                }
            }
            
            $this->db->commit();
            
            return [
                'success' => true,
                'rewards' => $rewards,
                'total_amount' => array_sum(array_column($rewards, 'amount'))
            ];
            
        } catch (Exception $e) {
            if ($this->db->inTransaction()) {
                $this->db->rollback();
            }
            return [
                'success' => false,
                'rewards' => [],
                'reason' => $e->getMessage()
            ];
        }
    }
    
    /**
     * محاسبه مبلغ و level پاداش بر اساس depth_diff
     * 
     * ⚠️ مهم: Level 3 یعنی depth_diff = 2 (نه 3!)
     */
    private function getLevelRewardAmount(int $depthDiff): array {
        switch ($depthDiff) {
            case 2:
                return [
                    'level' => 3,
                    'amount' => $this->settings['level_3'], // $10
                    'type' => 'level_3'
                ];
            case 4:
                return [
                    'level' => 5,
                    'amount' => $this->settings['level_5'], // $5
                    'type' => 'level_5'
                ];
            case 6:
                return [
                    'level' => 7,
                    'amount' => $this->settings['level_7'], // $2
                    'type' => 'level_7'
                ];
            default:
                return [
                    'level' => 0,
                    'amount' => 0,
                    'type' => 'none'
                ];
        }
    }
    
    /**
     * 🔄 پردازش همه پاداش‌ها برای یک فعال‌سازی
     * شامل: direct + level rewards
     */
    public function processActivationRewards(string $newUserId, string $referrerId): array {
        $results = [
            'direct' => null,
            'level' => null
        ];
        
        // 1. پاداش مستقیم به معرف
        if ($referrerId && $referrerId !== 'seed') {
            $results['direct'] = $this->distributeDirectReward($referrerId, $newUserId);
        }
        
        // 2. پاداش‌های سطحی به ancestors
        $results['level'] = $this->distributeLevelRewards($newUserId);
        
        return $results;
    }
    
    /**
     * 📊 دریافت آمار پاداش‌های یک کاربر
     */
    public function getUserRewardStats(string $userId): array {
        $stats = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total_transactions,
                SUM(CASE WHEN transaction_type = 'direct' THEN amount ELSE 0 END) as direct_total,
                SUM(CASE WHEN transaction_type = 'level_3' THEN amount ELSE 0 END) as level_3_total,
                SUM(CASE WHEN transaction_type = 'level_5' THEN amount ELSE 0 END) as level_5_total,
                SUM(CASE WHEN transaction_type = 'level_7' THEN amount ELSE 0 END) as level_7_total,
                SUM(amount) as grand_total
             FROM transactions
             WHERE user_id = ? AND status = 'completed'",
            [$userId]
        );
        
        return [
            'total_transactions' => (int)$stats['total_transactions'],
            'direct_total' => (float)($stats['direct_total'] ?? 0),
            'level_3_total' => (float)($stats['level_3_total'] ?? 0),
            'level_5_total' => (float)($stats['level_5_total'] ?? 0),
            'level_7_total' => (float)($stats['level_7_total'] ?? 0),
            'grand_total' => (float)($stats['grand_total'] ?? 0)
        ];
    }
    
    /**
     * تولید شناسه یکتا برای transaction
     */
    private function generateTransactionId(): string {
        return 'TX_' . strtoupper(uniqid() . substr(md5(microtime()), 0, 8));
    }
}
