<?php
/**
 * Queue Service - Async Job Processing
 * Handles background jobs for scalability
 */

class QueueService {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Add job to queue
     */
    public function addJob($type, $payload) {
        $jobId = $this->generateId();
        
        $this->db->insert('queue_jobs', [
            'job_id' => $jobId,
            'job_type' => $type,
            'payload' => json_encode($payload),
            'status' => 'pending',
            'attempts' => 0,
            'max_attempts' => 3
        ]);
        
        return $jobId;
    }
    
    /**
     * Process pending jobs
     */
    public function processPendingJobs($limit = 10) {
        $jobs = $this->db->fetchAll(
            "SELECT * FROM queue_jobs 
             WHERE status = 'pending' AND attempts < max_attempts 
             ORDER BY created_at ASC 
             LIMIT ?",
            [$limit]
        );
        
        $processed = 0;
        
        foreach ($jobs as $job) {
            if ($this->processJob($job)) {
                $processed++;
            }
        }
        
        return $processed;
    }
    
    /**
     * Process single job
     */
    private function processJob($job) {
        // Mark as processing
        $this->db->update('queue_jobs', 
            ['status' => 'processing'],
            ['job_id' => $job['job_id']]
        );
        
        try {
            $payload = json_decode($job['payload'], true);
            
            switch ($job['job_type']) {
                case 'placement':
                    $this->processPlacement($payload);
                    break;
                    
                case 'reward':
                    $this->processReward($payload);
                    break;
                    
                case 'activation':
                    $this->processActivation($payload);
                    break;
                    
                case 'withdrawal':
                    $this->processWithdrawal($payload);
                    break;
                    
                default:
                    throw new Exception("Unknown job type: {$job['job_type']}");
            }
            
            // Mark as completed
            $this->db->update('queue_jobs', 
                [
                    'status' => 'completed',
                    'processed_at' => date('Y-m-d H:i:s')
                ],
                ['job_id' => $job['job_id']]
            );
            
            return true;
            
        } catch (Exception $e) {
            // Increment attempts
            $attempts = $job['attempts'] + 1;
            
            $status = ($attempts >= $job['max_attempts']) ? 'failed' : 'pending';
            
            $this->db->update('queue_jobs', 
                [
                    'status' => $status,
                    'attempts' => $attempts,
                    'error_message' => $e->getMessage()
                ],
                ['job_id' => $job['job_id']]
            );
            
            return false;
        }
    }
    
    /**
     * Process placement job
     */
    private function processPlacement($payload) {
        require_once __DIR__ . '/TreeService.php';
        $treeService = new TreeService();
        
        $treeService->placeInTree(
            $payload['user_id'],
            $payload['referrer_id']
        );
    }
    
    /**
     * Process reward calculation
     */
    private function processReward($payload) {
        require_once __DIR__ . '/RewardService.php';
        $rewardService = new RewardService();
        
        $rewardService->calculateRewards($payload['user_id']);
    }
    
    /**
     * Process activation
     */
    private function processActivation($payload) {
        require_once __DIR__ . '/UserService.php';
        $userService = new UserService();
        
        $userService->activate(
            $payload['user_id'],
            $payload['payment_id'] ?? null
        );
    }
    
    /**
     * Process withdrawal
     */
    private function processWithdrawal($payload) {
        require_once __DIR__ . '/WithdrawalService.php';
        $withdrawalService = new WithdrawalService();
        
        // This would be for auto-withdrawal if implemented
        // For now, just log
    }
    
    /**
     * Clean old completed/failed jobs
     */
    public function cleanOldJobs($days = 7) {
        $date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $this->db->query(
            "DELETE FROM queue_jobs 
             WHERE status IN ('completed', 'failed') 
             AND created_at < ?",
            [$date]
        );
    }
    
    /**
     * Get queue statistics
     */
    public function getStats() {
        return $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'processing' THEN 1 ELSE 0 END) as processing,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
             FROM queue_jobs"
        );
    }
    
    private function generateId() {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff)
        );
    }
}
