<?php
/**
 * سرویس مدیریت پرداخت با BSC/Tron RPC
 * پشتیبانی از USDT-BEP20 و TRC20
 */

require_once __DIR__ . '/Database.php';

class PaymentService {
    private $db;
    private $network;
    private $rpcUrl;
    private $usdtContract;
    private $systemWallet;
    
    public function __construct(string $network = 'BSC') {
        $this->db = Database::getInstance();
        $this->network = $network;
        $this->loadConfig();
    }
    
    /**
     * بارگذاری تنظیمات
     */
    private function loadConfig() {
        if ($this->network === 'BSC') {
            $this->rpcUrl = 'https://bsc-dataseed.binance.org/';
            $this->usdtContract = '0x55d398326f99059fF775485246999027B3197955'; // USDT-BSC
        } elseif ($this->network === 'TRON') {
            $this->rpcUrl = 'https://api.trongrid.io';
            $this->usdtContract = 'TR7NHqjeKQxGTCi8q8ZY4pL8otSzgjLj6t'; // USDT-TRC20
        }
        
        // آدرس کیف پول سیستم (از settings)
        $wallet = $this->db->fetchOne(
            "SELECT setting_value FROM settings WHERE setting_key = 'system_wallet_address'"
        );
        $this->systemWallet = $wallet ? $wallet['setting_value'] : null;
    }
    
    /**
     * 💳 ساخت درخواست پرداخت
     */
    public function createPayment(string $userId, float $amount = 100): array {
        try {
            $paymentId = $this->generatePaymentId();
            
            // ثبت پرداخت
            $this->db->insert('payments', [
                'payment_id' => $paymentId,
                'user_id' => $userId,
                'amount' => $amount,
                'currency' => 'USDT',
                'network' => $this->network,
                'wallet_address' => $this->systemWallet,
                'status' => 'pending'
            ]);
            
            // QR Code برای پرداخت
            $qrData = $this->generateQRCode($this->systemWallet, $amount);
            
            return [
                'success' => true,
                'payment_id' => $paymentId,
                'wallet_address' => $this->systemWallet,
                'amount' => $amount,
                'currency' => 'USDT',
                'network' => $this->network,
                'qr_code' => $qrData,
                'instructions' => "لطفاً {$amount} USDT به آدرس زیر ارسال کنید و Transaction Hash را وارد نمایید."
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * ✅ تایید پرداخت با Transaction Hash
     */
    public function verifyPayment(string $paymentId, string $txHash): array {
        try {
            // دریافت اطلاعات پرداخت
            $payment = $this->db->fetchOne(
                "SELECT * FROM payments WHERE payment_id = ?",
                [$paymentId]
            );
            
            if (!$payment) {
                throw new Exception('Payment not found');
            }
            
            if ($payment['status'] === 'confirmed') {
                return [
                    'success' => true,
                    'already_confirmed' => true,
                    'message' => 'Payment already confirmed'
                ];
            }
            
            // بررسی تراکنش در blockchain
            $txResult = $this->checkTransaction($txHash);
            
            if (!$txResult['success']) {
                return $txResult;
            }
            
            // بررسی اینکه مبلغ صحیح است
            if ($txResult['amount'] < $payment['amount']) {
                throw new Exception('Amount mismatch');
            }
            
            // بررسی اینکه به آدرس صحیح ارسال شده
            if (strtolower($txResult['to']) !== strtolower($this->systemWallet)) {
                throw new Exception('Wrong recipient address');
            }
            
            // به‌روزرسانی پرداخت
            $this->db->update(
                'payments',
                [
                    'status' => 'confirmed',
                    'tx_hash' => $txHash,
                    'confirmed_at' => date('Y-m-d H:i:s')
                ],
                'payment_id = ?',
                [$paymentId]
            );
            
            return [
                'success' => true,
                'confirmed' => true,
                'amount' => $txResult['amount'],
                'message' => 'Payment verified successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * 🔍 بررسی تراکنش در blockchain
     */
    private function checkTransaction(string $txHash): array {
        if ($this->network === 'BSC') {
            return $this->checkBSCTransaction($txHash);
        } elseif ($this->network === 'TRON') {
            return $this->checkTronTransaction($txHash);
        }
        
        return ['success' => false, 'error' => 'Unknown network'];
    }
    
    /**
     * بررسی تراکنش BSC
     */
    private function checkBSCTransaction(string $txHash): array {
        try {
            // دریافت اطلاعات تراکنش
            $tx = $this->rpcCall('eth_getTransactionByHash', [$txHash]);
            
            if (!$tx) {
                return ['success' => false, 'error' => 'Transaction not found'];
            }
            
            // بررسی تایید شدن
            $receipt = $this->rpcCall('eth_getTransactionReceipt', [$txHash]);
            
            if (!$receipt) {
                return ['success' => false, 'error' => 'Transaction not confirmed yet'];
            }
            
            if ($receipt['status'] !== '0x1') {
                return ['success' => false, 'error' => 'Transaction failed'];
            }
            
            // بررسی اینکه تراکنش USDT است
            if (strtolower($tx['to']) !== strtolower($this->usdtContract)) {
                return ['success' => false, 'error' => 'Not a USDT transaction'];
            }
            
            // استخراج اطلاعات transfer
            $decoded = $this->decodeBEP20Transfer($tx['input']);
            
            return [
                'success' => true,
                'confirmed' => true,
                'amount' => $decoded['amount'],
                'to' => $decoded['to'],
                'from' => $tx['from']
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * بررسی تراکنش Tron
     */
    private function checkTronTransaction(string $txHash): array {
        try {
            $url = "{$this->rpcUrl}/wallet/gettransactionbyid?value={$txHash}";
            
            $response = file_get_contents($url);
            $tx = json_decode($response, true);
            
            if (!$tx || !isset($tx['ret'])) {
                return ['success' => false, 'error' => 'Transaction not found'];
            }
            
            // بررسی موفقیت
            if ($tx['ret'][0]['contractRet'] !== 'SUCCESS') {
                return ['success' => false, 'error' => 'Transaction failed'];
            }
            
            // استخراج اطلاعات
            $contract = $tx['raw_data']['contract'][0];
            
            if ($contract['type'] !== 'TriggerSmartContract') {
                return ['success' => false, 'error' => 'Not a USDT transaction'];
            }
            
            $parameter = $contract['parameter']['value'];
            
            // Decode amount (از hex)
            $data = $parameter['data'];
            $amountHex = substr($data, 72, 64);
            $amount = hexdec($amountHex) / 1e6; // USDT has 6 decimals
            
            // Decode recipient
            $toHex = substr($data, 32, 40);
            $toAddress = $this->hexToTronAddress($toHex);
            
            return [
                'success' => true,
                'confirmed' => true,
                'amount' => $amount,
                'to' => $toAddress,
                'from' => $parameter['owner_address']
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Decode BEP20 transfer input
     */
    private function decodeBEP20Transfer(string $input): array {
        // Transfer: 0xa9059cbb + 32 bytes address + 32 bytes amount
        if (strlen($input) < 138) {
            throw new Exception('Invalid input');
        }
        
        // آدرس مقصد
        $toHex = substr($input, 34, 40);
        $to = '0x' . $toHex;
        
        // مقدار
        $amountHex = substr($input, 74, 64);
        $amount = hexdec($amountHex) / 1e18; // USDT-BEP20 has 18 decimals
        
        return [
            'to' => $to,
            'amount' => $amount
        ];
    }
    
    /**
     * RPC Call برای BSC
     */
    private function rpcCall(string $method, array $params = []): ?array {
        $data = json_encode([
            'jsonrpc' => '2.0',
            'method' => $method,
            'params' => $params,
            'id' => 1
        ]);
        
        $ch = curl_init($this->rpcUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        return $result['result'] ?? null;
    }
    
    /**
     * تبدیل hex به آدرس Tron
     */
    private function hexToTronAddress(string $hex): string {
        // ساده‌سازی شده - در production از کتابخانه استفاده کنید
        return 'T' . $hex;
    }
    
    /**
     * تولید QR Code
     */
    private function generateQRCode(string $address, float $amount): string {
        // URL برای QR Code Generator
        $data = urlencode("{$address}?amount={$amount}");
        return "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data={$data}";
    }
    
    /**
     * تولید Payment ID
     */
    private function generatePaymentId(): string {
        return 'PAY_' . strtoupper(uniqid() . substr(bin2hex(random_bytes(4)), 0, 8));
    }
    
    /**
     * 📊 دریافت آمار پرداخت‌ها
     */
    public function getPaymentStats(): array {
        $stats = $this->db->fetchOne(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'confirmed' THEN 1 ELSE 0 END) as confirmed,
                SUM(CASE WHEN status = 'confirmed' THEN amount ELSE 0 END) as total_amount
             FROM payments"
        );
        
        return [
            'total_payments' => (int)$stats['total'],
            'pending_payments' => (int)$stats['pending'],
            'confirmed_payments' => (int)$stats['confirmed'],
            'total_amount' => (float)$stats['total_amount']
        ];
    }
}
