<?php
/**
 * سرویس اعلان‌رسانی (Email, SMS, Telegram)
 */

require_once __DIR__ . '/Database.php';

class NotificationService {
    private $db;
    private $emailEnabled = true;
    private $smsEnabled = false;
    
    // تنظیمات Email
    private $smtpHost = 'smtp.gmail.com';
    private $smtpPort = 587;
    private $smtpUsername = '';
    private $smtpPassword = '';
    private $fromEmail = 'noreply@mlm.local';
    private $fromName = 'MLM System';
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->loadSettings();
    }
    
    /**
     * بارگذاری تنظیمات
     */
    private function loadSettings() {
        $settings = $this->db->fetchAll(
            "SELECT setting_key, setting_value 
             FROM settings 
             WHERE setting_key LIKE 'email.%' OR setting_key LIKE 'sms.%'"
        );
        
        foreach ($settings as $setting) {
            $parts = explode('.', $setting['setting_key']);
            if (count($parts) === 2) {
                $type = $parts[0]; // email or sms
                $key = $parts[1];
                
                if ($type === 'email') {
                    $varName = 'smtp' . ucfirst($key);
                    if (property_exists($this, $varName)) {
                        $this->$varName = $setting['setting_value'];
                    }
                }
            }
        }
    }
    
    /**
     * 📧 ارسال ایمیل خوش‌آمدگویی
     */
    public function sendWelcomeEmail(string $email, string $userId, string $referralCode): bool {
        $subject = 'خوش آمدید به سیستم MLM';
        
        $message = $this->getEmailTemplate('welcome', [
            'user_id' => $userId,
            'referral_code' => $referralCode,
            'referral_link' => $this->getReferralLink($referralCode)
        ]);
        
        return $this->sendEmail($email, $subject, $message);
    }
    
    /**
     * 📧 ارسال ایمیل فعال‌سازی
     */
    public function sendActivationEmail(string $email, string $userId): bool {
        $subject = 'حساب شما فعال شد!';
        
        $message = $this->getEmailTemplate('activation', [
            'user_id' => $userId
        ]);
        
        return $this->sendEmail($email, $subject, $message);
    }
    
    /**
     * 📧 ارسال ایمیل پاداش جدید
     */
    public function sendRewardNotification(string $email, float $amount, string $type): bool {
        $subject = 'پاداش جدید دریافت کردید!';
        
        $typeText = [
            'direct' => 'پاداش مستقیم',
            'level_3' => 'پاداش سطح 3',
            'level_5' => 'پاداش سطح 5',
            'level_7' => 'پاداش سطح 7'
        ];
        
        $message = $this->getEmailTemplate('reward', [
            'amount' => $amount,
            'type' => $typeText[$type] ?? $type
        ]);
        
        return $this->sendEmail($email, $subject, $message);
    }
    
    /**
     * 📧 ارسال ایمیل برداشت
     */
    public function sendWithdrawalNotification(string $email, string $status, float $amount, string $reason = null): bool {
        $subjects = [
            'pending' => 'درخواست برداشت ثبت شد',
            'completed' => 'برداشت تایید شد',
            'rejected' => 'درخواست برداشت رد شد'
        ];
        
        $subject = $subjects[$status] ?? 'وضعیت برداشت';
        
        $message = $this->getEmailTemplate('withdrawal', [
            'status' => $status,
            'amount' => $amount,
            'reason' => $reason
        ]);
        
        return $this->sendEmail($email, $subject, $message);
    }
    
    /**
     * 📧 ارسال ایمیل (متد اصلی)
     */
    private function sendEmail(string $to, string $subject, string $message): bool {
        if (!$this->emailEnabled) {
            return false;
        }
        
        try {
            // استفاده از PHPMailer (اگر موجود باشد)
            if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
                return $this->sendWithPHPMailer($to, $subject, $message);
            }
            
            // استفاده از mail() ساده PHP
            return $this->sendWithPHPMail($to, $subject, $message);
            
        } catch (Exception $e) {
            $this->logError("Email send failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * ارسال با PHPMailer
     */
    private function sendWithPHPMailer(string $to, string $subject, string $message): bool {
        require_once __DIR__ . '/../vendor/autoload.php';
        
        $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
        
        try {
            // تنظیمات SMTP
            $mail->isSMTP();
            $mail->Host = $this->smtpHost;
            $mail->SMTPAuth = true;
            $mail->Username = $this->smtpUsername;
            $mail->Password = $this->smtpPassword;
            $mail->SMTPSecure = 'tls';
            $mail->Port = $this->smtpPort;
            $mail->CharSet = 'UTF-8';
            
            // فرستنده و گیرنده
            $mail->setFrom($this->fromEmail, $this->fromName);
            $mail->addAddress($to);
            
            // محتوا
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $message;
            
            $mail->send();
            return true;
            
        } catch (Exception $e) {
            $this->logError("PHPMailer Error: " . $mail->ErrorInfo);
            return false;
        }
    }
    
    /**
     * ارسال با mail() ساده
     */
    private function sendWithPHPMail(string $to, string $subject, string $message): bool {
        $headers = [
            'From: ' . $this->fromName . ' <' . $this->fromEmail . '>',
            'Reply-To: ' . $this->fromEmail,
            'Content-Type: text/html; charset=UTF-8',
            'MIME-Version: 1.0'
        ];
        
        return mail($to, $subject, $message, implode("\r\n", $headers));
    }
    
    /**
     * 📱 ارسال SMS
     */
    public function sendSMS(string $phone, string $message): bool {
        if (!$this->smsEnabled) {
            return false;
        }
        
        // اتصال به SMS Gateway (مثلاً کاوه‌نگار، مدیا پیامک و غیره)
        // این بخش بسته به سرویس SMS شما متفاوت است
        
        return false; // پیاده‌سازی نشده
    }
    
    /**
     * 📱 ارسال پیام تلگرام
     */
    public function sendTelegramMessage(string $chatId, string $message): bool {
        $botToken = ''; // از settings بخوانید
        
        if (empty($botToken)) {
            return false;
        }
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        $data = [
            'chat_id' => $chatId,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $result = json_decode($response, true);
        return $result['ok'] ?? false;
    }
    
    /**
     * 📄 دریافت قالب ایمیل
     */
    private function getEmailTemplate(string $type, array $vars = []): string {
        $templates = [
            'welcome' => '
                <html dir="rtl">
                <body style="font-family: Tahoma, Arial; direction: rtl; text-align: right;">
                    <h2>خوش آمدید!</h2>
                    <p>به سیستم MLM ما خوش آمدید.</p>
                    <p><strong>شناسه کاربری:</strong> {user_id}</p>
                    <p><strong>کد معرفی شما:</strong> {referral_code}</p>
                    <p><strong>لینک معرفی:</strong> <a href="{referral_link}">{referral_link}</a></p>
                    <p>برای فعال‌سازی حساب خود، لطفاً مبلغ 100 USDT پرداخت نمایید.</p>
                </body>
                </html>
            ',
            
            'activation' => '
                <html dir="rtl">
                <body style="font-family: Tahoma, Arial; direction: rtl; text-align: right;">
                    <h2>🎉 حساب شما فعال شد!</h2>
                    <p>تبریک! حساب کاربری شما با موفقیت فعال شد.</p>
                    <p>اکنون می‌توانید از تمام امکانات سیستم استفاده کنید.</p>
                    <p>از زیرمجموعه‌های خود دعوت کنید و پاداش دریافت نمایید!</p>
                </body>
                </html>
            ',
            
            'reward' => '
                <html dir="rtl">
                <body style="font-family: Tahoma, Arial; direction: rtl; text-align: right;">
                    <h2>💰 پاداش جدید!</h2>
                    <p>شما یک پاداش جدید دریافت کردید:</p>
                    <p><strong>نوع:</strong> {type}</p>
                    <p><strong>مبلغ:</strong> {amount} USDT</p>
                    <p>موجودی شما به‌روزرسانی شده است.</p>
                </body>
                </html>
            ',
            
            'withdrawal' => '
                <html dir="rtl">
                <body style="font-family: Tahoma, Arial; direction: rtl; text-align: right;">
                    <h2>وضعیت برداشت</h2>
                    <p><strong>مبلغ:</strong> {amount} USDT</p>
                    <p><strong>وضعیت:</strong> {status}</p>
                    {reason_section}
                </body>
                </html>
            '
        ];
        
        $template = $templates[$type] ?? '';
        
        // جایگزینی متغیرها
        foreach ($vars as $key => $value) {
            $template = str_replace('{' . $key . '}', $value, $template);
        }
        
        // پاک کردن متغیرهای باقی‌مانده
        $template = preg_replace('/\{[^}]+\}/', '', $template);
        
        return $template;
    }
    
    /**
     * دریافت لینک معرفی
     */
    private function getReferralLink(string $code): string {
        $baseUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
        return "{$baseUrl}/register?ref={$code}";
    }
    
    /**
     * ثبت خطا
     */
    private function logError(string $message) {
        $logFile = __DIR__ . '/../logs/notification_errors.log';
        $timestamp = date('Y-m-d H:i:s');
        file_put_contents($logFile, "[{$timestamp}] {$message}\n", FILE_APPEND);
    }
}
