<?php
/**
 * کلاس مدیریت اتصال Database
 * استفاده از PDO با Connection Pooling
 */

class Database {
    private static $instance = null;
    private $pdo;
    
    // تنظیمات پیش‌فرض
    private $host = 'localhost';
    private $dbname = 'mlm_production';
    private $username = 'root';
    private $password = '';
    private $charset = 'utf8mb4';
    
    private function __construct() {
        $this->loadConfig();
        $this->connect();
    }
    
    /**
     * بارگذاری تنظیمات از فایل config
     */
    private function loadConfig() {
        $config_file = __DIR__ . '/../config/database.php';
        if (file_exists($config_file)) {
            $config = require $config_file;
            $this->host = $config['host'] ?? $this->host;
            $this->dbname = $config['dbname'] ?? $this->dbname;
            $this->username = $config['username'] ?? $this->username;
            $this->password = $config['password'] ?? $this->password;
            $this->charset = $config['charset'] ?? $this->charset;
        }
    }
    
    /**
     * برقراری اتصال به دیتابیس
     */
    private function connect() {
        $dsn = "mysql:host={$this->host};dbname={$this->dbname};charset={$this->charset}";
        
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
            PDO::ATTR_PERSISTENT => true, // Connection pooling
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$this->charset} COLLATE utf8mb4_unicode_ci"
        ];
        
        try {
            $this->pdo = new PDO($dsn, $this->username, $this->password, $options);
        } catch (PDOException $e) {
            $this->logError('Database connection failed: ' . $e->getMessage());
            throw new Exception('خطا در اتصال به پایگاه داده');
        }
    }
    
    /**
     * Singleton pattern - فقط یک instance
     */
    public static function getInstance(): Database {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * دریافت PDO instance
     */
    public function getConnection(): PDO {
        return $this->pdo;
    }
    
    /**
     * اجرای query ساده
     */
    public function query(string $sql, array $params = []): PDOStatement {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            $this->logError("Query failed: {$sql}\nError: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * دریافت یک ردیف
     */
    public function fetchOne(string $sql, array $params = []): ?array {
        $stmt = $this->query($sql, $params);
        $result = $stmt->fetch();
        return $result ?: null;
    }
    
    /**
     * دریافت چند ردیف
     */
    public function fetchAll(string $sql, array $params = []): array {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }
    
    /**
     * Insert و برگرداندن last insert id
     */
    public function insert(string $table, array $data): string {
        $fields = array_keys($data);
        $placeholders = array_fill(0, count($fields), '?');
        
        $sql = sprintf(
            "INSERT INTO %s (%s) VALUES (%s)",
            $table,
            implode(', ', $fields),
            implode(', ', $placeholders)
        );
        
        $this->query($sql, array_values($data));
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Update
     */
    public function update(string $table, array $data, string $where, array $whereParams = []): int {
        $setParts = [];
        foreach (array_keys($data) as $field) {
            $setParts[] = "$field = ?";
        }
        
        $sql = sprintf(
            "UPDATE %s SET %s WHERE %s",
            $table,
            implode(', ', $setParts),
            $where
        );
        
        $params = array_merge(array_values($data), $whereParams);
        $stmt = $this->query($sql, $params);
        return $stmt->rowCount();
    }
    
    /**
     * Delete
     */
    public function delete(string $table, string $where, array $whereParams = []): int {
        $sql = sprintf("DELETE FROM %s WHERE %s", $table, $where);
        $stmt = $this->query($sql, $whereParams);
        return $stmt->rowCount();
    }
    
    /**
     * شروع Transaction
     */
    public function beginTransaction(): bool {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * Commit Transaction
     */
    public function commit(): bool {
        return $this->pdo->commit();
    }
    
    /**
     * Rollback Transaction
     */
    public function rollback(): bool {
        return $this->pdo->rollBack();
    }
    
    /**
     * بررسی وجود Transaction فعال
     */
    public function inTransaction(): bool {
        return $this->pdo->inTransaction();
    }
    
    /**
     * ثبت خطا در log
     */
    private function logError(string $message) {
        $log_file = __DIR__ . '/../logs/database_errors.log';
        $log_dir = dirname($log_file);
        
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[{$timestamp}] {$message}\n";
        file_put_contents($log_file, $log_message, FILE_APPEND);
    }
    
    /**
     * جلوگیری از clone
     */
    private function __clone() {}
    
    /**
     * جلوگیری از unserialize
     */
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}
