<?php
/**
 * Cache Service - cPanel Version (No Redis Required)
 * Works with MySQL only - Simple and Reliable
 */

class CacheService {
    private static $instance = null;
    private $db = null;
    private $ttl = 3600;
    
    private function __construct() {
        $this->db = Database::getInstance();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function get($key) {
        try {
            $result = $this->db->fetchOne(
                "SELECT cache_value FROM cache WHERE cache_key = ? AND expires_at > NOW()",
                [$key]
            );
            return $result ? $result['cache_value'] : null;
        } catch (Exception $e) {
            return null;
        }
    }
    
    public function set($key, $value, $ttl = null) {
        $ttl = $ttl ?? $this->ttl;
        $expiresAt = date('Y-m-d H:i:s', time() + $ttl);
        
        try {
            $this->db->query("DELETE FROM cache WHERE cache_key = ?", [$key]);
            $this->db->insert('cache', [
                'cache_key' => $key,
                'cache_value' => $value,
                'expires_at' => $expiresAt
            ]);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function delete($key) {
        try {
            $this->db->query("DELETE FROM cache WHERE cache_key = ?", [$key]);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function flush() {
        try {
            $this->db->query("TRUNCATE TABLE cache");
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function remember($key, $callback, $ttl = null) {
        $value = $this->get($key);
        if ($value !== null) {
            $decoded = @json_decode($value, true);
            return $decoded !== null ? $decoded : $value;
        }
        $value = $callback();
        $this->set($key, json_encode($value), $ttl);
        return $value;
    }
    
    public function invalidateUser($userId) {
        $this->delete("user_tree:{$userId}");
        $this->delete("user_stats:{$userId}");
    }
    
    public function cleanExpired() {
        try {
            $this->db->query("DELETE FROM cache WHERE expires_at < NOW()");
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}
