<?php
/**
 * Cron Job: پردازش پاداش‌های معوق (Deferred Rewards)
 * 
 * اجرا: هر 1-5 دقیقه از cPanel Cron Jobs
 * دستور: /usr/bin/php /home/username/public_html/cron/process_rewards.php
 */

// اجرا فقط از CLI
if (php_sapi_name() !== 'cli') {
    die('This script can only be run from CLI');
}

require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/UserService.php';
require_once __DIR__ . '/../includes/RewardService.php';

// تنظیمات
define('BATCH_SIZE', 50);  // تعداد eventها در هر batch
define('MAX_RETRY', 3);    // حداکثر تلاش مجدد
define('LOCK_FILE', __DIR__ . '/../logs/cron_rewards.lock');

/**
 * کلاس پردازش پاداش‌ها
 */
class RewardProcessor {
    private $db;
    private $rewardService;
    private $logFile;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->rewardService = new RewardService();
        $this->logFile = __DIR__ . '/../logs/cron_rewards.log';
        $this->ensureLogDir();
    }
    
    /**
     * اطمینان از وجود پوشه logs
     */
    private function ensureLogDir() {
        $logDir = dirname($this->logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }
    
    /**
     * ثبت log
     */
    private function log(string $message, string $level = 'INFO') {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] [{$level}] {$message}\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND);
        echo $logMessage; // برای نمایش در CLI
    }
    
    /**
     * پردازش eventهای معوق
     */
    public function processPendingEvents(): array {
        $this->log("Starting reward processing...");
        
        $processedCount = 0;
        $errorCount = 0;
        
        try {
            // دریافت eventهای pending
            $events = $this->db->fetchAll(
                "SELECT event_id, user_id, event_type, retry_count
                 FROM reward_events
                 WHERE status = 'pending'
                   AND retry_count < ?
                 ORDER BY created_at ASC
                 LIMIT ?",
                [MAX_RETRY, BATCH_SIZE]
            );
            
            $this->log("Found " . count($events) . " pending events");
            
            foreach ($events as $event) {
                try {
                    $this->processEvent($event);
                    $processedCount++;
                } catch (Exception $e) {
                    $this->log("Error processing event {$event['event_id']}: " . $e->getMessage(), 'ERROR');
                    $this->markEventFailed($event['event_id'], $event['retry_count'], $e->getMessage());
                    $errorCount++;
                }
            }
            
            $this->log("Processing complete. Processed: {$processedCount}, Errors: {$errorCount}");
            
            return [
                'success' => true,
                'processed' => $processedCount,
                'errors' => $errorCount
            ];
            
        } catch (Exception $e) {
            $this->log("Fatal error: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * پردازش یک event
     */
    private function processEvent(array $event): void {
        $this->log("Processing event {$event['event_id']} for user {$event['user_id']}");
        
        // Mark as processing
        $this->db->update(
            'reward_events',
            ['status' => 'processing'],
            'event_id = ?',
            [$event['event_id']]
        );
        
        // دریافت اطلاعات کاربر
        $user = $this->db->fetchOne(
            "SELECT user_id, referrer_id, status FROM users WHERE user_id = ?",
            [$event['user_id']]
        );
        
        if (!$user || $user['status'] !== 'active') {
            throw new Exception("User not active or not found");
        }
        
        // پردازش بر اساس نوع event
        switch ($event['event_type']) {
            case 'activation':
                $this->processActivationRewards($event['user_id'], $user['referrer_id']);
                break;
                
            default:
                throw new Exception("Unknown event type: {$event['event_type']}");
        }
        
        // Mark as completed
        $this->db->update(
            'reward_events',
            [
                'status' => 'completed',
                'processed_at' => date('Y-m-d H:i:s')
            ],
            'event_id = ?',
            [$event['event_id']]
        );
        
        $this->log("Event {$event['event_id']} completed successfully");
    }
    
    /**
     * پردازش پاداش‌های فعال‌سازی
     */
    private function processActivationRewards(string $userId, string $referrerId): void {
        // 1. پاداش مستقیم
        if ($referrerId && $referrerId !== 'seed') {
            $directResult = $this->rewardService->distributeDirectReward($referrerId, $userId);
            if ($directResult['success']) {
                $this->log("Direct reward distributed: {$directResult['amount']} to {$referrerId}");
            }
        }
        
        // 2. پاداش‌های سطحی
        $levelResult = $this->rewardService->distributeLevelRewards($userId);
        if ($levelResult['success']) {
            $this->log("Level rewards distributed: {$levelResult['total_amount']} to " . count($levelResult['rewards']) . " users");
        }
    }
    
    /**
     * علامت‌گذاری event به عنوان failed
     */
    private function markEventFailed(string $eventId, int $currentRetry, string $errorMessage): void {
        $newRetry = $currentRetry + 1;
        $status = $newRetry >= MAX_RETRY ? 'failed' : 'pending';
        
        $this->db->update(
            'reward_events',
            [
                'status' => $status,
                'retry_count' => $newRetry,
                'error_message' => $errorMessage
            ],
            'event_id = ?',
            [$eventId]
        );
    }
    
    /**
     * پاکسازی eventهای قدیمی (بیشتر از 30 روز)
     */
    public function cleanupOldEvents(): void {
        $this->log("Cleaning up old events...");
        
        $deleted = $this->db->delete(
            'reward_events',
            'status = ? AND created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)',
            ['completed']
        );
        
        $this->log("Cleaned up {$deleted} old events");
    }
}

// ==================== MAIN EXECUTION ====================

// بررسی lock file (جلوگیری از اجرای همزمان)
if (file_exists(LOCK_FILE)) {
    $lockTime = filemtime(LOCK_FILE);
    $currentTime = time();
    
    // اگر بیشتر از 5 دقیقه lock بوده، پاک کن (شاید crash کرده)
    if ($currentTime - $lockTime > 300) {
        unlink(LOCK_FILE);
    } else {
        echo "Another instance is running. Exiting.\n";
        exit(0);
    }
}

// ساخت lock file
file_put_contents(LOCK_FILE, getmypid());

try {
    $processor = new RewardProcessor();
    
    // پردازش eventها
    $result = $processor->processPendingEvents();
    
    // پاکسازی eventهای قدیمی (هر 100 اجرا یک بار)
    if (rand(1, 100) === 1) {
        $processor->cleanupOldEvents();
    }
    
    // حذف lock file
    unlink(LOCK_FILE);
    
    exit($result['success'] ? 0 : 1);
    
} catch (Exception $e) {
    echo "Fatal error: " . $e->getMessage() . "\n";
    
    // حذف lock file
    if (file_exists(LOCK_FILE)) {
        unlink(LOCK_FILE);
    }
    
    exit(1);
}
