<?php
/**
 * MLM API Router
 * RESTful endpoints برای React frontend
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/UserService.php';
require_once __DIR__ . '/../includes/TreeService.php';
require_once __DIR__ . '/../includes/RewardService.php';
require_once __DIR__ . '/../includes/PaymentService.php';
require_once __DIR__ . '/../includes/WithdrawalService.php';
require_once __DIR__ . '/../includes/NotificationService.php';

// Error handling
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

// Parse request
$method = $_SERVER['REQUEST_METHOD'];
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$path = str_replace('/api', '', $path); // حذف prefix
$segments = array_filter(explode('/', $path));

// Get request body
$input = json_decode(file_get_contents('php://input'), true) ?? [];

// Merge with $_POST برای form-data
$input = array_merge($_POST, $input);

// Initialize services
$userService = new UserService();
$treeService = new TreeService();
$rewardService = new RewardService();
$paymentService = new PaymentService();
$withdrawalService = new WithdrawalService();
$notificationService = new NotificationService();

try {
    // ==================== ROUTES ====================
    
    // POST /api/register
    if ($method === 'POST' && $segments[1] === 'register') {
        $result = $userService->register($input);
        echo json_encode($result);
        http_response_code($result['success'] ? 201 : 400);
        exit();
    }
    
    // POST /api/login
    if ($method === 'POST' && $segments[1] === 'login') {
        $result = $userService->login($input['email'] ?? '', $input['password'] ?? '');
        echo json_encode($result);
        http_response_code($result['success'] ? 200 : 401);
        exit();
    }
    
    // POST /api/activate
    if ($method === 'POST' && $segments[1] === 'activate') {
        $result = $userService->activate($input['user_id'] ?? '', $input['payment_id'] ?? '');
        echo json_encode($result);
        http_response_code($result['success'] ? 200 : 400);
        exit();
    }
    
    // GET /api/user/{user_id}
    if ($method === 'GET' && $segments[1] === 'user' && isset($segments[2])) {
        $userId = $segments[2];
        $user = $userService->getUserInfo($userId);
        
        if ($user) {
            echo json_encode(['success' => true, 'data' => $user]);
            http_response_code(200);
        } else {
            echo json_encode(['success' => false, 'error' => 'User not found']);
            http_response_code(404);
        }
        exit();
    }
    
    // GET /api/tree/{user_id}?depth=3
    if ($method === 'GET' && $segments[1] === 'tree' && isset($segments[2])) {
        $userId = $segments[2];
        $maxDepth = isset($_GET['depth']) ? (int)$_GET['depth'] : 3;
        
        $tree = $treeService->buildTree($userId, $maxDepth);
        echo json_encode(['success' => true, 'data' => $tree]);
        http_response_code(200);
        exit();
    }
    
    // GET /api/tree-stats/{user_id}
    if ($method === 'GET' && $segments[1] === 'tree-stats' && isset($segments[2])) {
        $userId = $segments[2];
        $stats = $treeService->getTreeStats($userId);
        echo json_encode(['success' => true, 'data' => $stats]);
        http_response_code(200);
        exit();
    }
    
    // GET /api/reward-stats/{user_id}
    if ($method === 'GET' && $segments[1] === 'reward-stats' && isset($segments[2])) {
        $userId = $segments[2];
        $stats = $rewardService->getUserRewardStats($userId);
        echo json_encode(['success' => true, 'data' => $stats]);
        http_response_code(200);
        exit();
    }
    
    // GET /api/transactions/{user_id}?page=1&limit=50
    if ($method === 'GET' && $segments[1] === 'transactions' && isset($segments[2])) {
        $userId = $segments[2];
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = ($page - 1) * $limit;
        
        $db = Database::getInstance();
        $transactions = $db->fetchAll(
            "SELECT 
                transaction_id,
                source_user_id,
                transaction_type,
                amount,
                status,
                level,
                depth_diff,
                description,
                created_at,
                processed_at
             FROM transactions
             WHERE user_id = ?
             ORDER BY created_at DESC
             LIMIT ? OFFSET ?",
            [$userId, $limit, $offset]
        );
        
        $total = $db->fetchOne(
            "SELECT COUNT(*) as count FROM transactions WHERE user_id = ?",
            [$userId]
        )['count'];
        
        echo json_encode([
            'success' => true,
            'data' => $transactions,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ]);
        http_response_code(200);
        exit();
    }
    
    // POST /api/payment/create
    if ($method === 'POST' && $segments[1] === 'payment' && $segments[2] === 'create') {
        $db = Database::getInstance();
        
        $paymentId = 'PAY_' . strtoupper(uniqid());
        
        $db->insert('payments', [
            'payment_id' => $paymentId,
            'user_id' => $input['user_id'],
            'amount' => $input['amount'] ?? 100.0,
            'currency' => $input['currency'] ?? 'USDT',
            'network' => $input['network'] ?? 'TRC20',
            'wallet_address' => $input['wallet_address'],
            'status' => 'pending'
        ]);
        
        echo json_encode([
            'success' => true,
            'payment_id' => $paymentId,
            'message' => 'Payment created. Please send funds to activate.'
        ]);
        http_response_code(201);
        exit();
    }
    
    // POST /api/payment/verify
    if ($method === 'POST' && $segments[1] === 'payment' && $segments[2] === 'verify') {
        $result = $paymentService->verifyPayment($input['payment_id'] ?? '', $input['tx_hash'] ?? '');
        echo json_encode($result);
        http_response_code($result['success'] ? 200 : 400);
        exit();
    }
    
    // POST /api/withdrawal/request
    if ($method === 'POST' && $segments[1] === 'withdrawal' && $segments[2] === 'request') {
        $result = $withdrawalService->requestWithdrawal(
            $input['user_id'] ?? '',
            (float)($input['amount'] ?? 0),
            $input['wallet_address'] ?? '',
            $input['network'] ?? 'BSC'
        );
        echo json_encode($result);
        http_response_code($result['success'] ? 200 : 400);
        exit();
    }
    
    // GET /api/withdrawal/list/{user_id}
    if ($method === 'GET' && $segments[1] === 'withdrawal' && $segments[2] === 'list' && isset($segments[3])) {
        $userId = $segments[3];
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        
        $result = $withdrawalService->getUserWithdrawals($userId, $page);
        echo json_encode(['success' => true, 'data' => $result]);
        http_response_code(200);
        exit();
    }
    
    // POST /api/payment/confirm
    if ($method === 'POST' && $segments[1] === 'payment' && $segments[2] === 'confirm') {
        $db = Database::getInstance();
        
        $db->update(
            'payments',
            [
                'status' => 'confirmed',
                'tx_hash' => $input['tx_hash'] ?? null,
                'confirmed_at' => date('Y-m-d H:i:s')
            ],
            'payment_id = ?',
            [$input['payment_id']]
        );
        
        echo json_encode([
            'success' => true,
            'message' => 'Payment confirmed'
        ]);
        http_response_code(200);
        exit();
    }
    
    // GET /api/referral-link/{user_id}
    if ($method === 'GET' && $segments[1] === 'referral-link' && isset($segments[2])) {
        $userId = $segments[2];
        $db = Database::getInstance();
        
        $user = $db->fetchOne(
            "SELECT referral_code FROM users WHERE user_id = ?",
            [$userId]
        );
        
        if ($user) {
            $baseUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
            $referralLink = "{$baseUrl}/register?ref={$user['referral_code']}";
            
            echo json_encode([
                'success' => true,
                'referral_code' => $user['referral_code'],
                'referral_link' => $referralLink
            ]);
            http_response_code(200);
        } else {
            echo json_encode(['success' => false, 'error' => 'User not found']);
            http_response_code(404);
        }
        exit();
    }
    
    // Default 404
    echo json_encode([
        'success' => false,
        'error' => 'Endpoint not found',
        'path' => $path
    ]);
    http_response_code(404);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'trace' => $_ENV['APP_DEBUG'] ?? false ? $e->getTraceAsString() : null
    ]);
    http_response_code(500);
}
