<?php
session_start();
require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/UserService.php';
require_once __DIR__ . '/../includes/TreeService.php';
require_once __DIR__ . '/../includes/RewardService.php';

// بررسی ادمین
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

$db = Database::getInstance();
$userService = new UserService();
$treeService = new TreeService();
$rewardService = new RewardService();

$message = '';
$messageType = '';

// فعال‌سازی کاربر
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['activate_user'])) {
    $userId = $_POST['user_id'];
    
    try {
        // فعال‌سازی
        $result = $userService->activateUser($userId);
        
        if ($result['success']) {
            $message = "کاربر با موفقیت فعال شد!";
            $messageType = 'success';
        } else {
            $message = "خطا: " . $result['error'];
            $messageType = 'error';
        }
    } catch (Exception $e) {
        $message = "خطا: " . $e->getMessage();
        $messageType = 'error';
    }
}

// ساخت کاربران تستی
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_test_users'])) {
    $count = (int)$_POST['count'];
    $autoActivate = isset($_POST['auto_activate']);
    
    try {
        $created = [];
        for ($i = 0; $i < $count; $i++) {
            $testEmail = "test_" . uniqid() . "@test.com";
            $testPassword = "test123";
            
            // ثبت‌نام
            $registerResult = $userService->register($testEmail, $testPassword, 'SEED');
            
            if ($registerResult['success']) {
                $testUserId = $registerResult['user_id'];
                
                // فعال‌سازی خودکار
                if ($autoActivate) {
                    $activateResult = $userService->activateUser($testUserId);
                    if ($activateResult['success']) {
                        $created[] = $testEmail . " (فعال)";
                    } else {
                        $created[] = $testEmail . " (خطا در فعال‌سازی)";
                    }
                } else {
                    $created[] = $testEmail . " (pending)";
                }
            }
        }
        
        $message = count($created) . " کاربر تستی ایجاد شد:\n" . implode("\n", $created);
        $messageType = 'success';
        
    } catch (Exception $e) {
        $message = "خطا: " . $e->getMessage();
        $messageType = 'error';
    }
}

// دریافت لیست کاربران pending
$pendingUsers = $db->fetchAll(
    "SELECT user_id, email, referral_code, created_at 
     FROM users 
     WHERE status = 'pending' 
     ORDER BY created_at DESC 
     LIMIT 50"
);

// دریافت لیست کاربران active
$activeUsers = $db->fetchAll(
    "SELECT user_id, email, referral_code, depth, total_points, created_at, activated_at
     FROM users 
     WHERE status = 'active' AND user_id NOT IN ('seed', 'admin')
     ORDER BY activated_at DESC 
     LIMIT 50"
);

// آمار
$stats = [
    'total' => $db->fetchOne("SELECT COUNT(*) as count FROM users WHERE user_id NOT IN ('seed', 'admin')")['count'],
    'active' => $db->fetchOne("SELECT COUNT(*) as count FROM users WHERE status = 'active' AND user_id NOT IN ('seed', 'admin')")['count'],
    'pending' => $db->fetchOne("SELECT COUNT(*) as count FROM users WHERE status = 'pending'")['count']
];
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت کاربران - پنل ادمین</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; font-family: Tahoma, Arial; }
        body { background: #f5f5f5; padding: 20px; }
        .container { max-width: 1400px; margin: 0 auto; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 10px; margin-bottom: 20px; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px; }
        .stat-card { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .stat-card h3 { color: #666; font-size: 14px; margin-bottom: 10px; }
        .stat-card .value { font-size: 32px; font-weight: bold; color: #667eea; }
        .card { background: white; padding: 25px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); margin-bottom: 20px; }
        .card h2 { margin-bottom: 20px; color: #333; border-bottom: 2px solid #667eea; padding-bottom: 10px; }
        .message { padding: 15px; border-radius: 5px; margin-bottom: 20px; white-space: pre-line; }
        .message.success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message.error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        table th { background: #f8f9fa; padding: 12px; text-align: right; border-bottom: 2px solid #dee2e6; font-size: 13px; }
        table td { padding: 10px; border-bottom: 1px solid #eee; font-size: 13px; }
        .btn { padding: 8px 16px; border: none; border-radius: 5px; cursor: pointer; font-size: 13px; text-decoration: none; display: inline-block; }
        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-danger { background: #dc3545; color: white; }
        .btn:hover { opacity: 0.9; }
        .badge { padding: 4px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; }
        .badge-success { background: #d4edda; color: #155724; }
        .badge-warning { background: #fff3cd; color: #856404; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        .form-group input, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 14px; }
        .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
        .checkbox-label { display: flex; align-items: center; gap: 8px; }
        .checkbox-label input { width: auto; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔧 مدیریت کاربران</h1>
            <p>فعال‌سازی، ساخت کاربران تستی و مدیریت</p>
        </div>

        <?php if ($message): ?>
            <div class="message <?= $messageType ?>">
                <?= htmlspecialchars($message) ?>
            </div>
        <?php endif; ?>

        <!-- آمار -->
        <div class="stats">
            <div class="stat-card">
                <h3>کل کاربران</h3>
                <div class="value"><?= $stats['total'] ?></div>
            </div>
            <div class="stat-card">
                <h3>فعال</h3>
                <div class="value"><?= $stats['active'] ?></div>
            </div>
            <div class="stat-card">
                <h3>در انتظار</h3>
                <div class="value"><?= $stats['pending'] ?></div>
            </div>
        </div>

        <!-- ساخت کاربران تستی -->
        <div class="card">
            <h2>➕ ساخت کاربران تستی</h2>
            <form method="POST">
                <div class="form-row">
                    <div class="form-group">
                        <label>تعداد کاربران:</label>
                        <input type="number" name="count" value="10" min="1" max="100" required>
                    </div>
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="auto_activate" checked>
                            <span>فعال‌سازی خودکار</span>
                        </label>
                    </div>
                </div>
                <button type="submit" name="create_test_users" class="btn btn-primary">
                    ساخت کاربران تستی
                </button>
            </form>
        </div>

        <!-- کاربران در انتظار فعال‌سازی -->
        <?php if (count($pendingUsers) > 0): ?>
        <div class="card">
            <h2>⏳ کاربران در انتظار فعال‌سازی (<?= count($pendingUsers) ?>)</h2>
            <table>
                <thead>
                    <tr>
                        <th>ایمیل</th>
                        <th>کد معرفی</th>
                        <th>تاریخ ثبت‌نام</th>
                        <th>عملیات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pendingUsers as $user): ?>
                    <tr>
                        <td><?= htmlspecialchars($user['email']) ?></td>
                        <td><?= htmlspecialchars($user['referral_code']) ?></td>
                        <td><?= $user['created_at'] ?></td>
                        <td>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="user_id" value="<?= $user['user_id'] ?>">
                                <button type="submit" name="activate_user" class="btn btn-success">
                                    ✅ فعال‌سازی
                                </button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>

        <!-- کاربران فعال -->
        <div class="card">
            <h2>✅ کاربران فعال (<?= count($activeUsers) ?>)</h2>
            <table>
                <thead>
                    <tr>
                        <th>ایمیل</th>
                        <th>کد معرفی</th>
                        <th>سطح</th>
                        <th>امتیاز</th>
                        <th>تاریخ فعال‌سازی</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($activeUsers as $user): ?>
                    <tr>
                        <td><?= htmlspecialchars($user['email']) ?></td>
                        <td><?= htmlspecialchars($user['referral_code']) ?></td>
                        <td><?= $user['depth'] ?></td>
                        <td><?= number_format($user['total_points'], 2) ?></td>
                        <td><?= $user['activated_at'] ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div style="text-align: center; margin-top: 30px;">
            <a href="index.php" class="btn btn-primary">بازگشت به داشبورد</a>
        </div>
    </div>
</body>
</html>
