import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from './ui/dialog';
import { MessageSquare, Send, Inbox, User, Bell, Trash2 } from 'lucide-react';

const API = process.env.REACT_APP_BACKEND_URL;

export default function MessagingSystem({ userType = 'user' }) {
  const [messages, setMessages] = useState([]);
  const [selectedMessage, setSelectedMessage] = useState(null);
  const [replyContent, setReplyContent] = useState('');
  const [unreadCount, setUnreadCount] = useState(0);
  
  // فرم پیام جدید
  const [newMessage, setNewMessage] = useState({
    subject: '',
    content: '',
    recipient_id: ''
  });

  const getAuthHeaders = () => ({
    Authorization: `Bearer ${localStorage.getItem('token')}`
  });

  useEffect(() => {
    fetchMessages();
    if (userType === 'user') {
      fetchUnreadCount();
    }
  }, [userType]);

  const fetchMessages = async () => {
    try {
      const headers = getAuthHeaders();
      const endpoint = userType === 'admin' 
        ? `${API}/api/admin/messages` 
        : `${API}/api/user/messages`;
      
      const response = await axios.get(endpoint, { headers });
      setMessages(response.data);
    } catch (error) {
      console.error(error);
    }
  };

  const fetchUnreadCount = async () => {
    try {
      const headers = getAuthHeaders();
      const response = await axios.get(`${API}/api/user/messages/unread-count`, { headers });
      setUnreadCount(response.data.unread_count);
    } catch (error) {
      console.error(error);
    }
  };

  const sendMessage = async (e) => {
    e.preventDefault();
    
    if (!newMessage.subject || !newMessage.content) {
      toast.error('لطفاً موضوع و متن پیام را وارد کنید');
      return;
    }

    try {
      const headers = getAuthHeaders();
      const endpoint = userType === 'admin' 
        ? `${API}/api/admin/messages` 
        : `${API}/api/user/messages`;
      
      await axios.post(endpoint, newMessage, { headers });
      
      toast.success('✅ پیام ارسال شد');
      setNewMessage({ subject: '', content: '', recipient_id: '' });
      fetchMessages();
    } catch (error) {
      console.error(error);
      toast.error('خطا در ارسال پیام');
    }
  };

  const sendReply = async (messageId) => {
    if (!replyContent) {
      toast.error('لطفاً متن پاسخ را وارد کنید');
      return;
    }

    try {
      const headers = getAuthHeaders();
      await axios.post(
        `${API}/api/messages/${messageId}/reply`,
        { content: replyContent },
        { headers }
      );
      
      toast.success('✅ پاسخ ارسال شد');
      setReplyContent('');
      fetchMessages();
    } catch (error) {
      console.error(error);
      toast.error('خطا در ارسال پاسخ');
    }
  };

  const markAsRead = async (messageId) => {
    try {
      const headers = getAuthHeaders();
      await axios.put(`${API}/api/messages/${messageId}/read`, {}, { headers });
      fetchMessages();
      if (userType === 'user') {
        fetchUnreadCount();
      }
    } catch (error) {
      console.error(error);
    }
  };

  const deleteMessage = async (messageId) => {
    if (!confirm('آیا از حذف این پیام اطمینان دارید؟')) return;
    
    try {
      const headers = getAuthHeaders();
      await axios.delete(`${API}/api/admin/messages/${messageId}`, { headers });
      toast.success('✅ پیام حذف شد');
      fetchMessages();
    } catch (error) {
      toast.error('خطا در حذف پیام');
    }
  };

  return (
    <div className="p-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <MessageSquare className="w-6 h-6" />
              سیستم پیام‌رسانی
            </div>
            {userType === 'user' && unreadCount > 0 && (
              <span className="bg-red-500 text-white px-3 py-1 rounded-full text-sm">
                {unreadCount} پیام خوانده نشده
              </span>
            )}
          </CardTitle>
        </CardHeader>
        <CardContent>
          <Tabs defaultValue="inbox">
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="inbox">
                <Inbox className="w-4 h-4 ml-2" />
                صندوق ورودی
              </TabsTrigger>
              <TabsTrigger value="send">
                <Send className="w-4 h-4 ml-2" />
                ارسال پیام
              </TabsTrigger>
            </TabsList>

            {/* صندوق ورودی */}
            <TabsContent value="inbox" className="space-y-3">
              {messages.length === 0 ? (
                <div className="text-center text-gray-500 py-8">
                  پیامی وجود ندارد
                </div>
              ) : (
                messages.map((msg) => (
                  <Dialog key={msg.message_id}>
                    <DialogTrigger asChild>
                      <div
                        onClick={() => {
                          setSelectedMessage(msg);
                          if (msg.status === 'unread') {
                            markAsRead(msg.message_id);
                          }
                        }}
                        className={`border-2 rounded-lg p-4 cursor-pointer hover:shadow-lg transition ${
                          msg.status === 'unread' 
                            ? 'bg-blue-50 border-blue-300' 
                            : 'bg-white border-gray-200'
                        }`}
                      >
                        <div className="flex justify-between items-start">
                          <div className="flex-1">
                            <div className="flex items-center gap-2 mb-1">
                              {msg.status === 'unread' && (
                                <Bell className="w-4 h-4 text-blue-600" />
                              )}
                              <span className="font-semibold">{msg.subject}</span>
                            </div>
                            <div className="text-sm text-gray-600">
                              از: {msg.sender_email === 'admin@mlm.com' ? 'ادمین' : (msg.sender_email || 'ادمین')}
                            </div>
                            <div className="text-xs text-gray-500 mt-1">
                              {new Date(msg.created_at).toLocaleDateString('fa-IR')}
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            {msg.recipient_type === 'broadcast' && (
                              <span className="bg-purple-100 text-purple-700 px-2 py-1 rounded text-xs">
                                عمومی
                              </span>
                            )}
                            {userType === 'admin' && (
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={(e) => {
                                  e.stopPropagation();
                                  deleteMessage(msg.message_id);
                                }}
                                className="text-red-600 hover:text-red-800 hover:bg-red-50"
                              >
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            )}
                          </div>
                        </div>
                      </div>
                    </DialogTrigger>
                    
                    <DialogContent className="max-w-2xl">
                      <DialogHeader>
                        <DialogTitle>{msg.subject}</DialogTitle>
                      </DialogHeader>
                      
                      <div className="space-y-4">
                        <div className="bg-gray-50 p-4 rounded-lg">
                          <div className="text-sm text-gray-600 mb-2">
                            از: {msg.sender_email === 'admin@mlm.com' ? 'ادمین' : (msg.sender_email || 'ادمین')}
                          </div>
                          <div className="whitespace-pre-wrap">{msg.content}</div>
                        </div>
                        
                        {/* نمایش پاسخ‌ها */}
                        {msg.replies && msg.replies.length > 0 && (
                          <div className="space-y-2">
                            <h4 className="font-semibold">پاسخ‌ها:</h4>
                            {msg.replies.map((reply) => (
                              <div key={reply.reply_id} className="bg-blue-50 p-3 rounded-lg border-r-4 border-blue-400">
                                <div className="text-xs text-gray-600 mb-1">
                                  {reply.sender_email} - {new Date(reply.created_at).toLocaleDateString('fa-IR')}
                                </div>
                                <div className="text-sm">{reply.content}</div>
                              </div>
                            ))}
                          </div>
                        )}
                        
                        {/* فرم پاسخ */}
                        <div className="border-t pt-4">
                          <Label>پاسخ:</Label>
                          <Textarea
                            value={replyContent}
                            onChange={(e) => setReplyContent(e.target.value)}
                            placeholder="متن پاسخ خود را بنویسید..."
                            className="mt-2"
                            rows={3}
                          />
                          <Button
                            onClick={() => sendReply(msg.message_id)}
                            className="mt-2 w-full"
                          >
                            ارسال پاسخ
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                ))
              )}
            </TabsContent>

            {/* ارسال پیام جدید */}
            <TabsContent value="send">
              <form onSubmit={sendMessage} className="space-y-4">
                {userType === 'admin' && (
                  <div>
                    <Label>گیرنده (اختیاری - خالی بگذارید برای پیام عمومی)</Label>
                    <Input
                      value={newMessage.recipient_id}
                      onChange={(e) => setNewMessage({...newMessage, recipient_id: e.target.value})}
                      placeholder="User ID گیرنده"
                    />
                    <p className="text-xs text-gray-500 mt-1">
                      برای پیام عمومی (Broadcast)، این فیلد را خالی بگذارید
                    </p>
                  </div>
                )}
                
                <div>
                  <Label>موضوع *</Label>
                  <Input
                    value={newMessage.subject}
                    onChange={(e) => setNewMessage({...newMessage, subject: e.target.value})}
                    placeholder="موضوع پیام"
                    required
                  />
                </div>
                
                <div>
                  <Label>متن پیام *</Label>
                  <Textarea
                    value={newMessage.content}
                    onChange={(e) => setNewMessage({...newMessage, content: e.target.value})}
                    placeholder="متن پیام خود را بنویسید..."
                    rows={6}
                    required
                  />
                </div>
                
                <Button type="submit" className="w-full">
                  <Send className="w-4 h-4 ml-2" />
                  ارسال پیام
                </Button>
              </form>
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
}
