import { useState, useEffect } from "react";
import axios from "axios";
import { Card, CardHeader, CardTitle, CardDescription, CardContent } from "./ui/card";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Switch } from "./ui/switch";
import { Textarea } from "./ui/textarea";
import { toast } from "sonner";
import { MessageSquare, Save } from "lucide-react";
import AdminTelegramBroadcast from "./AdminTelegramBroadcast";

const API = process.env.REACT_APP_BACKEND_URL || 'http://localhost:8001';

export default function AdminTelegramSettingsSimple() {
  const [settings, setSettings] = useState({
    enabled: false,
    bot_token: '',
    bot_username: '',
    webapp_url: '',
    welcome_message: '',
    welcome_with_ref_message: ''
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [restarting, setRestarting] = useState(false);

  // بارگذاری تنظیمات هنگام mount
  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      console.log('🔍 دریافت تنظیمات...');
      
      const response = await axios.get(`${API}/api/admin/telegram/config`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      
      console.log('📥 تنظیمات دریافت شد:', response.data);
      
      setSettings({
        enabled: response.data.enabled || false,
        bot_token: response.data.bot_token || '',
        bot_username: response.data.bot_username || '',
        webapp_url: response.data.webapp_url || '',
        welcome_message: response.data.welcome_message || '',
        welcome_with_ref_message: response.data.welcome_with_ref_message || ''
      });
      
      console.log('✅ State به‌روز شد');
    } catch (error) {
      console.error('❌ خطا در دریافت:', error);
      toast.error('خطا در دریافت تنظیمات');
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      const token = localStorage.getItem('token');
      console.log('💾 ذخیره تنظیمات:', settings);
      
      const response = await axios.post(
        `${API}/api/admin/telegram/config`,
        settings,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      
      console.log('📤 پاسخ:', response.data);
      
      if (response.data.success) {
        toast.success('✅ تنظیمات ذخیره شد!', { duration: 4000 });
        // بارگذاری مجدد
        await loadSettings();
      } else {
        toast.error('❌ خطا در ذخیره');
      }
    } catch (error) {
      console.error('❌ خطا:', error);
      toast.error('خطا در ذخیره تنظیمات');
    } finally {
      setSaving(false);
    }
  };

  const restartBot = async () => {
    setRestarting(true);
    try {
      const token = localStorage.getItem('token');
      const response = await axios.post(
        `${API}/api/admin/telegram/restart`,
        {},
        { headers: { Authorization: `Bearer ${token}` } }
      );
      
      if (response.data.success) {
        toast.success('✅ Bot راه‌اندازی مجدد شد!', { duration: 4000 });
      } else {
        toast.error('❌ خطا در راه‌اندازی مجدد');
      }
    } catch (error) {
      console.error('❌ خطا:', error);
      toast.error('خطا در راه‌اندازی مجدد');
    } finally {
      setRestarting(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-12">
        <div className="text-center space-y-4">
          <div className="w-12 h-12 mx-auto border-4 border-blue-600 border-t-transparent rounded-full animate-spin"></div>
          <p className="text-gray-600">در حال بارگذاری...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <Card className="shadow-xl">
        <CardHeader className="bg-gradient-to-r from-blue-50 to-cyan-50">
          <CardTitle className="flex items-center gap-2 text-2xl">
            <MessageSquare className="w-6 h-6 text-blue-600" />
            تنظیمات ربات تلگرام
          </CardTitle>
          <CardDescription>
            پیکربندی ربات تلگرام
          </CardDescription>
        </CardHeader>
        <CardContent className="pt-6 space-y-6">
          {/* فعال/غیرفعال */}
          <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
            <div>
              <Label className="text-base font-semibold">فعال‌سازی ربات</Label>
              <p className="text-sm text-gray-600 mt-1">ربات روشن یا خاموش</p>
            </div>
            <Switch
              checked={settings.enabled}
              onCheckedChange={(checked) => setSettings({...settings, enabled: checked})}
            />
          </div>

          {/* Bot Token */}
          <div className="space-y-2">
            <Label className="text-base font-semibold">Bot Token</Label>
            <Input
              type="password"
              value={settings.bot_token}
              onChange={(e) => setSettings({...settings, bot_token: e.target.value})}
              placeholder="1234567890:ABCdef..."
              disabled={!settings.enabled}
            />
          </div>

          {/* Bot Username */}
          <div className="space-y-2">
            <Label className="text-base font-semibold">Bot Username</Label>
            <div className="flex gap-2">
              <div className="flex items-center px-3 bg-gray-100 rounded-r-lg border">
                <span className="text-gray-600">@</span>
              </div>
              <Input
                value={settings.bot_username}
                onChange={(e) => setSettings({...settings, bot_username: e.target.value})}
                placeholder="MyBot"
                className="rounded-r-none"
                disabled={!settings.enabled}
              />
            </div>
          </div>

          {/* WebApp URL */}
          <div className="space-y-2">
            <Label className="text-base font-semibold">آدرس سایت</Label>
            <Input
              value={settings.webapp_url}
              onChange={(e) => setSettings({...settings, webapp_url: e.target.value})}
              placeholder="https://yoursite.com"
              disabled={!settings.enabled}
            />
          </div>

          {/* پیام عادی */}
          <div className="space-y-2">
            <Label className="text-base font-semibold">پیام خوش‌آمدگویی (عادی)</Label>
            <Textarea
              value={settings.welcome_message}
              onChange={(e) => setSettings({...settings, welcome_message: e.target.value})}
              placeholder="سلام {user_name}!"
              rows={5}
              disabled={!settings.enabled}
              className="font-mono text-sm"
            />
            <p className="text-xs text-gray-500">
              💡 متغیر: <code className="bg-gray-200 px-1 rounded">{'{user_name}'}</code>
            </p>
          </div>

          {/* پیام با رفرال */}
          <div className="space-y-2">
            <Label className="text-base font-semibold">پیام خوش‌آمدگویی (با رفرال)</Label>
            <Textarea
              value={settings.welcome_with_ref_message}
              onChange={(e) => setSettings({...settings, welcome_with_ref_message: e.target.value})}
              placeholder="سلام {user_name}! کد معرف: {referral_code}"
              rows={5}
              disabled={!settings.enabled}
              className="font-mono text-sm"
            />
            <p className="text-xs text-gray-500">
              💡 متغیرها: <code className="bg-gray-200 px-1 rounded">{'{user_name}'}</code>, <code className="bg-gray-200 px-1 rounded">{'{referral_code}'}</code>
            </p>
          </div>

          {/* دکمه‌ها */}
          <div className="grid grid-cols-2 gap-4 pt-4">
            <Button
              onClick={saveSettings}
              disabled={saving}
              className="bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 h-12 text-base font-bold"
            >
              {saving ? (
                <>
                  <div className="w-5 h-5 ml-2 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                  در حال ذخیره...
                </>
              ) : (
                <>
                  <Save className="w-5 h-5 ml-2" />
                  ذخیره تنظیمات
                </>
              )}
            </Button>
            
            <Button
              onClick={restartBot}
              disabled={restarting}
              className="bg-gradient-to-r from-orange-600 to-red-600 hover:from-orange-700 hover:to-red-700 h-12 text-base font-bold"
            >
              {restarting ? (
                <>
                  <div className="w-5 h-5 ml-2 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                  در حال راه‌اندازی...
                </>
              ) : (
                <>
                  <svg className="w-5 h-5 ml-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                  </svg>
                  راه‌اندازی مجدد Bot
                </>
              )}
            </Button>
          </div>

          {/* نکته */}
          <div className="p-4 bg-yellow-50 border-2 border-yellow-300 rounded-lg">
            <p className="text-sm text-yellow-900 font-semibold">
              💡 <strong>مهم:</strong> بعد از تغییر پیام‌ها، حتماً روی "راه‌اندازی مجدد Bot" کلیک کنید!
            </p>
          </div>
        </CardContent>
      </Card>

      {/* کامپوننت ارسال پیام عمومی */}
      <div className="mt-6">
        <AdminTelegramBroadcast />
      </div>
    </div>
  );
}
