"""Tree Statistics Service - بهینه‌سازی شده برای تعداد بالای کاربر"""

from typing import Dict, List
import logging
import asyncio
from .path_manager import PathManager

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class TreeStatsService:
    """سرویس آمار درخت - بهینه‌سازی شده"""
    
    @staticmethod
    async def update_descendants_count(placement_path: str, new_user_depth: int, increment: int = 1):
        """
        به‌روزرسانی تعداد descendants و max_descendant_depth
        
        بهینه‌سازی:
        - تک query bulk update
        - O(1) complexity
        
        Args:
            placement_path: path کاربر جدید
            new_user_depth: depth کاربر جدید
            increment: مقدار افزایش
        """
        ancestor_ids = PathManager.extract_ancestors(placement_path)
        
        # seed همیشه ancestor است
        if "seed" not in ancestor_ids and placement_path != "":
            ancestor_ids.insert(0, "seed")
        
        if not ancestor_ids:
            return
        
        from pymongo import UpdateOne
        
        # 🚀 بهینه‌سازی: دریافت سریع با projection محدود
        ancestors_data = await _db.users.find(
            {"user_id": {"$in": ancestor_ids}},
            {"_id": 0, "user_id": 1, "depth": 1}
        ).to_list(len(ancestor_ids))
        
        # ساخت bulk operations
        bulk_ops = []
        
        for ancestor in ancestors_data:
            ancestor_depth = ancestor["depth"]
            depth_diff = new_user_depth - ancestor_depth
            
            bulk_ops.append(
                UpdateOne(
                    {"user_id": ancestor["user_id"]},
                    {
                        "$inc": {"total_descendants": increment},
                        "$max": {"max_descendant_depth": depth_diff}
                    }
                )
            )
        
        if bulk_ops:
            await _db.users.bulk_write(bulk_ops, ordered=False)
            logger.info(f"✅ Updated stats for {len(ancestor_ids)} ancestors")
    
    @staticmethod
    async def check_tree_completion(placement_path: str):
        """
        🚀 OPTIMIZED: بررسی تکمیل هرم با حداکثر سرعت
        
        بهینه‌سازی‌ها:
        1. Parallel count queries با asyncio.gather
        2. فقط incomplete ancestors چک می‌شن
        3. Early exit اگه ancestor نداشته باشیم
        4. Bulk update atomic
        
        هرم کامل = دقیقاً 729 نفر فعال در depth_diff=6
        
        Args:
            placement_path: path کاربر جدید
        """
        ancestor_ids = PathManager.extract_ancestors(placement_path)
        
        # seed همیشه ancestor است
        if "seed" not in ancestor_ids and placement_path != "":
            ancestor_ids.insert(0, "seed")
        
        if not ancestor_ids:
            return
        
        # 🚀 بهینه‌سازی 1: فقط incomplete ancestors
        ancestors_data = await _db.users.find(
            {
                "user_id": {"$in": ancestor_ids},
                "is_tree_complete": False,
                "status": "active"
            },
            {"_id": 0, "user_id": 1, "depth": 1, "placement_path": 1}
        ).to_list(len(ancestor_ids))
        
        if not ancestors_data:
            return
        
        # 🚀 بهینه‌سازی 2: Parallel count queries
        async def count_level_7(ancestor: Dict) -> tuple:
            """شمارش موازی descendants در لول 7"""
            ancestor_id = ancestor["user_id"]
            ancestor_depth = ancestor["depth"]
            ancestor_path = ancestor.get("placement_path", "")
            target_depth = ancestor_depth + 6
            
            # ساخت query بهینه
            if ancestor_path == "":
                # seed (root)
                query = {
                    "status": "active",
                    "depth": target_depth
                }
            else:
                # regex optimized با ^
                query = {
                    "status": "active",
                    "depth": target_depth,
                    "placement_path": {"$regex": f"^{ancestor_path}/"}
                }
            
            count = await _db.users.count_documents(query)
            return (ancestor_id, count)
        
        # 🚀 اجرای موازی همه count ها
        tasks = [count_level_7(ancestor) for ancestor in ancestors_data]
        results = await asyncio.gather(*tasks)
        
        # 🚀 بهینه‌سازی 3: ساخت bulk update فقط برای complete ها
        from pymongo import UpdateOne
        bulk_ops = []
        
        for ancestor_id, count in results:
            if count >= 729:
                bulk_ops.append(
                    UpdateOne(
                        {"user_id": ancestor_id},
                        {"$set": {"is_tree_complete": True}}
                    )
                )
                logger.info(f"🎉 {ancestor_id} completed! ({count}/729 in level 7)")
        
        # اجرای bulk update
        if bulk_ops:
            result = await _db.users.bulk_write(bulk_ops, ordered=False)
            logger.info(f"✅ {result.modified_count} users marked as complete")
    
    @staticmethod
    async def get_tree_for_display(user_id: str, max_depth: int = 3) -> Dict:
        """
        دریافت درخت برای نمایش
        
        بهینه‌سازی:
        - Single query با regex
        - محدودیت depth
        - Projection محدود
        
        Args:
            user_id: شناسه کاربر ریشه
            max_depth: حداکثر عمق
            
        Returns:
            درخت nested
        """
        # دریافت کاربر ریشه
        root_user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0, "user_id": 1, "email": 1, "status": 1, "depth": 1, 
             "total_points": 1, "direct_children_count": 1, "placement_path": 1}
        )
        
        if not root_user:
            return None
        
        root_path = root_user.get("placement_path", user_id)
        root_depth = root_user.get("depth", 0)
        
        # 🚀 Single query با limit
        descendants = await _db.users.find(
            {
                "placement_path": {"$regex": f"^{root_path}/"}, 
                "depth": {"$lte": root_depth + max_depth}
            },
            {"_id": 0, "user_id": 1, "email": 1, "status": 1, "depth": 1,
             "total_points": 1, "direct_children_count": 1, "placement_parent_id": 1}
        ).limit(1000).to_list(1000)
        
        # ساخت درخت در حافظه
        nodes = {root_user["user_id"]: {
            "user_id": root_user["user_id"],
            "email": root_user["email"],
            "status": root_user["status"],
            "depth": root_user["depth"],
            "total_points": root_user["total_points"],
            "direct_children_count": root_user["direct_children_count"],
            "children": []
        }}
        
        for desc in descendants:
            nodes[desc["user_id"]] = {
                "user_id": desc["user_id"],
                "email": desc["email"],
                "status": desc["status"],
                "depth": desc["depth"],
                "total_points": desc["total_points"],
                "direct_children_count": desc["direct_children_count"],
                "children": []
            }
        
        # اتصال parent-child
        for desc in descendants:
            parent_id = desc.get("placement_parent_id")
            if parent_id and parent_id in nodes:
                nodes[parent_id]["children"].append(nodes[desc["user_id"]])
        
        return nodes[root_user["user_id"]]
    
    @staticmethod
    async def get_descendants_count(user_id: str) -> Dict:
        """
        دریافت تعداد descendants
        
        بهینه‌سازی: فقط یک field می‌خونه
        
        Args:
            user_id: شناسه کاربر
            
        Returns:
            آمار descendants
        """
        user = await _db.users.find_one(
            {"user_id": user_id},
            {"_id": 0, "total_descendants": 1}
        )
        
        if not user:
            return {"total": 0}
        
        return {
            "total": user.get("total_descendants", 0)
        }
