"""Audit Logging Service - ثبت تمام رویدادهای مهم سیستم"""

from typing import Dict, Optional
import uuid
import logging
from datetime import datetime, timezone

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class AuditLogService:
    """سرویس ثبت رویدادهای سیستم برای troubleshooting و transparency"""
    
    # انواع رویدادها
    EVENT_TYPES = {
        # User events
        "USER_REGISTERED": "user_registered",
        "USER_ACTIVATED": "user_activated",
        "USER_DEACTIVATED": "user_deactivated",
        
        # Placement events
        "PLACEMENT_SUCCESS": "placement_success",
        "PLACEMENT_FAILED": "placement_failed",
        "PLACEMENT_SPILLOVER": "placement_spillover",
        
        # Reward events
        "REWARD_DIRECT": "reward_direct",
        "REWARD_LEVEL": "reward_level",
        "REWARD_BLOCKED": "reward_blocked",
        
        # Tree events
        "TREE_COMPLETE": "tree_complete",
        "TREE_INCOMPLETE": "tree_incomplete",
        
        # System events
        "DESCENDANTS_UPDATED": "descendants_updated",
        "STATS_RECALCULATED": "stats_recalculated"
    }
    
    @staticmethod
    async def log_event(
        event_type: str,
        user_id: str,
        details: Dict,
        related_user_id: Optional[str] = None,
        amount: Optional[float] = None
    ):
        """
        ثبت یک رویداد در audit log
        
        Args:
            event_type: نوع رویداد (از EVENT_TYPES)
            user_id: شناسه کاربر اصلی
            details: جزئیات رویداد (dict)
            related_user_id: شناسه کاربر مرتبط (اختیاری)
            amount: مبلغ (برای reward events)
        """
        
        audit_log = {
            "log_id": str(uuid.uuid4()),
            "event_type": event_type,
            "user_id": user_id,
            "related_user_id": related_user_id,
            "amount": amount,
            "details": details,
            "timestamp": datetime.now(timezone.utc).isoformat()
        }
        
        try:
            await _db.audit_logs.insert_one(audit_log)
            logger.debug(f"📝 Audit log: {event_type} for {user_id}")
        except Exception as e:
            # اگر audit log fail شد، سیستم باید ادامه بده
            logger.error(f"❌ Failed to create audit log: {e}")
    
    @staticmethod
    async def log_user_activation(
        user_id: str,
        placement_parent_id: str,
        placement_position: int,
        depth: int,
        placement_path: str
    ):
        """ثبت فعال‌سازی کاربر"""
        await AuditLogService.log_event(
            AuditLogService.EVENT_TYPES["USER_ACTIVATED"],
            user_id,
            {
                "placement_parent_id": placement_parent_id,
                "placement_position": placement_position,
                "depth": depth,
                "placement_path": placement_path
            }
        )
    
    @staticmethod
    async def log_placement_spillover(
        user_id: str,
        referrer_id: str,
        actual_parent_id: str,
        reason: str
    ):
        """ثبت spillover placement"""
        await AuditLogService.log_event(
            AuditLogService.EVENT_TYPES["PLACEMENT_SPILLOVER"],
            user_id,
            {
                "referrer_id": referrer_id,
                "actual_parent_id": actual_parent_id,
                "reason": reason
            }
        )
    
    @staticmethod
    async def log_reward(
        recipient_id: str,
        from_user_id: str,
        reward_type: str,
        amount: float,
        description: str
    ):
        """ثبت پاداش"""
        event_type = (
            AuditLogService.EVENT_TYPES["REWARD_DIRECT"] 
            if reward_type == "direct" 
            else AuditLogService.EVENT_TYPES["REWARD_LEVEL"]
        )
        
        await AuditLogService.log_event(
            event_type,
            recipient_id,
            {
                "reward_type": reward_type,
                "description": description
            },
            related_user_id=from_user_id,
            amount=amount
        )
    
    @staticmethod
    async def log_reward_blocked(
        user_id: str,
        from_user_id: str,
        reward_type: str,
        amount: float,
        reason: str
    ):
        """ثبت پاداش مسدود شده"""
        await AuditLogService.log_event(
            AuditLogService.EVENT_TYPES["REWARD_BLOCKED"],
            user_id,
            {
                "reward_type": reward_type,
                "blocked_amount": amount,
                "reason": reason
            },
            related_user_id=from_user_id,
            amount=0
        )
    
    @staticmethod
    async def log_tree_completion(user_id: str, max_descendant_depth: int):
        """ثبت تکمیل هرم"""
        await AuditLogService.log_event(
            AuditLogService.EVENT_TYPES["TREE_COMPLETE"],
            user_id,
            {
                "max_descendant_depth": max_descendant_depth,
                "message": "هرم 7 سطحی کامل شد"
            }
        )
    
    @staticmethod
    async def log_descendants_update(
        user_id: str,
        old_count: int,
        new_count: int,
        old_max_depth: int,
        new_max_depth: int
    ):
        """ثبت به‌روزرسانی descendants"""
        await AuditLogService.log_event(
            AuditLogService.EVENT_TYPES["DESCENDANTS_UPDATED"],
            user_id,
            {
                "old_total_descendants": old_count,
                "new_total_descendants": new_count,
                "old_max_descendant_depth": old_max_depth,
                "new_max_descendant_depth": new_max_depth
            }
        )
    
    @staticmethod
    async def get_user_audit_logs(
        user_id: str,
        event_type: Optional[str] = None,
        limit: int = 50
    ):
        """
        دریافت audit logs یک کاربر
        
        Args:
            user_id: شناسه کاربر
            event_type: نوع رویداد (اختیاری)
            limit: تعداد رکوردها
            
        Returns:
            لیست audit logs
        """
        query = {"user_id": user_id}
        
        if event_type:
            query["event_type"] = event_type
        
        logs = await _db.audit_logs.find(
            query,
            {"_id": 0}
        ).sort("timestamp", -1).limit(limit).to_list(limit)
        
        return logs
    
    @staticmethod
    async def get_reward_history(user_id: str, limit: int = 50):
        """دریافت تاریخچه پاداش‌های یک کاربر"""
        logs = await _db.audit_logs.find(
            {
                "user_id": user_id,
                "event_type": {
                    "$in": [
                        AuditLogService.EVENT_TYPES["REWARD_DIRECT"],
                        AuditLogService.EVENT_TYPES["REWARD_LEVEL"],
                        AuditLogService.EVENT_TYPES["REWARD_BLOCKED"]
                    ]
                }
            },
            {"_id": 0}
        ).sort("timestamp", -1).limit(limit).to_list(limit)
        
        return logs
