"""Messages Routes - اندپوینت‌های پیام"""

import uuid
from datetime import datetime, timezone

from fastapi import APIRouter, HTTPException, Depends, Query
from typing import Optional

from models.message import MessageCreate
from services.auth_service import get_current_user, get_admin_user
from database import get_db

router = APIRouter()


@router.post("/user/messages")
async def create_message(data: MessageCreate, current_user: dict = Depends(get_current_user)):
    """ارسال پیام جدید"""
    db = await get_db()
    
    # بررسی گیرنده
    recipient = await db.users.find_one({"user_id": data.recipient_id}, {"_id": 0, "email": 1})
    if not recipient:
        raise HTTPException(status_code=404, detail="گیرنده یافت نشد")
    
    message = {
        "message_id": str(uuid.uuid4()),
        "sender_id": current_user["user_id"],
        "sender_email": current_user["email"],
        "recipient_id": data.recipient_id,
        "recipient_email": recipient["email"],
        "subject": data.subject,
        "content": data.content,
        "is_read": False,
        "parent_id": None,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "read_at": None
    }
    
    await db.messages.insert_one(message)
    
    return {"message_id": message["message_id"], "message": "پیام ارسال شد"}


@router.get("/user/messages")
async def get_user_messages(current_user: dict = Depends(get_current_user)):
    """لیست پیام‌های کاربر"""
    db = await get_db()
    
    messages = await db.messages.find(
        {"$or": [
            {"sender_id": current_user["user_id"]},
            {"recipient_id": current_user["user_id"]}
        ]},
        {"_id": 0}
    ).sort("created_at", -1).to_list(100)
    
    return messages


@router.get("/user/messages/unread-count")
async def get_unread_count(current_user: dict = Depends(get_current_user)):
    """تعداد پیام‌های خوانده نشده"""
    db = await get_db()
    
    count = await db.messages.count_documents({
        "recipient_id": current_user["user_id"],
        "is_read": False
    })
    
    return {"unread_count": count}


@router.post("/messages/{message_id}/reply")
async def reply_to_message(message_id: str, data: MessageCreate, current_user: dict = Depends(get_current_user)):
    """پاسخ به پیام"""
    db = await get_db()
    
    original = await db.messages.find_one({"message_id": message_id})
    if not original:
        raise HTTPException(status_code=404, detail="پیام یافت نشد")
    
    # گیرنده = فرستنده پیام اصلی
    recipient_id = original["sender_id"]
    recipient = await db.users.find_one({"user_id": recipient_id}, {"_id": 0, "email": 1})
    
    reply = {
        "message_id": str(uuid.uuid4()),
        "sender_id": current_user["user_id"],
        "sender_email": current_user["email"],
        "recipient_id": recipient_id,
        "recipient_email": recipient["email"] if recipient else "",
        "subject": f"Re: {original['subject']}",
        "content": data.content,
        "is_read": False,
        "parent_id": message_id,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "read_at": None
    }
    
    await db.messages.insert_one(reply)
    
    return {"message_id": reply["message_id"], "message": "پاسخ ارسال شد"}


@router.put("/messages/{message_id}/read")
async def mark_as_read(message_id: str, current_user: dict = Depends(get_current_user)):
    """علامت‌گذاری به عنوان خوانده شده"""
    db = await get_db()
    
    await db.messages.update_one(
        {"message_id": message_id, "recipient_id": current_user["user_id"]},
        {"$set": {
            "is_read": True,
            "read_at": datetime.now(timezone.utc).isoformat()
        }}
    )
    
    return {"message": "پیام خوانده شد"}


# Admin Routes

@router.get("/admin/messages")
async def get_admin_messages(
    page: int = Query(1, ge=1),
    limit: int = Query(20, ge=1, le=100),
    admin_user: dict = Depends(get_admin_user)
):
    """لیست همه پیام‌ها"""
    db = await get_db()
    
    skip = (page - 1) * limit
    total = await db.messages.count_documents({})
    
    messages = await db.messages.find(
        {},
        {"_id": 0}
    ).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    
    return {
        "messages": messages,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit
    }


@router.post("/admin/messages")
async def admin_send_message(data: MessageCreate, admin_user: dict = Depends(get_admin_user)):
    """ارسال پیام توسط ادمین"""
    db = await get_db()
    
    recipient = await db.users.find_one({"user_id": data.recipient_id}, {"_id": 0, "email": 1})
    if not recipient:
        raise HTTPException(status_code=404, detail="گیرنده یافت نشد")
    
    message = {
        "message_id": str(uuid.uuid4()),
        "sender_id": admin_user["user_id"],
        "sender_email": admin_user["email"],
        "recipient_id": data.recipient_id,
        "recipient_email": recipient["email"],
        "subject": data.subject,
        "content": data.content,
        "is_read": False,
        "parent_id": None,
        "created_at": datetime.now(timezone.utc).isoformat(),
        "read_at": None,
        "from_admin": True
    }
    
    await db.messages.insert_one(message)
    
    return {"message_id": message["message_id"], "message": "پیام ارسال شد"}


@router.delete("/admin/messages/{message_id}")
async def delete_message(message_id: str, admin_user: dict = Depends(get_admin_user)):
    """حذف پیام"""
    db = await get_db()
    await db.messages.delete_one({"message_id": message_id})
    return {"message": "پیام حذف شد"}
