"""Admin Routes - اندپوینت‌های مدیریتی"""

import uuid
import logging
from datetime import datetime, timezone
from typing import Optional

import bcrypt
from fastapi import APIRouter, HTTPException, Depends, Query

from models.admin import TestUserCreate, PointsAdjustment, ConfigUpdate
from models.user import ChangePasswordData
from models.reward import RewardSettings
from services.auth_service import get_admin_user, hash_password
from services.user.activation import UserActivationService, set_database as set_activation_db
from services.tree.placement import PlacementService, set_database as set_placement_db
from database import get_db

router = APIRouter()


@router.get("/admin/users")
async def get_all_users(
    page: int = Query(1, ge=1),
    limit: int = Query(20, ge=1, le=100),
    status: Optional[str] = None,
    search: Optional[str] = None,
    admin_user: dict = Depends(get_admin_user)
):
    """دریافت لیست کاربران"""
    db = await get_db()
    
    # ساخت filter
    filter_query = {}
    if status:
        filter_query["status"] = status
    if search:
        filter_query["$or"] = [
            {"email": {"$regex": search, "$options": "i"}},
            {"user_id": {"$regex": search, "$options": "i"}}
        ]
    
    # شمارش کل
    total = await db.users.count_documents(filter_query)
    
    # دریافت کاربران
    skip = (page - 1) * limit
    users = await db.users.find(
        filter_query,
        {"_id": 0, "password": 0}
    ).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    
    return {
        "users": users,
        "total": total,
        "page": page,
        "pages": (total + limit - 1) // limit
    }


@router.get("/admin/users/{user_id}/report")
async def get_user_report(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """گزارش کامل کاربر"""
    db = await get_db()
    
    user = await db.users.find_one({"user_id": user_id}, {"_id": 0, "password": 0})
    if not user:
        raise HTTPException(status_code=404, detail="کاربر یافت نشد")
    
    # تراکنش‌ها
    transactions = await db.transactions.find(
        {"user_id": user_id},
        {"_id": 0}
    ).sort("timestamp", -1).limit(50).to_list(50)
    
    # زیرمجموعه‌ها
    referrals = await db.users.find(
        {"referrer_id": user_id},
        {"_id": 0, "user_id": 1, "email": 1, "status": 1, "created_at": 1}
    ).to_list(100)
    
    return {
        "user": user,
        "transactions": transactions,
        "referrals": referrals
    }


@router.get("/admin/seed/dashboard")
async def get_seed_dashboard(admin_user: dict = Depends(get_admin_user)):
    """
    Dashboard seed - نمای کاربری seed
    
    دقیقاً مثل dashboard کاربران عادی، ولی برای seed
    """
    db = await get_db()
    
    # دریافت seed
    seed = await db.users.find_one({"user_id": "seed"}, {"_id": 0, "password": 0})
    if not seed:
        raise HTTPException(status_code=404, detail="seed یافت نشد")
    
    # شمارش زیرمجموعه‌های مستقیم
    direct_referrals = await db.users.count_documents({"referrer_id": "seed", "status": "active"})
    pending_referrals = await db.users.count_documents({"referrer_id": "seed", "status": "pending"})
    
    # محاسبه سطوح (7 سطح)
    level_counts = {}
    for level in range(1, 8):
        if level == 1:
            # سطح 1 = خود seed
            count = 1
        else:
            # سطح 2 تا 7 = depth 1 تا 6
            target_depth = level - 1
            count = await db.users.count_documents({
                "depth": target_depth,
                "status": "active"
            })
        level_counts[f"level_{level}"] = count
    
    # تراکنش‌های اخیر seed
    recent_transactions = await db.transactions.find(
        {"user_id": "seed"},
        {"_id": 0}
    ).sort("timestamp", -1).limit(20).to_list(20)
    
    # لیست زیرمجموعه‌های مستقیم
    direct_children_list = await db.users.find(
        {"referrer_id": "seed"},
        {"_id": 0, "user_id": 1, "email": 1, "status": 1, "created_at": 1, "total_points": 1}
    ).sort("created_at", -1).limit(50).to_list(50)
    
    return {
        "user_info": {
            "user_id": "seed",
            "email": seed.get("email"),
            "status": seed.get("status"),
            "referral_code": seed.get("referral_code"),
            "total_points": seed.get("total_points", 0.0),
            "depth": 0,
            "created_at": seed.get("created_at")
        },
        "tree_stats": {
            "direct_children_count": seed.get("direct_children_count", 0),
            "total_descendants": seed.get("total_descendants", 0),
            "direct_referrals": direct_referrals,
            "pending_referrals": pending_referrals
        },
        "level_counts": level_counts,
        "recent_transactions": recent_transactions,
        "direct_children": direct_children_list
    }


@router.post("/admin/users/{user_id}/activate")
async def activate_user(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """فعال‌سازی کاربر"""
    db = await get_db()
    
    # Set database connections for services
    set_activation_db(db)
    set_placement_db(db)
    
    try:
        # Use the proper activation service
        activation_result = await UserActivationService.activate_user(user_id, admin_user["user_id"])
        logging.info(f"✅ User {user_id} activated by admin {admin_user['user_id']}")
        return {"message": "کاربر فعال شد", "result": activation_result}
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))
    except Exception as e:
        logging.error(f"❌ Activation failed: {e}")
        raise HTTPException(status_code=500, detail=f"خطا در فعال‌سازی: {str(e)}")


@router.post("/admin/users/{user_id}/deactivate")
async def deactivate_user(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """غیرفعال کردن کاربر"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {"status": "pending"}}
    )
    return {"message": "کاربر غیرفعال شد"}


@router.post("/admin/users/{user_id}/block-rewards")
async def block_rewards(
    user_id: str, 
    block_direct: bool = False, 
    block_level: bool = False, 
    admin_user: dict = Depends(get_admin_user)
):
    """مسدود کردن پاداش کاربر"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {
            "blocked_direct_reward": block_direct,
            "blocked_level_reward": block_level
        }}
    )
    return {"message": "وضعیت مسدودی پاداش بروزرسانی شد"}


@router.post("/admin/users/{user_id}/unblock-rewards")
async def unblock_rewards(user_id: str, admin_user: dict = Depends(get_admin_user)):
    """رفع مسدودی پاداش کاربر"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {
            "blocked_direct_reward": False,
            "blocked_level_reward": False
        }}
    )
    return {"message": "مسدودی پاداش برداشته شد"}


@router.put("/admin/users/{user_id}/password")
async def change_user_password(user_id: str, data: ChangePasswordData, admin_user: dict = Depends(get_admin_user)):
    """تغییر رمز عبور کاربر توسط ادمین"""
    db = await get_db()
    
    new_hash = hash_password(data.new_password)
    await db.users.update_one(
        {"user_id": user_id},
        {"$set": {"password": new_hash}}
    )
    return {"message": "رمز عبور تغییر یافت"}


@router.post("/admin/users/{user_id}/adjust-points")
async def adjust_points(user_id: str, data: PointsAdjustment, admin_user: dict = Depends(get_admin_user)):
    """تنظیم امتیاز کاربر"""
    db = await get_db()
    
    await db.users.update_one(
        {"user_id": user_id},
        {"$inc": {"total_points": data.amount}}
    )
    
    # ثبت تراکنش
    await db.transactions.insert_one({
        "transaction_id": str(uuid.uuid4()),
        "user_id": user_id,
        "from_user_id": admin_user["user_id"],
        "reward_type": "admin_adjustment",
        "amount": data.amount,
        "timestamp": datetime.now(timezone.utc).isoformat(),
        "description": f"تنظیم توسط ادمین: {data.reason}"
    })
    
    return {"message": "امتیاز تنظیم شد"}


@router.get("/admin/settings")
async def get_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات پاداش"""
    db = await get_db()
    settings = await db.settings.find_one({"setting_id": "rewards"}, {"_id": 0})
    return settings


@router.put("/admin/settings")
async def update_settings(data: RewardSettings, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی تنظیمات پاداش"""
    db = await get_db()
    
    await db.settings.update_one(
        {"setting_id": "rewards"},
        {"$set": data.model_dump()},
        upsert=True
    )
    return {"message": "تنظیمات بروزرسانی شد"}


@router.get("/admin/stats")
async def get_stats(admin_user: dict = Depends(get_admin_user)):
    """آمار کلی سیستم"""
    db = await get_db()
    
    total_users = await db.users.count_documents({})
    active_users = await db.users.count_documents({"status": "active"})
    pending_users = await db.users.count_documents({"status": "pending"})
    
    # جمع کل امتیازات
    pipeline = [{"$group": {"_id": None, "total": {"$sum": "$total_points"}}}]
    result = await db.users.aggregate(pipeline).to_list(1)
    total_points = result[0]["total"] if result else 0
    
    return {
        "total_users": total_users,
        "active_users": active_users,
        "pending_users": pending_users,
        "total_points": total_points
    }


@router.post("/admin/test/create-users")
async def create_test_users(data: TestUserCreate, admin_user: dict = Depends(get_admin_user)):
    """ایجاد کاربران تستی"""
    db = await get_db()
    
    # Set database connections for services
    set_activation_db(db)
    set_placement_db(db)
    
    created = []
    
    for i in range(data.count):
        user_id = str(uuid.uuid4())
        email = f"test_{user_id[:8]}@test.com"
        referral_code = str(uuid.uuid4())[:8].upper()
        
        # پیدا کردن referrer
        referrer_id = "seed"
        if data.referral_code:
            referrer = await db.users.find_one({"referral_code": data.referral_code.upper()})
            if referrer:
                referrer_id = referrer["user_id"]
        
        new_user = {
            "user_id": user_id,
            "email": email,
            "password": hash_password("test123"),
            "role": "user",
            "status": "pending",
            "referrer_id": referrer_id,
            "placement_parent_id": None,
            "placement_path": None,
            "placement_position": None,
            "depth": None,
            "ancestor_depths": [],
            "direct_children": [],
            "direct_children_count": 0,
            "total_descendants": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": None,
            "referral_code": referral_code,
            "is_test_user": True,
            "telegram_chat_id": None,
            "wallet_address": None,
            "wallet_network": None
        }
        
        await db.users.insert_one(new_user)
        created.append({"user_id": user_id, "email": email, "referral_code": referral_code})
        
        # فعال‌سازی خودکار با PlacementService
        if data.auto_activate:
            try:
                # Full activation with proper placement logic
                activation_result = await UserActivationService.activate_user(user_id)
                logging.info(f"✅ Test user {user_id} activated successfully")
            except Exception as e:
                logging.error(f"❌ Test user {user_id} activation failed: {e}")
                # User remains pending
    
    return {"created": len(created), "users": created}


@router.delete("/admin/test/clear-test-users")
async def clear_test_users(admin_user: dict = Depends(get_admin_user)):
    """پاک کردن کاربران تستی"""
    db = await get_db()
    result = await db.users.delete_many({"is_test_user": True})
    return {"deleted": result.deleted_count}


@router.delete("/admin/users/clear-all")
async def clear_all_users(admin_user: dict = Depends(get_admin_user)):
    """پاک کردن همه کاربران (به جز seed)"""
    db = await get_db()
    
    # پاک کردن کاربران
    result = await db.users.delete_many({"user_id": {"$ne": "seed"}})
    
    # 🔧 FIX: Reset کامل seed - حالا همیشه صحیح کار می‌کنه
    await db.users.update_one(
        {"user_id": "seed"},
        {"$set": {
            "direct_children": [],
            "direct_children_count": 0,      # ✅ همیشه reset میشه
            "total_descendants": 0,           # ✅ همیشه reset میشه
            "total_points": 0.0
        }}
    )
    
    # پاک کردن تراکنش‌ها و پرداخت‌ها
    await db.transactions.delete_many({})
    await db.payments.delete_many({})
    
    logging.warning(f"⚠️ All users cleared by admin {admin_user['user_id']}")
    
    return {"deleted": result.deleted_count, "message": "همه کاربران پاک شدند و seed reset شد"}


@router.get("/admin/dashboard/stats")
async def get_dashboard_stats(admin_user: dict = Depends(get_admin_user)):
    """آمار دشبورد"""
    db = await get_db()
    
    total_users = await db.users.count_documents({})
    active_users = await db.users.count_documents({"status": "active"})
    pending_users = await db.users.count_documents({"status": "pending"})
    total_transactions = await db.transactions.count_documents({})
    
    # جمع کل امتیازات
    pipeline = [{"$group": {"_id": None, "total": {"$sum": "$total_points"}}}]
    result = await db.users.aggregate(pipeline).to_list(1)
    total_points = result[0]["total"] if result else 0
    
    # تراکنش‌های اخیر
    recent_transactions = await db.transactions.find(
        {}, {"_id": 0}
    ).sort("timestamp", -1).limit(10).to_list(10)
    
    # کاربران جدید
    recent_users = await db.users.find(
        {}, {"_id": 0, "password": 0}
    ).sort("created_at", -1).limit(10).to_list(10)
    
    return {
        "stats": {
            "total_users": total_users,
            "active_users": active_users,
            "pending_users": pending_users,
            "total_transactions": total_transactions,
            "total_points": total_points
        },
        "recent_transactions": recent_transactions,
        "recent_users": recent_users
    }


@router.put("/admin/config")
async def update_config(data: ConfigUpdate, admin_user: dict = Depends(get_admin_user)):
    """بروزرسانی تنظیمات"""
    db = await get_db()
    
    await db.config.update_one(
        {"key": data.key},
        {"$set": {"value": data.value}},
        upsert=True
    )
    return {"message": "تنظیمات بروزرسانی شد"}


@router.get("/admin/config/{key}")
async def get_config(key: str, admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات"""
    db = await get_db()
    config = await db.config.find_one({"key": key}, {"_id": 0})
    return config
