import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { toast } from 'sonner';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from './ui/dialog';

const API = process.env.REACT_APP_BACKEND_URL || '';

export default function TransferTest() {
  const [toAddress, setToAddress] = useState('');
  const [amount, setAmount] = useState('1');
  const [network, setNetwork] = useState('BSC');
  const [testing, setTesting] = useState(false);
  const [result, setResult] = useState(null);
  const [withdrawals, setWithdrawals] = useState([]);
  const [selectedWithdrawal, setSelectedWithdrawal] = useState(null);
  const [loading, setLoading] = useState(false);
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [rejectReason, setRejectReason] = useState('');
  const [rejecting, setRejecting] = useState(false);

  useEffect(() => {
    fetchWithdrawals();
  }, []);

  const getAuthHeaders = () => {
    const token = localStorage.getItem('token');
    return { Authorization: `Bearer ${token}` };
  };

  const fetchWithdrawals = async () => {
    setLoading(true);
    try {
      const headers = getAuthHeaders();
      const response = await axios.get(`${API}/api/admin/withdrawals`, { headers });
      // فقط درخواست‌های pending_admin را نشان بده
      const pending = response.data.filter(w => w.status === 'pending_admin');
      setWithdrawals(pending);
    } catch (error) {
      console.error('خطا در دریافت درخواست‌ها:', error);
      toast.error('خطا در دریافت درخواست‌های برداشت');
    } finally {
      setLoading(false);
    }
  };

  const handleSelectWithdrawal = (withdrawal) => {
    setSelectedWithdrawal(withdrawal);
    // پر کردن خودکار فرم
    setToAddress(withdrawal.crypto_address);
    setAmount(withdrawal.final_transfer_amount || withdrawal.amount);
    // تشخیص شبکه
    if (withdrawal.network === 'USDT-TRC20' || withdrawal.network === 'TRON') {
      setNetwork('TRON');
    } else if (withdrawal.network === 'USDT-BEP20' || withdrawal.network === 'BSC') {
      setNetwork('BSC');
    }
    toast.success('✅ اطلاعات درخواست در فرم پر شد');
  };

  const handleTest = async () => {
    if (!toAddress || !amount) {
      toast.error('لطفاً همه فیلدها را پر کنید');
      return;
    }

    // Validation
    if (network === 'BSC' && !toAddress.startsWith('0x')) {
      toast.error('آدرس BSC باید با 0x شروع شود');
      return;
    }

    if (network === 'TRON' && !toAddress.startsWith('T')) {
      toast.error('آدرس TRON باید با T شروع شود');
      return;
    }

    setTesting(true);
    setResult(null);

    try {
      const token = localStorage.getItem('token');
      
      toast.loading('در حال تست انتقال...');
      
      const response = await axios.post(
        `${API}/api/admin/test-transfer`,
        {
          to_address: toAddress,
          amount: parseFloat(amount),
          network: network
        },
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        }
      );

      toast.dismiss();
      
      if (response.data.success) {
        setResult({
          success: true,
          tx_hash: response.data.tx_hash,
          network: response.data.network
        });
        toast.success('✅ تست موفق! تراکنش ارسال شد');
        
        // اگر این انتقال برای یک withdrawal بود، آن را به completed تغییر بده
        if (selectedWithdrawal) {
          await updateWithdrawalStatus(selectedWithdrawal.withdrawal_id, response.data.tx_hash);
        }
      } else {
        setResult({
          success: false,
          error: response.data.error
        });
        toast.error('❌ تست ناموفق');
      }
    } catch (error) {
      toast.dismiss();
      const errorMsg = error.response?.data?.detail || error.message || 'خطای ناشناخته';
      setResult({
        success: false,
        error: errorMsg
      });
      toast.error(`❌ ${errorMsg}`);
      // اگر تراکنش fail شد، selectedWithdrawal را نگه دار تا کاربر دوباره امتحان کند
    } finally {
      setTesting(false);
    }
  };

  const handleReject = () => {
    if (!selectedWithdrawal) {
      toast.error('لطفاً ابتدا یک درخواست انتخاب کنید');
      return;
    }
    setShowRejectDialog(true);
  };

  const confirmReject = async () => {
    if (!rejectReason.trim()) {
      toast.error('لطفاً دلیل رد را وارد کنید');
      return;
    }

    setRejecting(true);
    try {
      const headers = getAuthHeaders();
      await axios.put(
        `${API}/api/admin/withdrawals/${selectedWithdrawal.withdrawal_id}/reject?reason=${encodeURIComponent(rejectReason)}`,
        null,
        { headers }
      );
      
      toast.success('✅ درخواست رد شد و موجودی به کاربر بازگشت');
      
      // Reset form
      setShowRejectDialog(false);
      setRejectReason('');
      setSelectedWithdrawal(null);
      setToAddress('');
      setAmount('1');
      
      // Refresh list
      await fetchWithdrawals();
    } catch (error) {
      console.error('خطا در رد درخواست:', error);
      toast.error(error.response?.data?.detail || 'خطا در رد درخواست');
    } finally {
      setRejecting(false);
    }
  };

  const updateWithdrawalStatus = async (withdrawalId, txHash) => {
    try {
      const headers = getAuthHeaders();
      await axios.put(
        `${API}/api/admin/withdrawals/${withdrawalId}/approve`,
        { transaction_hash: txHash },
        { headers }
      );
      toast.success('✅ وضعیت درخواست به "انجام شده" تغییر کرد');
      
      // پاک کردن انتخاب و رفرش لیست
      setSelectedWithdrawal(null);
      setToAddress('');
      setAmount('1');
      await fetchWithdrawals();
    } catch (error) {
      console.error('خطا در به‌روزرسانی وضعیت:', error);
      toast.error('⚠️ انتقال موفق بود اما وضعیت به‌روز نشد');
    }
  };

  return (
    <div className="p-6 space-y-6">
      {/* لیست درخواست‌های برداشت */}
      <Card>
        <CardHeader className="bg-gradient-to-r from-blue-500 to-cyan-500 text-white">
          <CardTitle className="text-xl">📋 درخواست‌های برداشت ({withdrawals.length})</CardTitle>
          <p className="text-sm opacity-90 mt-1">
            روی هر درخواست کلیک کنید تا اطلاعات در فرم پر شود
          </p>
        </CardHeader>
        <CardContent className="p-4">
          {loading ? (
            <div className="text-center py-8">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto"></div>
              <p className="text-gray-600 mt-2">در حال بارگذاری...</p>
            </div>
          ) : withdrawals.length === 0 ? (
            <div className="text-center py-8 text-gray-500">
              <p>هیچ درخواست برداشتی در انتظار نیست</p>
            </div>
          ) : (
            <div className="space-y-3">
              {withdrawals.map((w) => (
                <div
                  key={w.withdrawal_id}
                  onClick={() => handleSelectWithdrawal(w)}
                  className={`border-2 rounded-lg p-4 cursor-pointer transition-all hover:shadow-md ${
                    selectedWithdrawal?.withdrawal_id === w.withdrawal_id
                      ? 'border-blue-500 bg-blue-50'
                      : 'border-gray-200 hover:border-blue-300'
                  }`}
                >
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <p className="font-semibold text-gray-800">👤 {w.user_email}</p>
                      <p className="text-2xl font-bold text-blue-600 mt-1">
                        {w.final_transfer_amount || w.amount} <span className="text-sm">USDT</span>
                      </p>
                    </div>
                    <span className="px-3 py-1 bg-amber-100 text-amber-700 rounded-full text-xs font-semibold">
                      در انتظار
                    </span>
                  </div>
                  <div className="text-sm text-gray-600 mb-2">
                    🌐 شبکه: <span className="font-semibold">{w.network === 'USDT-BEP20' ? 'BSC' : 'TRON'}</span>
                  </div>
                  <div className="bg-gray-100 p-2 rounded">
                    <p className="text-xs text-gray-500 mb-1">آدرس کیف پول:</p>
                    <p className="font-mono text-xs text-gray-800 break-all">{w.crypto_address}</p>
                  </div>
                  {selectedWithdrawal?.withdrawal_id === w.withdrawal_id && (
                    <div className="mt-2 text-center">
                      <p className="text-xs text-blue-600 font-semibold">✓ انتخاب شده</p>
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* فرم تست انتقال */}
      <Card>
        <CardHeader className="bg-gradient-to-r from-purple-500 to-pink-500 text-white">
          <CardTitle className="text-2xl">🧪 تست سیستم انتقال</CardTitle>
          <p className="text-sm opacity-90 mt-2">
            {selectedWithdrawal 
              ? `💡 اطلاعات برای: ${selectedWithdrawal.user_email}` 
              : 'برای تست دستی، فیلدها را پر کنید'}
          </p>
        </CardHeader>
        <CardContent className="p-6 space-y-4">
          <div>
            <Label className="text-base font-semibold">شبکه</Label>
            <select
              value={network}
              onChange={(e) => setNetwork(e.target.value)}
              className="w-full border-2 border-gray-300 rounded-lg p-3 mt-2"
            >
              <option value="BSC">BSC (BEP20)</option>
              <option value="TRON">TRON (TRC20)</option>
            </select>
          </div>

          <div>
            <Label className="text-base font-semibold">آدرس مقصد</Label>
            <Input
              value={toAddress}
              onChange={(e) => setToAddress(e.target.value)}
              placeholder={network === 'BSC' ? '0x...' : 'T...'}
              className="mt-2 font-mono"
            />
            <p className="text-xs text-gray-500 mt-1">
              {network === 'BSC' 
                ? '💡 آدرس BSC باید با 0x شروع شود و 42 کاراکتر باشد' 
                : '💡 آدرس TRON باید با T شروع شود و 34 کاراکتر باشد'}
            </p>
          </div>

          <div>
            <Label className="text-base font-semibold">مبلغ (USDT)</Label>
            <Input
              type="number"
              step="0.01"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="1"
              className="mt-2"
            />
          </div>

          <div className="bg-yellow-50 border-2 border-yellow-300 p-4 rounded-lg">
            <p className="text-sm text-yellow-800">
              ⚠️ <strong>توجه:</strong> این تست واقعاً تراکنش ارسال می‌کند! 
              مطمئن شوید که:
            </p>
            <ul className="text-xs text-yellow-700 mt-2 mr-4 list-disc">
              <li>Private Key در تنظیمات ذخیره شده</li>
              <li>موجودی کافی دارید (USDT + gas fee)</li>
              <li>آدرس را دوباره چک کنید</li>
            </ul>
          </div>

          {selectedWithdrawal && (
            <div className="bg-blue-50 border-2 border-blue-300 p-3 rounded-lg mb-4">
              <p className="text-sm text-blue-800 font-semibold">
                ✓ درخواست انتخاب شده: {selectedWithdrawal.user_email}
              </p>
              <p className="text-xs text-blue-700 mt-1">
                با کلیک روی دکمه "پرداخت"، انتقال انجام می‌شود. اگر موفق نبود، می‌توانید دوباره تلاش کنید یا درخواست را رد کنید.
              </p>
            </div>
          )}

          <div className={`grid gap-3 ${selectedWithdrawal ? 'grid-cols-2' : 'grid-cols-1'}`}>
            <Button
              onClick={handleTest}
              disabled={testing || rejecting}
              className={`text-lg py-6 ${
                selectedWithdrawal
                  ? 'bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700'
                  : 'bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 w-full'
              }`}
            >
              {testing ? (
                <>
                  <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white ml-2"></div>
                  در حال انتقال...
                </>
              ) : selectedWithdrawal ? (
                <>
                  ✅ پرداخت به {selectedWithdrawal.user_email}
                </>
              ) : (
                '🚀 تست انتقال'
              )}
            </Button>

            {selectedWithdrawal && (
              <Button
                onClick={handleReject}
                disabled={testing || rejecting}
                className="bg-gradient-to-r from-red-600 to-rose-600 hover:from-red-700 hover:to-rose-700 text-lg py-6"
              >
                {rejecting ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white ml-2"></div>
                    در حال رد...
                  </>
                ) : (
                  <>
                    ❌ رد درخواست
                  </>
                )}
              </Button>
            )}
          </div>

          {result && (
            <div className={`p-4 rounded-lg border-2 ${
              result.success 
                ? 'bg-green-50 border-green-300' 
                : 'bg-red-50 border-red-300'
            }`}>
              <h3 className="font-bold text-lg mb-2">
                {result.success ? '✅ موفق!' : '❌ ناموفق'}
              </h3>
              {result.success ? (
                <div className="space-y-2 text-sm">
                  <p><strong>شبکه:</strong> {result.network}</p>
                  <p><strong>Transaction Hash:</strong></p>
                  <p className="font-mono text-xs break-all bg-white p-2 rounded">
                    {result.tx_hash}
                  </p>
                  <p className="text-green-700 font-semibold mt-3">
                    🎉 سیستم انتقال کاملاً سالم است!
                  </p>
                </div>
              ) : (
                <div className="space-y-2 text-sm text-red-700">
                  <p><strong>خطا:</strong> {result.error}</p>
                  <p className="mt-3">
                    💡 لطفاً بررسی کنید:
                  </p>
                  <ul className="list-disc mr-4 text-xs">
                    <li>Private Key صحیح است؟</li>
                    <li>موجودی کافی دارید؟</li>
                    <li>آدرس صحیح است؟</li>
                  </ul>
                </div>
              )}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Dialog برای رد کردن درخواست */}
      <Dialog open={showRejectDialog} onOpenChange={setShowRejectDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>رد درخواست برداشت</DialogTitle>
            <DialogDescription>
              {selectedWithdrawal && (
                <>
                  درخواست {selectedWithdrawal.user_email} برای برداشت {selectedWithdrawal.amount} USDT
                </>
              )}
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 pt-4">
            <div>
              <Label htmlFor="reject-reason" className="text-base font-semibold">
                دلیل رد <span className="text-red-500">*</span>
              </Label>
              <Input
                id="reject-reason"
                value={rejectReason}
                onChange={(e) => setRejectReason(e.target.value)}
                placeholder="مثلاً: موجودی کافی نیست، آدرس نامعتبر است و..."
                className="mt-2"
              />
              <p className="text-xs text-gray-500 mt-2">
                این پیام به کاربر نمایش داده می‌شود
              </p>
            </div>

            <div className="bg-yellow-50 border-2 border-yellow-300 p-3 rounded-lg">
              <p className="text-sm text-yellow-800">
                ⚠️ <strong>توجه:</strong> با رد این درخواست:
              </p>
              <ul className="text-xs text-yellow-700 mt-2 mr-4 list-disc">
                <li>موجودی به حساب کاربر بازگشت داده می‌شود</li>
                <li>دلیل رد برای کاربر نمایش داده می‌شود</li>
                <li>این عملیات قابل بازگشت نیست</li>
              </ul>
            </div>

            <div className="grid grid-cols-2 gap-3">
              <Button
                onClick={() => {
                  setShowRejectDialog(false);
                  setRejectReason('');
                }}
                variant="outline"
                disabled={rejecting}
              >
                انصراف
              </Button>
              <Button
                onClick={confirmReject}
                disabled={rejecting || !rejectReason.trim()}
                className="bg-red-600 hover:bg-red-700"
              >
                {rejecting ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white ml-2"></div>
                    در حال رد...
                  </>
                ) : (
                  'تایید رد'
                )}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
