"""
Telegram Settings Module
خواندن تنظیمات تلگرام از دیتابیس MongoDB
"""

import os
from motor.motor_asyncio import AsyncIOMotorClient
import logging

logger = logging.getLogger(__name__)


async def get_settings() -> dict:
    """
    خواندن تنظیمات تلگرام از دیتابیس
    اگر وجود نداشت، تنظیمات پیش‌فرض برمی‌گرداند
    """
    try:
        mongo_url = os.getenv('MONGO_URL', 'mongodb://localhost:27017')
        db_name = os.getenv('DB_NAME', 'mlm_database')
        
        client = AsyncIOMotorClient(mongo_url)
        db = client[db_name]
        
        # خواندن از collection settings
        settings = await db.settings.find_one(
            {"setting_id": "telegram"},
            {"_id": 0}
        )
        
        client.close()
        
        if settings:
            logger.info("✅ تنظیمات تلگرام از دیتابیس خوانده شد")
            return settings
        else:
            logger.warning("⚠️ تنظیمات تلگرام در دیتابیس نیست - از پیش‌فرض استفاده می‌شود")
            return {
                "setting_id": "telegram",
                "enabled": False,
                "bot_token": "",
                "bot_username": "",
                "webapp_url": "",
                "welcome_message": "👋 سلام {user_name} عزیز! به سیستم ما خوش آمدید",
                "welcome_with_ref_message": "🎉 سلام {user_name} عزیز! شما با لینک دعوت {referral_code} وارد شدید",
                "activation_message": "✅ حساب شما فعال شد! از طریق دکمه زیر وارد داشبورد شوید"
            }
    
    except Exception as e:
        logger.error(f"❌ خطا در خواندن تنظیمات: {e}")
        # برگشت تنظیمات پیش‌فرض در صورت خطا
        return {
            "setting_id": "telegram",
            "enabled": False,
            "bot_token": "",
            "bot_username": "",
            "webapp_url": "",
            "welcome_message": "👋 سلام! به سیستم ما خوش آمدید",
            "welcome_with_ref_message": "🎉 سلام! شما با لینک دعوت وارد شدید",
            "activation_message": "✅ حساب شما فعال شد!"
        }


async def save_settings(settings_dict: dict) -> bool:
    """
    ذخیره تنظیمات تلگرام در دیتابیس
    """
    try:
        mongo_url = os.getenv('MONGO_URL', 'mongodb://localhost:27017')
        db_name = os.getenv('DB_NAME', 'mlm_database')
        
        client = AsyncIOMotorClient(mongo_url)
        db = client[db_name]
        
        settings_dict["setting_id"] = "telegram"
        
        result = await db.settings.update_one(
            {"setting_id": "telegram"},
            {"$set": settings_dict},
            upsert=True
        )
        
        client.close()
        
        logger.info("✅ تنظیمات تلگرام در دیتابیس ذخیره شد")
        return True
    
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره تنظیمات: {e}")
        return False
