"""
Bot تلگرام - نسخه کامل و اصلاح شده
✅ ذخیره در telegram_chats
✅ ارسال پیام فعالسازی
✅ خواندن تنظیمات از دیتابیس
"""

import asyncio
import logging
import sys
import os
sys.path.insert(0, '/app/backend')

from telegram import Update, WebAppInfo, InlineKeyboardButton, InlineKeyboardMarkup, Bot
from telegram.ext import Application, CommandHandler, ContextTypes
from telegram_settings import get_settings
from motor.motor_asyncio import AsyncIOMotorClient
from datetime import datetime, timezone
import uuid

# Logging
logging.basicConfig(
    format='%(asctime)s - %(levelname)s - %(message)s',
    level=logging.INFO
)
logger = logging.getLogger(__name__)


async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    Handler برای /start
    هر بار تنظیمات تازه از دیتابیس می‌خواند
    """
    user = update.effective_user
    logger.info(f"📩 /start از کاربر {user.id} (@{user.username})")
    
    # ذخیره chat_id در دیتابیس برای broadcast
    try:
        client = AsyncIOMotorClient(os.getenv('MONGO_URL', 'mongodb://localhost:27017'))
        db = client[os.getenv('DB_NAME', 'mlm_database')]
        
        # ذخیره یا آپدیت chat_id در telegram_chats
        await db.telegram_chats.update_one(
            {"chat_id": user.id},
            {
                "$set": {
                    "chat_id": user.id,
                    "username": user.username,
                    "first_name": user.first_name,
                    "last_name": user.last_name,
                    "last_interaction": datetime.now(timezone.utc).isoformat()
                }
            },
            upsert=True
        )
        logger.info(f"💾 chat_id {user.id} در telegram_chats ذخیره شد")
        client.close()
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره chat_id: {e}")
    
    # خواندن تنظیمات تازه
    settings = await get_settings()
    
    # چک enabled
    if not settings.get("enabled"):
        logger.warning(f"⚠️ Bot غیرفعال است")
        await update.message.reply_text("⚠️ ربات در حال حاضر غیرفعال است.")
        return
    
    # دریافت URL و پیام‌ها
    webapp_url = settings.get("webapp_url", "")
    welcome_msg = settings.get("welcome_message", "")
    welcome_ref_msg = settings.get("welcome_with_ref_message", "")
    
    if not webapp_url:
        logger.error("❌ webapp_url خالی است")
        await update.message.reply_text("⚠️ تنظیمات ربات ناقص است. لطفاً با پشتیبانی تماس بگیرید.")
        return
    
    # چک کردن referral code
    referral_code = None
    if context.args and len(context.args) > 0:
        referral_code = context.args[0]
    
    # ایجاد یک session_id منحصر به فرد برای ارتباط با ثبت‌نام
    session_id = str(uuid.uuid4())
    
    # ذخیره session_id با chat_id
    try:
        client = AsyncIOMotorClient(os.getenv('MONGO_URL', 'mongodb://localhost:27017'))
        db = client[os.getenv('DB_NAME', 'mlm_database')]
        
        await db.telegram_sessions.update_one(
            {"session_id": session_id},
            {
                "$set": {
                    "session_id": session_id,
                    "chat_id": user.id,
                    "username": user.username,
                    "created_at": datetime.now(timezone.utc).isoformat(),
                    "used": False
                }
            },
            upsert=True
        )
        logger.info(f"💾 session_id {session_id} ذخیره شد")
        client.close()
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره session: {e}")
    
    # ساخت URL و پیام
    if referral_code:
        url = f"{webapp_url}/register?ref={referral_code}&source=telegram&tg_session={session_id}"
        message = welcome_ref_msg.format(
            user_name=user.first_name or "کاربر",
            referral_code=referral_code
        ) if welcome_ref_msg else f"🎉 سلام {user.first_name}!"
        logger.info(f"🔗 با رفرال: {referral_code}")
    else:
        url = f"{webapp_url}?source=telegram&tg_session={session_id}"
        message = welcome_msg.format(
            user_name=user.first_name or "کاربر"
        ) if welcome_msg else f"👋 سلام {user.first_name}!"
        logger.info("🔗 بدون رفرال")
    
    # ساخت دکمه
    keyboard = [
        [InlineKeyboardButton("🚀 ورود به سیستم", web_app=WebAppInfo(url=url))]
    ]
    reply_markup = InlineKeyboardMarkup(keyboard)
    
    # ارسال پیام
    try:
        await update.message.reply_text(
            message,
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        logger.info(f"✅ پیام ارسال شد ({len(message)} حرف)")
    except Exception as e:
        logger.error(f"❌ خطا در ارسال پیام: {e}")


async def help_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """Help command"""
    await update.message.reply_text("📚 برای شروع، دستور /start را بفرستید")


async def send_activation_message(telegram_chat_id: int, user_email: str, referral_code: str):
    """
    ارسال پیام فعالسازی به کاربر
    شامل لینک‌های دعوت (سایت و تلگرام)
    این تابع از auth.py فراخوانی می‌شود
    """
    try:
        settings = await get_settings()
        token = settings.get("bot_token", "")
        webapp_url = settings.get("webapp_url", "")
        bot_username = settings.get("bot_username", "")
        activation_msg_template = settings.get("activation_message", "")
        
        if not token or not settings.get("enabled"):
            logger.warning("⚠️ ربات غیرفعال است یا توکن خالی است")
            return False
        
        if not webapp_url or not bot_username:
            logger.error("❌ webapp_url یا bot_username خالی است")
            return False
        
        # ساخت لینک‌های دعوت
        website_referral_link = f"{webapp_url}/register?ref={referral_code}"
        telegram_referral_link = f"https://t.me/{bot_username}?start={referral_code}"
        
        # پیام فعالسازی (با قالب کاستوم یا پیش‌فرض)
        if activation_msg_template:
            message = activation_msg_template.format(
                user_email=user_email,
                website_link=website_referral_link,
                telegram_link=telegram_referral_link
            )
        else:
            message = f"""🎉 تبریک {user_email} عزیز!

✅ حساب شما با موفقیت فعال شد!

🔗 لینک‌های دعوت شما:

🌐 **لینک دعوت (وب):**
{website_referral_link}

✈️ **لینک دعوت (تلگرام):**
{telegram_referral_link}

💰 با دعوت دوستان خود، پاداش دریافت کنید!

🚀 برای مشاهده داشبورد خود، روی دکمه زیر کلیک کنید."""

        # ساخت دکمه
        keyboard = [
            [InlineKeyboardButton("📊 داشبورد من", web_app=WebAppInfo(url=webapp_url))]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        # ارسال پیام
        bot = Bot(token)
        await bot.send_message(
            chat_id=telegram_chat_id,
            text=message,
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        logger.info(f"✅ پیام فعالسازی به {telegram_chat_id} ({user_email}) ارسال شد")
        return True
        
    except Exception as e:
        logger.error(f"❌ خطا در ارسال پیام فعالسازی: {e}")
        return False


async def send_notification(telegram_chat_id: int, message: str):
    """
    ارسال نوتیفیکیشن به یک کاربر
    برای استفاده در سیستم (پاداش‌ها، اعلانات و...)
    """
    try:
        settings = await get_settings()
        token = settings.get("bot_token", "")
        
        if not token or not settings.get("enabled"):
            logger.warning("⚠️ ربات غیرفعال است")
            return False
        
        bot = Bot(token)
        await bot.send_message(
            chat_id=telegram_chat_id,
            text=message,
            parse_mode='Markdown'
        )
        
        logger.info(f"✅ نوتیفیکیشن به {telegram_chat_id} ارسال شد")
        return True
        
    except Exception as e:
        logger.error(f"❌ خطا در ارسال نوتیفیکیشن: {e}")
        return False


async def send_broadcast_message(message: str):
    """
    ارسال پیام عمومی به همه کاربران
    از collection telegram_chats استفاده می‌کند
    """
    try:
        settings = await get_settings()
        token = settings.get("bot_token", "")
        
        if not token or not settings.get("enabled"):
            logger.warning("⚠️ ربات غیرفعال است")
            return {"success": 0, "failed": 0}
        
        # دریافت لیست chat_id ها
        client = AsyncIOMotorClient(os.getenv('MONGO_URL', 'mongodb://localhost:27017'))
        db = client[os.getenv('DB_NAME', 'mlm_database')]
        
        chats = await db.telegram_chats.find({}, {"chat_id": 1}).to_list(100000)
        client.close()
        
        if not chats:
            logger.warning("⚠️ هیچ chat_id برای ارسال نیست")
            return {"success": 0, "failed": 0}
        
        bot = Bot(token)
        success_count = 0
        failed_count = 0
        
        for chat in chats:
            try:
                chat_id = chat.get("chat_id")
                if chat_id:
                    await bot.send_message(
                        chat_id=chat_id,
                        text=message,
                        parse_mode='Markdown'
                    )
                    success_count += 1
                    logger.info(f"✅ پیام به {chat_id} ارسال شد")
                    await asyncio.sleep(0.05)  # جلوگیری از rate limit
            except Exception as e:
                failed_count += 1
                logger.error(f"❌ خطا در ارسال به {chat_id}: {e}")
        
        logger.info(f"📊 Broadcast تمام شد - موفق: {success_count}, ناموفق: {failed_count}")
        return {"success": success_count, "failed": failed_count}
        
    except Exception as e:
        logger.error(f"❌ خطا در broadcast: {e}")
        return {"success": 0, "failed": 0}


def main():
    """راه‌اندازی Bot"""
    logger.info("🤖 شروع Bot...")
    
    # خواندن تنظیمات
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    settings = loop.run_until_complete(get_settings())
    
    if not settings.get("enabled"):
        logger.error("❌ Bot غیرفعال است!")
        logger.info("💡 برای فعالسازی، وارد پنل ادمین شوید و تنظیمات تلگرام را فعال کنید")
        loop.close()
        return
    
    token = settings.get("bot_token", "")
    if not token:
        logger.error("❌ Token خالی است!")
        logger.info("💡 برای تنظیم توکن، وارد پنل ادمین شوید")
        loop.close()
        return
    
    # نمایش تنظیمات
    logger.info("="*50)
    logger.info("✅ تنظیمات:")
    logger.info(f"   Enabled: {settings.get('enabled')}")
    logger.info(f"   Username: @{settings.get('bot_username')}")
    logger.info(f"   Token: {token[:20]}...")
    logger.info(f"   WebApp URL: {settings.get('webapp_url')}")
    logger.info(f"   پیام عادی: {len(settings.get('welcome_message', ''))} حرف")
    logger.info(f"   پیام رفرال: {len(settings.get('welcome_with_ref_message', ''))} حرف")
    logger.info(f"   پیام فعالسازی: {len(settings.get('activation_message', ''))} حرف")
    logger.info("="*50)
    
    # ساخت Application
    application = Application.builder().token(token).build()
    
    # Handlers
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("help", help_command))
    
    # شروع
    logger.info("✅ Bot در حال اجرا...")
    logger.info("📱 برای تست، پیام /start را در تلگرام ارسال کنید")
    
    try:
        application.run_polling(allowed_updates=Update.ALL_TYPES)
    except KeyboardInterrupt:
        logger.info("🛑 Bot متوقف شد")
    finally:
        loop.close()


if __name__ == '__main__':
    main()
