"""User Registration Service - ثبت‌نام کاربر"""

from typing import Dict
import uuid
import logging
import bcrypt
from datetime import datetime, timezone

logger = logging.getLogger(__name__)

_db = None

def set_database(db):
    """تنظیم database connection"""
    global _db
    _db = db


class UserRegistrationService:
    """سرویس ثبت‌نام کاربر"""
    
    @staticmethod
    def hash_password(password: str) -> str:
        """هش کردن رمز عبور"""
        return bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
    
    @staticmethod
    def generate_referral_code() -> str:
        """تولید کد معرفی"""
        return str(uuid.uuid4())[:8].upper()
    
    @staticmethod
    async def register_user(
        email: str,
        password: str,
        phone_number: str = None,
        referral_code: str = None
    ) -> Dict:
        """
        ثبت‌نام کاربر جدید - وضعیت pending
        
        Args:
            email: ایمیل کاربر
            password: رمز عبور
            phone_number: شماره تماس (اختیاری)
            referral_code: کد معرفی (اختیاری)
            
        Returns:
            اطلاعات کاربر ثبت شده
            
        Queries: 3
        """
        
        email = email.strip().lower()
        
        # Query 1: Check duplicate email
        existing = await _db.users.find_one({"email": email})
        if existing:
            logger.warning(f"❌ Email already exists: {email}")
            raise ValueError("این ایمیل قبلاً ثبت شده است")
        
        # Query 2: Find referrer
        referrer_id = "seed"  # default
        if referral_code:
            referral_code_clean = referral_code.strip().upper()
            referrer = await _db.users.find_one(
                {"referral_code": referral_code_clean},
                {"_id": 0, "user_id": 1}
            )
            if referrer:
                referrer_id = referrer["user_id"]
                logger.info(f"✅ Referrer found: {referrer_id}")
        
        # Create user
        user_id = str(uuid.uuid4())
        user_referral_code = UserRegistrationService.generate_referral_code()
        password_hash = UserRegistrationService.hash_password(password)
        
        new_user = {
            "user_id": user_id,
            "email": email,
            "password": password_hash,
            "phone_number": phone_number,
            "referrer_id": referrer_id,
            "status": "pending",
            "role": "user",
            
            # 🆕 Materialized Path fields - null تا فعال‌سازی
            "placement_path": None,
            "placement_parent_id": None,
            "placement_position": None,
            "depth": None,
            "ancestor_depths": [],
            
            # Stats
            "direct_children": [],
            "direct_children_count": 0,
            "total_descendants": 0,
            "max_descendant_depth": 0,  # عمیق‌ترین descendant
            "total_points": 0.0,
            
            # Flags
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            
            # Timestamps
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": None,
            "referral_code": user_referral_code,
            
            # Optional fields
            "telegram_chat_id": None,
            "wallet_address": None,
            "wallet_network": None
        }
        
        # Query 3: Insert user
        await _db.users.insert_one(new_user)
        
        logger.info(f"✅ User registered: {email} with referrer: {referrer_id}")
        
        return {
            "user_id": user_id,
            "email": email,
            "referral_code": user_referral_code,
            "status": "pending",
            "message": "ثبت‌نام موفق - منتظر فعال‌سازی باشید"
        }
