"""Telegram Routes - اندپوینت‌های تلگرام (نسخه اصلاح شده)"""

import logging
from fastapi import APIRouter, HTTPException, Depends, Body
from typing import Optional

from models.telegram import TelegramSettings, TelegramBroadcast
from services.auth_service import get_current_user, get_admin_user
from database import get_db

router = APIRouter()
logger = logging.getLogger(__name__)


@router.get("/admin/telegram/settings")
async def get_telegram_settings(admin_user: dict = Depends(get_admin_user)):
    """دریافت تنظیمات تلگرام"""
    try:
        db = await get_db()
        settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
        
        if not settings:
            # تنظیمات پیش‌فرض
            default_settings = {
                "setting_id": "telegram",
                "enabled": False,
                "bot_token": "",
                "bot_username": "",
                "webapp_url": "",
                "welcome_message": "👋 سلام {user_name} عزیز! به سیستم ما خوش آمدید",
                "welcome_with_ref_message": "🎉 سلام {user_name} عزیز! شما با لینک دعوت {referral_code} وارد شدید",
                "activation_message": "✅ حساب شما فعال شد!"
            }
            return default_settings
        
        return settings
    except Exception as e:
        logger.error(f"❌ خطا در دریافت تنظیمات: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.put("/admin/telegram/settings")
async def update_telegram_settings(
    settings: TelegramSettings, 
    admin_user: dict = Depends(get_admin_user)
):
    """بروزرسانی تنظیمات تلگرام"""
    try:
        db = await get_db()
        
        settings_dict = settings.model_dump()
        settings_dict["setting_id"] = "telegram"
        
        result = await db.settings.update_one(
            {"setting_id": "telegram"},
            {"$set": settings_dict},
            upsert=True
        )
        
        logger.info(f"✅ تنظیمات تلگرام ذخیره شد - Modified: {result.modified_count}, Upserted: {result.upserted_id}")
        
        return {
            "message": "تنظیمات تلگرام بروزرسانی شد",
            "success": True
        }
    except Exception as e:
        logger.error(f"❌ خطا در بروزرسانی تنظیمات: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/admin/telegram/test")
async def test_telegram_bot(admin_user: dict = Depends(get_admin_user)):
    """تست ربات تلگرام"""
    try:
        db = await get_db()
        
        settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
        
        if not settings or not settings.get("bot_token"):
            raise HTTPException(status_code=400, detail="توکن ربات تنظیم نشده")
        
        # تست اتصال
        import httpx
        async with httpx.AsyncClient(timeout=10) as client:
            response = await client.get(
                f"https://api.telegram.org/bot{settings['bot_token']}/getMe"
            )
            data = response.json()
            
            if data.get("ok"):
                bot_info = data.get("result", {})
                logger.info(f"✅ تست بات موفق - Username: @{bot_info.get('username')}")
                return {
                    "success": True,
                    "bot": bot_info
                }
            else:
                error_msg = data.get("description", "خطا در اتصال به بات")
                logger.error(f"❌ تست بات ناموفق: {error_msg}")
                return {
                    "success": False,
                    "error": error_msg
                }
    except httpx.TimeoutException:
        logger.error("❌ Timeout در تست بات")
        return {"success": False, "error": "زمان اتصال به تلگرام تمام شد"}
    except Exception as e:
        logger.error(f"❌ خطا در تست بات: {e}")
        return {"success": False, "error": str(e)}


@router.post("/admin/telegram/broadcast")
async def broadcast_telegram_message(
    broadcast_data: TelegramBroadcast,
    admin_user: dict = Depends(get_admin_user)
):
    """
    ارسال پیام عمومی تلگرام
    از collection telegram_chats استفاده می‌کند (نه users)
    """
    try:
        db = await get_db()
        
        # دریافت تنظیمات
        settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
        
        if not settings or not settings.get("bot_token"):
            raise HTTPException(status_code=400, detail="توکن ربات تنظیم نشده")
        
        if not settings.get("enabled"):
            raise HTTPException(status_code=400, detail="ربات غیرفعال است")
        
        # دریافت لیست chat_id ها از telegram_chats (درست شده)
        chats = await db.telegram_chats.find(
            {},
            {"_id": 0, "chat_id": 1}
        ).to_list(100000)
        
        if not chats:
            return {
                "success": False,
                "message": "هیچ کاربری در لیست نیست",
                "sent": 0,
                "failed": 0,
                "total": 0
            }
        
        sent = 0
        failed = 0
        failed_ids = []
        
        import httpx
        import asyncio
        
        async with httpx.AsyncClient(timeout=10) as client:
            for chat in chats:
                try:
                    chat_id = chat.get("chat_id")
                    if chat_id:
                        response = await client.post(
                            f"https://api.telegram.org/bot{settings['bot_token']}/sendMessage",
                            json={
                                "chat_id": chat_id,
                                "text": broadcast_data.message,
                                "parse_mode": "Markdown"
                            }
                        )
                        
                        if response.status_code == 200:
                            sent += 1
                            logger.info(f"✅ پیام به {chat_id} ارسال شد")
                        else:
                            failed += 1
                            failed_ids.append(chat_id)
                            logger.error(f"❌ خطا در ارسال به {chat_id}: {response.text}")
                        
                        # جلوگیری از rate limit تلگرام
                        await asyncio.sleep(0.05)  # 50ms delay
                        
                except Exception as e:
                    failed += 1
                    if chat.get("chat_id"):
                        failed_ids.append(chat["chat_id"])
                    logger.error(f"❌ خطا در ارسال: {e}")
        
        logger.info(f"📊 Broadcast تمام شد - موفق: {sent}, ناموفق: {failed}")
        
        return {
            "success": True,
            "sent": sent,
            "failed": failed,
            "total": len(chats),
            "failed_ids": failed_ids[:10]  # فقط 10 اولی
        }
        
    except Exception as e:
        logger.error(f"❌ خطا در broadcast: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/user/telegram/bot-info")
async def get_telegram_bot_info():
    """دریافت اطلاعات عمومی ربات"""
    try:
        db = await get_db()
        
        settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
        
        if not settings or not settings.get("enabled"):
            return {"enabled": False}
        
        return {
            "enabled": True,
            "bot_username": settings.get("bot_username"),
            "webapp_url": settings.get("webapp_url")
        }
    except Exception as e:
        logger.error(f"❌ خطا در دریافت اطلاعات بات: {e}")
        return {"enabled": False}


@router.post("/user/telegram/save-chat-id")
async def save_telegram_chat_id(
    telegram_chat_id: int = Body(..., embed=True),
    current_user: dict = Depends(get_current_user)
):
    """
    ذخیره chat_id تلگرام کاربر
    در collection users برای ارسال نوتیفیکیشن شخصی
    """
    try:
        db = await get_db()
        
        # آپدیت در users
        await db.users.update_one(
            {"user_id": current_user["user_id"]},
            {"$set": {"telegram_chat_id": telegram_chat_id}}
        )
        
        logger.info(f"✅ chat_id {telegram_chat_id} برای کاربر {current_user['user_id']} ذخیره شد")
        
        return {
            "message": "chat_id ذخیره شد",
            "success": True
        }
    except Exception as e:
        logger.error(f"❌ خطا در ذخیره chat_id: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/public/login-tabs")
async def get_login_tabs():
    """دریافت تنظیمات تب‌های ورود"""
    try:
        db = await get_db()
        
        config = await db.config.find_one({"key": "login_tabs"}, {"_id": 0})
        
        return config.get("value", {"tab1": True, "tab2": True}) if config else {"tab1": True, "tab2": True}
    except Exception as e:
        logger.error(f"❌ خطا در دریافت تنظیمات تب: {e}")
        return {"tab1": True, "tab2": True}


@router.put("/admin/login-tabs/{tab_number}")
async def update_login_tab(
    tab_number: int, 
    data: dict, 
    admin_user: dict = Depends(get_admin_user)
):
    """بروزرسانی تنظیمات تب ورود"""
    try:
        db = await get_db()
        
        current = await db.config.find_one({"key": "login_tabs"}, {"_id": 0})
        tabs = current.get("value", {}) if current else {}
        tabs[f"tab{tab_number}"] = data.get("enabled", True)
        
        await db.config.update_one(
            {"key": "login_tabs"},
            {"$set": {"value": tabs}},
            upsert=True
        )
        
        logger.info(f"✅ تنظیمات تب {tab_number} بروزرسانی شد")
        
        return {"message": "تنظیمات بروزرسانی شد", "success": True}
    except Exception as e:
        logger.error(f"❌ خطا در بروزرسانی تب: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/admin/telegram/send-to-user")
async def send_message_to_user(
    user_id: str = Body(...),
    message: str = Body(...),
    admin_user: dict = Depends(get_admin_user)
):
    """ارسال پیام به یک کاربر خاص"""
    try:
        db = await get_db()
        
        # دریافت تنظیمات
        settings = await db.settings.find_one({"setting_id": "telegram"}, {"_id": 0})
        
        if not settings or not settings.get("bot_token"):
            raise HTTPException(status_code=400, detail="توکن ربات تنظیم نشده")
        
        # پیدا کردن chat_id کاربر
        user = await db.users.find_one({"user_id": user_id}, {"telegram_chat_id": 1})
        
        if not user or not user.get("telegram_chat_id"):
            raise HTTPException(status_code=404, detail="chat_id کاربر پیدا نشد")
        
        # ارسال پیام
        import httpx
        async with httpx.AsyncClient(timeout=10) as client:
            response = await client.post(
                f"https://api.telegram.org/bot{settings['bot_token']}/sendMessage",
                json={
                    "chat_id": user["telegram_chat_id"],
                    "text": message,
                    "parse_mode": "Markdown"
                }
            )
            
            if response.status_code == 200:
                logger.info(f"✅ پیام به کاربر {user_id} ارسال شد")
                return {"success": True, "message": "پیام ارسال شد"}
            else:
                logger.error(f"❌ خطا در ارسال: {response.text}")
                return {"success": False, "error": response.text}
                
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ خطا در ارسال پیام: {e}")
        raise HTTPException(status_code=500, detail=str(e))
