"""Auth Routes - ورود و ثبت‌نام"""

import logging
import time
import uuid
from datetime import datetime, timezone
from collections import defaultdict

import bcrypt
from fastapi import APIRouter, HTTPException

from models.user import UserRegister, UserLogin
from services.auth_service import hash_password, verify_password, create_access_token
from utils.security import validate_email
from database import get_db

router = APIRouter()

# Rate limiting
login_attempts = defaultdict(list)
MAX_LOGIN_ATTEMPTS = 5
LOGIN_ATTEMPT_WINDOW = 300  # 5 minutes


@router.post("/auth/register")
async def register(data: UserRegister):
    """
    ثبت‌نام کاربر جدید
    
    - اعتبارسنجی ایمیل و رمز عبور
    - بررسی تکراری نبودن ایمیل
    - یافتن معرف
    - ایجاد کاربر با وضعیت pending
    """
    db = await get_db()
    
    try:
        email = data.email.strip().lower()
        password = data.password
        
        logging.info(f"📝 Registration attempt for: {email}")
        
        # اعتبارسنجی
        if not validate_email(email):
            raise HTTPException(status_code=400, detail="فرمت ایمیل نامعتبر است")
        
        if len(password) < 6:
            raise HTTPException(status_code=400, detail="رمز عبور باید حداقل 6 کاراکتر باشد")
        
        # بررسی تکراری
        existing = await db.users.find_one({"email": email})
        if existing:
            logging.warning(f"❌ Email already exists: {email}")
            raise HTTPException(status_code=400, detail="این ایمیل قبلاً ثبت شده است")
        
        # پیدا کردن referrer
        referrer_id = "seed"
        if data.referral_code:
            referral_code_clean = data.referral_code.strip().upper()
            referrer = await db.users.find_one({"referral_code": referral_code_clean}, {"_id": 0})
            if referrer:
                referrer_id = referrer["user_id"]
                logging.info(f"✅ Referrer found: {referrer_id}")
        
        # ایجاد کاربر
        user_id = str(uuid.uuid4())
        referral_code = str(uuid.uuid4())[:8].upper()
        password_hash = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        new_user = {
            "user_id": user_id,
            "email": email,
            "phone_number": data.phone_number if data.phone_number else None,
            "password": password_hash,
            "role": "user",
            "status": "pending",
            "referrer_id": referrer_id,
            "placement_parent_id": None,
            "direct_children": [],
            "depth": 0,
            "total_points": 0.0,
            "blocked_direct_reward": False,
            "blocked_level_reward": False,
            "is_tree_complete": False,
            "created_at": datetime.now(timezone.utc).isoformat(),
            "activated_at": None,
            "referral_code": referral_code,
            "telegram_chat_id": None
        }
        
        await db.users.insert_one(new_user)
        
        logging.info(f"✅ User registered: {email} with referrer: {referrer_id}")
        
        return {
            "message": "ثبت‌نام موفقیت‌آمیز بود. منتظر فعال‌سازی باشید.",
            "user_id": user_id,
            "referral_code": referral_code
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"❌ Registration error: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطای سرور: {str(e)}")


@router.post("/auth/login")
async def login(data: UserLogin):
    """
    ورود کاربر
    
    - محافظت در برابر brute force
    - بررسی ایمیل و رمز عبور
    - صدور JWT token
    """
    db = await get_db()
    
    try:
        email = data.email.strip().lower()
        password = data.password
        
        logging.info(f"🔐 Login attempt for: {email}")
        
        # Rate limiting
        now = time.time()
        login_attempts[email] = [t for t in login_attempts[email] if now - t < LOGIN_ATTEMPT_WINDOW]
        
        if len(login_attempts[email]) >= MAX_LOGIN_ATTEMPTS:
            logging.warning(f"Too many login attempts for: {email}")
            raise HTTPException(
                status_code=429,
                detail="تعداد تلاش‌های ورود بیش از حد است. لطفاً 5 دقیقه صبر کنید."
            )
        
        # پیدا کردن کاربر
        user = await db.users.find_one({"email": email}, {"_id": 0})
        
        if not user:
            logging.warning(f"❌ User not found: {email}")
            login_attempts[email].append(now)
            raise HTTPException(status_code=401, detail="ایمیل یا رمز عبور اشتباه است")
        
        logging.info(f"✅ User found: {email}")
        
        # بررسی رمز عبور
        stored_hash = user.get("password", "")
        
        try:
            password_valid = bcrypt.checkpw(
                password.encode('utf-8'), 
                stored_hash.encode('utf-8')
            )
        except Exception as e:
            logging.error(f"❌ Password check error: {e}")
            password_valid = False
        
        if not password_valid:
            logging.warning(f"❌ Wrong password for: {email}")
            login_attempts[email].append(now)
            raise HTTPException(status_code=401, detail="ایمیل یا رمز عبور اشتباه است")
        
        logging.info(f"✅ Password correct for: {email}")
        
        # پاک کردن تلاش‌های ناموفق
        if email in login_attempts:
            del login_attempts[email]
        
        # ساخت توکن
        token = create_access_token({"user_id": user["user_id"], "role": user["role"]})
        
        logging.info(f"✅ Login successful for: {email}")
        
        return {
            "token": token,
            "user": {
                "user_id": user["user_id"],
                "email": user["email"],
                "role": user["role"],
                "status": user["status"],
                "referral_code": user.get("referral_code", "")
            }
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logging.error(f"❌ Login error: {str(e)}")
        raise HTTPException(status_code=500, detail=f"خطای سرور: {str(e)}")
