#!/usr/bin/env python3
"""
Migration Script: Fix ancestor_depths for existing users
این اسکریپت ancestor_depths همه کاربران را بازسازی می‌کند
"""

import asyncio
import sys
from motor.motor_asyncio import AsyncIOMotorClient
from typing import List, Dict

# تنظیمات MongoDB (تغییر دهید اگر لازم است)
MONGO_URI = "mongodb://localhost:27017"
DB_NAME = "mlm_db"


class PathManager:
    """کپی از PathManager برای استفاده در migration"""
    
    @staticmethod
    def extract_ancestors(path: str) -> List[str]:
        """استخراج لیست اجداد از path"""
        if not path:
            return []
        
        parts = path.split('/')
        return parts[:-1] if len(parts) > 1 else []
    
    @staticmethod
    def build_ancestor_depths(path: str, ancestors_data: List[Dict]) -> List[Dict]:
        """ساخت آرایه ancestor_depths"""
        ancestor_ids = PathManager.extract_ancestors(path)
        
        # اضافه کردن seed اگر نباشد
        if "seed" not in ancestor_ids and path and path != "seed":
            ancestor_ids.insert(0, "seed")
        
        # ساخت dict برای دسترسی سریع
        ancestors_dict = {a["user_id"]: a for a in ancestors_data}
        
        # ساخت آرایه مرتب
        ancestor_depths = []
        for ancestor_id in ancestor_ids:
            if ancestor_id in ancestors_dict:
                ancestor_depths.append({
                    "user_id": ancestor_id,
                    "depth": ancestors_dict[ancestor_id]["depth"]
                })
        
        return ancestor_depths


async def migrate_ancestor_depths():
    """
    بازسازی ancestor_depths برای همه کاربران
    """
    print("=" * 60)
    print("🔧 Migration: Fix ancestor_depths")
    print("=" * 60)
    
    # اتصال به MongoDB
    print(f"\n📡 Connecting to MongoDB: {MONGO_URI}")
    client = AsyncIOMotorClient(MONGO_URI)
    db = client[DB_NAME]
    
    try:
        # تست اتصال
        await db.command('ping')
        print("✅ Connected to MongoDB")
        
        # دریافت همه کاربران فعال
        print("\n📊 Fetching all active users...")
        users = await db.users.find(
            {"status": "active"},
            {"_id": 0, "user_id": 1, "placement_path": 1, "depth": 1, "ancestor_depths": 1}
        ).to_list(None)
        
        total_users = len(users)
        print(f"✅ Found {total_users} active users")
        
        if total_users == 0:
            print("⚠️ No active users to migrate")
            return
        
        # ساخت dict برای دسترسی سریع به depth
        users_dict = {u["user_id"]: u for u in users}
        
        # پردازش هر کاربر
        print("\n🔄 Processing users...")
        updated_count = 0
        skipped_count = 0
        
        for i, user in enumerate(users, 1):
            user_id = user["user_id"]
            placement_path = user.get("placement_path")
            
            if not placement_path and user_id != "seed":
                print(f"⚠️ [{i}/{total_users}] Skipping {user_id} (no placement_path)")
                skipped_count += 1
                continue
            
            # محاسبه ancestor_depths صحیح
            ancestor_ids = PathManager.extract_ancestors(placement_path) if placement_path else []
            
            # اضافه کردن seed
            if user_id != "seed" and "seed" not in ancestor_ids:
                ancestor_ids.insert(0, "seed")
            
            # ساخت ancestor_depths
            ancestors_data = []
            for anc_id in ancestor_ids:
                if anc_id in users_dict:
                    ancestors_data.append({
                        "user_id": anc_id,
                        "depth": users_dict[anc_id]["depth"]
                    })
            
            new_ancestor_depths = PathManager.build_ancestor_depths(
                placement_path if placement_path else "",
                ancestors_data
            )
            
            # مقایسه با ancestor_depths فعلی
            old_ancestor_depths = user.get("ancestor_depths", [])
            
            if new_ancestor_depths != old_ancestor_depths:
                # به‌روزرسانی
                await db.users.update_one(
                    {"user_id": user_id},
                    {"$set": {"ancestor_depths": new_ancestor_depths}}
                )
                
                print(f"✅ [{i}/{total_users}] Updated {user_id}")
                print(f"   Old: {old_ancestor_depths}")
                print(f"   New: {new_ancestor_depths}")
                updated_count += 1
            else:
                print(f"⏭️  [{i}/{total_users}] Skipped {user_id} (already correct)")
                skipped_count += 1
        
        # خلاصه
        print("\n" + "=" * 60)
        print("📊 Migration Summary")
        print("=" * 60)
        print(f"Total users: {total_users}")
        print(f"Updated: {updated_count}")
        print(f"Skipped: {skipped_count}")
        print("\n✅ Migration completed successfully!")
        
    except Exception as e:
        print(f"\n❌ Error during migration: {e}")
        import traceback
        traceback.print_exc()
    
    finally:
        client.close()
        print("\n🔒 MongoDB connection closed")


async def verify_migration():
    """
    بررسی صحت migration
    """
    print("\n" + "=" * 60)
    print("🔍 Verification: Checking ancestor_depths")
    print("=" * 60)
    
    client = AsyncIOMotorClient(MONGO_URI)
    db = client[DB_NAME]
    
    try:
        # بررسی کاربران بدون ancestor_depths
        users_without_ancestors = await db.users.count_documents({
            "status": "active",
            "user_id": {"$ne": "seed"},
            "ancestor_depths": {"$size": 0}
        })
        
        print(f"\n📊 Users without ancestor_depths: {users_without_ancestors}")
        
        if users_without_ancestors == 0:
            print("✅ All users have ancestor_depths!")
        else:
            print(f"⚠️ Found {users_without_ancestors} users without ancestor_depths")
            
            # نمایش چند نمونه
            samples = await db.users.find(
                {
                    "status": "active",
                    "user_id": {"$ne": "seed"},
                    "ancestor_depths": {"$size": 0}
                },
                {"_id": 0, "user_id": 1, "email": 1, "placement_path": 1}
            ).limit(5).to_list(5)
            
            print("\nSamples:")
            for sample in samples:
                print(f"  - {sample['user_id']}: {sample.get('email')} (path: {sample.get('placement_path')})")
        
    finally:
        client.close()


async def main():
    """اجرای migration و verification"""
    print("""
╔══════════════════════════════════════════════════════════╗
║       MLM System - Ancestor Depths Migration            ║
║                                                          ║
║  این اسکریپت ancestor_depths همه کاربران را بازسازی    ║
║  می‌کند تا seed در آن قرار گیرد.                       ║
╚══════════════════════════════════════════════════════════╝
""")
    
    response = input("\n⚠️ آیا از اجرای migration مطمئن هستید؟ (yes/no): ")
    
    if response.lower() not in ['yes', 'y']:
        print("❌ Migration لغو شد")
        return
    
    # اجرای migration
    await migrate_ancestor_depths()
    
    # بررسی نتیجه
    await verify_migration()


if __name__ == "__main__":
    asyncio.run(main())
