#!/bin/bash

################################################################################
# MLM System - نصب کامل با پاکسازی کامل
# این اسکریپت همه چیز قدیمی را پاک کرده و fresh نصب می‌کند
################################################################################

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

INSTALL_DIR=$(pwd)
SERVER_IP=$(curl -4 -s ifconfig.me 2>/dev/null || echo "127.0.0.1")

clear
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${GREEN}🚀 MLM System - نصب کامل و Fresh${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""
echo -e "${BLUE}📍 IP سرور: $SERVER_IP${NC}"
echo ""

# 1. پاکسازی کامل
echo -e "${YELLOW}🧹 پاکسازی کامل سیستم قدیمی...${NC}"
echo ""

# توقف همه processes
echo "⏳ توقف processes..."
sudo pkill -f "uvicorn server:app" 2>/dev/null || true
sudo pkill -f "serve.*build" 2>/dev/null || true
sudo pkill -f "yarn start" 2>/dev/null || true
sudo pkill -f "npm start" 2>/dev/null || true
sleep 3

# حذف containers
echo "⏳ حذف containers قدیمی..."
sudo docker rm -f mlm-mongodb mlm-redis mongodb redis 2>/dev/null || true
sudo docker volume rm mlm-mongodb-data mlm-redis-data 2>/dev/null || true

# پاکسازی پوشه‌های build
echo "⏳ پاکسازی پوشه‌های build..."
rm -rf backend/venv backend/__pycache__ backend/*.pyc backend/.env 2>/dev/null || true
rm -rf frontend/node_modules frontend/build frontend/.cache 2>/dev/null || true

echo -e "${GREEN}✔ پاکسازی کامل شد${NC}"
echo ""

# 2. نصب Docker
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🐳 نصب Docker (اگر نباشد)...${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""

if ! command -v docker &> /dev/null; then
    echo "⏳ نصب Docker..."
    curl -fsSL https://get.docker.com -o get-docker.sh
    sudo sh get-docker.sh
    sudo systemctl start docker
    sudo systemctl enable docker
    rm get-docker.sh
    echo -e "${GREEN}✔ Docker نصب شد${NC}"
else
    echo -e "${GREEN}✔ Docker موجود است${NC}"
fi

echo ""

# 3. راه‌اندازی MongoDB
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🍃 راه‌اندازی MongoDB...${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""

sudo systemctl stop mongodb 2>/dev/null || true
sudo systemctl stop mongod 2>/dev/null || true
sleep 2

sudo docker run -d \
    --name mlm-mongodb \
    --restart unless-stopped \
    -p 0.0.0.0:27017:27017 \
    -v mlm-mongodb-data:/data/db \
    -e MONGO_INITDB_DATABASE=mlm_production \
    mongo:7.0

echo "⏳ در انتظار MongoDB (15 ثانیه)..."
sleep 15

if sudo docker exec mlm-mongodb mongosh --quiet --eval "db.version()" > /dev/null 2>&1; then
    echo -e "${GREEN}✔ MongoDB آماده${NC}"
else
    echo -e "${RED}❌ MongoDB راه‌اندازی نشد${NC}"
    exit 1
fi

echo ""

# 4. Backend Setup
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🔧 نصب Backend...${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""

cd "$INSTALL_DIR/backend"

# Virtual environment
echo "⏳ ایجاد Python virtual environment..."
python3 -m venv venv
source venv/bin/activate

# Dependencies
echo "⏳ نصب Python packages (2-3 دقیقه)..."
pip install --upgrade pip > /dev/null 2>&1
pip install -r requirements.txt

deactivate

# .env
cat > .env << EOF
MONGO_URL=mongodb://localhost:27017
DB_NAME=mlm_production
JWT_SECRET=$(openssl rand -hex 32)
CORS_ORIGINS=*
HOST=0.0.0.0
PORT=8001
EOF

echo -e "${GREEN}✔ Backend آماده${NC}"
cd "$INSTALL_DIR"

echo ""

# 5. Frontend Setup
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🎨 نصب Frontend...${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""

cd "$INSTALL_DIR/frontend"

# Dependencies
echo "⏳ نصب Node packages (3-4 دقیقه)..."
yarn install

# .env
cat > .env << EOF
REACT_APP_BACKEND_URL=
EOF

# Build
echo "⏳ Build کردن Frontend (2-3 دقیقه)..."
export REACT_APP_BACKEND_URL=""
yarn build

echo -e "${GREEN}✔ Frontend build شد${NC}"
cd "$INSTALL_DIR"

echo ""

# 6. Firewall
echo "⏳ تنظیم Firewall..."
sudo ufw allow 8001/tcp 2>/dev/null || true
sudo ufw allow 3000/tcp 2>/dev/null || true
echo -e "${GREEN}✔ Firewall OK${NC}"

echo ""

# 7. راه‌اندازی
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${YELLOW}🚀 راه‌اندازی سرویس‌ها...${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""

# Backend
cd "$INSTALL_DIR/backend"
source venv/bin/activate
nohup uvicorn server:app --host 0.0.0.0 --port 8001 > backend.log 2>&1 &
BACKEND_PID=$!
echo -e "${GREEN}✔ Backend started (PID: $BACKEND_PID)${NC}"

sleep 5

# Test backend
if curl -s http://localhost:8001/api/health > /dev/null 2>&1; then
    echo -e "${GREEN}✔ Backend health: OK${NC}"
else
    echo -e "${RED}❌ Backend failed${NC}"
    cat backend.log | tail -20
    exit 1
fi

deactivate

# Frontend
cd "$INSTALL_DIR/frontend"
nohup npx serve -s build -l 3000 --single --no-clipboard > frontend.log 2>&1 &
FRONTEND_PID=$!
echo -e "${GREEN}✔ Frontend started (PID: $FRONTEND_PID)${NC}"

sleep 3

echo ""
echo "════════════════════════════════════════════════════════════════════════"
echo -e "${BOLD}${GREEN}✅ سیستم آماده است!${NC}"
echo "════════════════════════════════════════════════════════════════════════"
echo ""
echo -e "${CYAN}🌐 دسترسی:${NC}"
echo -e "${BOLD}   http://$SERVER_IP:3000${NC}"
echo ""
echo -e "${CYAN}🔐 ورود ادمین:${NC}"
echo -e "${GREEN}   Email:    admin@mlm.com${NC}"
echo -e "${GREEN}   Password: admin123${NC}"
echo ""
echo -e "${CYAN}📊 وضعیت:${NC}"
echo "   Backend:  PID $BACKEND_PID"
echo "   Frontend: PID $FRONTEND_PID"
echo ""
echo -e "${CYAN}📝 Logs:${NC}"
echo "   tail -f backend/backend.log"
echo "   tail -f frontend/frontend.log"
echo ""
echo -e "${CYAN}🛑 توقف:${NC}"
echo "   kill $BACKEND_PID $FRONTEND_PID"
echo ""
